package de.fzi.wim.guibase.localization;

import java.util.Locale;
import java.util.Map;
import java.util.HashMap;
import java.net.URL;
import java.text.MessageFormat;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.KeyStroke;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JRadioButton;

import de.fzi.wim.guibase.util.Mnemonics;

/**
 * Represents a set of settings used with a localte. The settings include
 * language phrases, as well as methods useful for localization.
 */
public class LocalizationManager {
    /** The locale for which this settings are valid. */
    protected Locale m_locale;
    /** The parent locale settings. If something is not found in this object, parent is invoked */
    protected LocalizationManager m_parent;
    /** Contains elements (text and icons) that are dependent on the locale. */
    protected Map m_contents;

    /**
     * Creates a new locale settings object.
     *
     * @param locale                locale for which settings are created
     * @param parent                parent locale settings
     */
    public LocalizationManager(Locale locale,LocalizationManager parent) {
        m_locale=locale;
        m_parent=parent;
        m_contents=new HashMap();
    }
    /**
     * Returns the phrase for the provided key. Returns <code>key</code>
     * if the key cannot be found neither in this object nor its parents.
     *
     * @param key                   the key to the desired phrase
     * @return                      phrase with given key or the key itself if phrase couldn't be found
     */
    public String getPhrase(String key) {
        String result=(String)m_contents.get(key);
        if (result==null && m_parent!=null)
            result=m_parent.getPhrase(key);
        if (result==null)
            result=key;
        return result;
    }
    /**
     * Returns the label with given key.
     *
     * @param key                   the key for the label
     * @return                      the label with given key
     */
    public JLabel getLabel(String key) {
        JLabel label=new JLabel();
        Mnemonics.setText(label,getPhrase(key));
        return label;
    }
    /**
     * Returns the button with given key.
     *
     * @param key                   the key for the button
     * @return                      the button with given key
     */
    public JButton getButton(String key) {
        JButton button=new JButton();
        Mnemonics.setText(button,getPhrase(key));
        return button;
    }
    /**
     * Returns the checkbox with given key.
     *
     * @param key                   the key for the checkbox
     * @return                      the checkbox with given key
     */
    public JCheckBox getCheckBox(String key) {
        JCheckBox checkbox=new JCheckBox();
        Mnemonics.setText(checkbox,getPhrase(key));
        return checkbox;
    }
    /**
     * Returns the radiobutton with given key.
     *
     * @param key                   the key for the checkbox
     * @return                      the checkbox with given key
     */
    public JRadioButton getRadioButton(String key) {
        JRadioButton radiobutton=new JRadioButton();
        Mnemonics.setText(radiobutton,getPhrase(key));
        return radiobutton;
    }
    /**
     * Checks whether key exists in this localization manager
     *
     * @param key                   the key eing checked
     * @return                      <code>true</code> if given key exists in the localization manager
     */
    public boolean containsKey(String key) {
        if (m_contents.containsKey(key))
            return true;
        else if (m_parent!=null)
            return m_parent.containsKey(key);
        else
            return false;
    }
    /**
     * Returns the icon with the provided key. If icon with given key is not found,
     * then <code>null</code> is returned.
     *
     * @param key                   the key of the icon
     * @return                      icon with given key or <code>null</code> if the icon doesn't exist
     */
    public ImageIcon getImageIcon(String key) {
        Object object=m_contents.get(key);
        if (object instanceof ImageIcon)
            return (ImageIcon)object;
        else if (object instanceof String) {
            String imageName=(String)object;
            String description=getPhrase(key+".description");
            URL imageURL=getClass().getClassLoader().getResource(imageName);
            ImageIcon image;
            if (imageURL==null)
                image=null;
            else {
                image=new ImageIcon(imageURL,description);
                m_contents.put(key,image);
            }
            return image;
        }
        else if (m_parent!=null && !m_contents.containsKey(key))
            return m_parent.getImageIcon(key);
        else
            return null;
    }
    /**
     * Returns a message format from an entry with given key. If an entry doesn't exist,
     * a message format equals to the key is returned.
     *
     * @param key                   the key of the message format
     * @return                      the message format with the message specified under given key
     */
    public MessageFormat getMessageFormat(String key) {
        return new MessageFormat(getPhrase(key),m_locale);
    }
    /**
     * Formats given objects with the message whose contents is specified under given key.
     *
     * @param key                   the key of the message
     * @param arguments             the arguments to the format
     * @return                      formatted message
     */
    public String format(String key,Object[] arguments) {
        return getMessageFormat(key).format(arguments,new StringBuffer(),null).toString();
    }
    /**
     * Parses given key and extracts properties of the action described by it. Properties are returned as a hashtable that
     * is indexed by keys equal to those defined in <code>javax.swing.Action</code> interface. Returns <code>null</code>
     * if action name is not found in the localization manager.
     *
     * @param actionID                      if of the action to be parsed (<code>null</code> if action name is not found)
     * @return                              map of action properties
     */
    public Map getActionMap(String actionID) {
        Map actionMap=new HashMap(5);
        actionMap.put("actionID",actionID);
        if (containsKey(actionID)) {
            String value=getPhrase(actionID);
            actionMap.put(Action.NAME,Mnemonics.getLabelText(value));
            int mnemonic=Mnemonics.getLabelMnemonic(value);
            if (mnemonic!=-1)
                actionMap.put(Action.MNEMONIC_KEY,new Integer(mnemonic));
        }
        else
            actionMap.put(Action.NAME,"Action "+actionID+" has no name.");
        if (containsKey(actionID+".accelerator"))
            actionMap.put(Action.ACCELERATOR_KEY,KeyStroke.getKeyStroke(getPhrase(actionID+".accelerator")));
        if (containsKey(actionID+".short"))
            actionMap.put(Action.SHORT_DESCRIPTION,getPhrase(actionID+".short"));
        if (containsKey(actionID+".long"))
            actionMap.put(Action.LONG_DESCRIPTION,getPhrase(actionID+".long"));
        ImageIcon icon=getImageIcon(actionID+".icon");
        if (icon!=null)
            actionMap.put(Action.SMALL_ICON,icon);
        return actionMap;
    }
    /**
     * Returns the Locale assigned to this object.
     *
     * @return                      the locale of the object.
     */
    public Locale getLocale() {
        return m_locale;
    }
    /**
     * Returns the parent of this language.
     *
     * @return                      the parent of this object or <code>null</code>
     */
    public LocalizationManager getParent() {
        return m_parent;
    }
    /**
     * Sets the parent of this language.
     *
     * @param                       value new parent of this object
     */
    void setParent(LocalizationManager value) {
        m_parent=value;
    }
    /**
     * Adds a phrase to this Language, providing the key to find that
     * phrase again later. An already existing phrase with the same key will be overwritten.
     *
     * @param key                   the key of the phrase
     * @param phrase                the value of the phrase
     */
    void addPhrase(String key,String phrase) {
        m_contents.put(key,phrase);
    }
}
