package de.fzi.wim.guibase.lazytreemodel;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;

/**
 * Tree model that supports on-the-fly loading of nodes under its control. This class allows easy creation of tree models
 * that do not contain all data at once, but that load data as needed. This can dractically improve memory performance of the application.
 * Nodes under control of this model must implement {@link LazyTreeNode} interface. The idea is that each node doesn't need to have
 * the list of its children generated upfront. Rather, the model will take care of loading chinldren when they are requested.
 * There are two modes of operation:
 * <ul>
 *  <li>synchronus - children are loaded on the thread of the caller
 *  <li>asynchronus - children are loaded on a background thread and an event is generated when nodes are loaded
 * </ul>
 * Children of some node are produced by an instance implementing {@link LazyNodeLoader} interface. This interface can choose
 * to process the request synchronously or asynchronously. If loading is asynchornous, then a default no-children object can
 * be returned as the only child of nodes that are deing loaded. This node will typically display a message "Please wait...".
 * <p>It is possible to mix {@link LazyTreeNode} and <code>javax.swing.tree.TreeNode</code> objects in the same tree structure.
 * For nodes implementing <code>javax.swing.tree.TreeNode</code> this class will assume that they have been fully loaded.
 *
 * @see LazyTreeNode
 * @see LazyNodeLoader
 */
public class LazyTreeModel extends DefaultTreeModel {
    /** Magic constant denoting that node that has changed should be collapsed. */
    public static final int[] COLLAPSE_CHILDREN=new int[] { -2 };
	/** Node that will be returned as child when its parent is loading children asynchroniusly. */
	protected TreeNode m_noChildrenNode;
    /** Node loader that will be used to process loading requests. */
    protected LazyNodeLoader m_loader;

	/**
	 * Creates and instance of this class and attaches it to root.
	 *
	 * @param root                          root of the tree
     * @param loader                        loader that will produce children nodes on demand
	 */
	public LazyTreeModel(TreeNode root,LazyNodeLoader loader) {
		this(root,loader,null);
	}
	/**
	 * Creates and instance of this class, attaches it to root, sets appropriate lazy-load mode and sets node
	 * that will be shown when there are no children under their parent.
	 *
	 * @param root                          root of the tree
     * @param loader                        loader that will produce children nodes on demand
	 * @param noChildrenNode                object shown as child while parent is loading nodes asynchronously
	 */
	public LazyTreeModel(TreeNode root,LazyNodeLoader loader,TreeNode noChildrenNode) {
		super(root);
		m_loader=loader;
		m_noChildrenNode=noChildrenNode;
	}
	/**
	 * Returns the child of supplied parent at given index. If supplied node's children haven't been loaded yet,
	 * this method will attempt to load them (synchronously or asynchronously).
	 *
	 * @param parent                        parent node whose children are requested
	 * @param index                         index of the requested child
	 * @return                              child of the parent at given index
	 */
	public Object getChild(Object parent,int index) {
		if (parent instanceof LazyTreeNode) {
			LazyTreeNode parentNode=(LazyTreeNode)parent;
			if (parentNode.getNodeState()!=LazyTreeNode.LOADED) {
				if (loadNodeChildren(parentNode))
					return getChildWhenNodeIsLoading(parentNode,index);
			}
		}
		return super.getChild(parent,index);
	}
	/**
	 * Returns the number of children of supplied parent. If supplied node's children haven't been loaded yet,
	 * this method will attempt to load them (synchronously or asynchronously).
	 *
	 * @param parent                        parent node whose number of children is requested
	 * @return                              number of children of given parent
	 */
	public int getChildCount(Object parent) {
		if (parent instanceof LazyTreeNode) {
			LazyTreeNode parentNode=(LazyTreeNode)parent;
			if (parentNode.getNodeState()!=LazyTreeNode.LOADED) {
				if (loadNodeChildren(parentNode))
					return getChildrenCountWhenNodeIsLoading(parentNode);
			}
		}
		return super.getChildCount(parent);
	}
	/**
	 * Returns the index of supplied child under given parent. If supplied parent's children haven't been loaded yet,
	 * this method will attempt to load them (synchronously or asynchronously).
	 *
	 * @param parent                        parent node whose child index is requested
	 * @param child                         child whose index is requested
	 * @return                              index of given child under supplied parent
	 */
	public int getIndexOfChild(Object parent,Object child) {
		if (parent instanceof LazyTreeNode) {
			LazyTreeNode parentNode=(LazyTreeNode)parent;
	    	if (parentNode.getNodeState()!=LazyTreeNode.LOADED) {
				if (loadNodeChildren(parentNode))
					return getChildIndexWhenNodeIsLoading(parentNode,child);
			}
	    }
		return super.getIndexOfChild(parent,child);
	}
	/**
	 * Loads children of supplied node and returns <code>true</code> if children are being loaded asynchronously.
	 *
	 * @param node                          parent whose children should be loaded
	 * @return                              <code>true</code> if children will be loaded asynchronously
	 */
	protected boolean loadNodeChildren(LazyTreeNode node) {
		if (node.getNodeState()==LazyTreeNode.LOADING)
			return true;
		node.setNodeLoading();
        TreeNode[] children=getLazyNodeLoader(node).loadNodeChildren(this,node);
        if (children==LazyNodeLoader.LOAD_DEFERRED)
            return true;
   		node.setChildrenLoaded(children);
		return false;
	}
    /**
     * Returns a node loader for given node.
     *
     * @param node                          node for which loader is requested
     * @return                              loader capable of loading children of supplied node
     */
    public LazyNodeLoader getLazyNodeLoader(LazyTreeNode node) {
        return m_loader;
    }
	/**
	 * This method should be called by node loaders on main thread when they are done with loading some node. This method
     * should not be called in any other circumstance.
	 *
	 * @param node                          node whose children have been loaded
	 * @param children                      array of loaded children
	 */
	public void notifyChildrenLoaded(LazyTreeNode node,TreeNode[] children) {
        if (node.getNodeState()==LazyTreeNode.LOADING) {
            if (children!=LazyNodeLoader.LOAD_CANCELED)
        		node.setChildrenLoaded(children);
            else
                node.setChildrenLoaded(new TreeNode[0]);
            nodeStructureChanged(node);
        }
	}
	/**
	 * This method is called to return the number of children under a node that is loading children asynchronously.
	 * This implementation returns 0 if no-children node hasn't been set, or 1 if it has.
	 *
	 * @param node                          node for which children count is requested
	 * @return                              number of children under given node
	 */
	protected int getChildrenCountWhenNodeIsLoading(LazyTreeNode node) {
		return m_noChildrenNode==null ? 0 : 1;
	}
	/**
	 * This method is called to return a child under a node that is loading children asynchronously.
	 * This implementation returns no-children node if it has been set.
	 *
	 * @param node                          node for which children count is requested
	 * @param index                         index of child required
	 * @return                              child with given index
	 */
	protected Object getChildWhenNodeIsLoading(LazyTreeNode node,int index) {
		return m_noChildrenNode;
	}
	/**
	 * This method is called to return the index of the child under a node that is loading children asynchronously.
	 * This implementation returns 0 if supplied node is no-children node, or -1 otherwise.
	 *
	 * @param parentNode                    node for which child index is requested
	 * @param child                         child whose index is requested
	 * @return                              index of supplied child under parent node
	 */
	protected int getChildIndexWhenNodeIsLoading(LazyTreeNode parentNode,Object child) {
		return child!=null && child==m_noChildrenNode ? 0 : -1;
	}
}
