package de.fzi.wim.guibase.lazytreemodel;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import javax.swing.tree.TreeNode;

/**
 * Abstract base class for all lazy nodes. This node tracks its state by means of its <code>m_children</code> member.
 * If this member is <code>null</code>, this means that node wasn't loaded.
 * If this member is set to <code>NODE_IS_LOADING</code> this means that children of this node are being loaded. Otherwise,
 * <code>m_children</code> does actually represent the list of children of this node.
 */
public abstract class AbstractLazyTreeNode implements LazyTreeNode {
    /** Constant denoting that there a node has no children. */
    public static final TreeNode[] EMPTY_CHILDREN=new TreeNode[0];
    /** Constant indicating that children of this node are being loaded. */
    protected static final TreeNode[] NODE_IS_LOADING=new TreeNode[0];

    /** Parent of this node. */
    protected TreeNode m_parent;
    /** Array containing all children of this node. */
    protected TreeNode[] m_children;

    /**
     * Creates and initialized this instance.
     *
     * @param parent                parent of this node
     */
    public AbstractLazyTreeNode(TreeNode parent) {
        m_parent=parent;
        m_children=null;
    }
    /**
     * Returns an enumeration of children of this node.
     *
     * @return                      <code>Enumeration</code> of <code>TreeNode</code> objects that are children of this node
     */
    public Enumeration children() {
        return new ArrayEnumeration(m_children);
    }
    /**
     * Returns <code>true</code> since all nodes allow children.
     *
     * @return                      always <code>true</code>
     */
    public boolean getAllowsChildren() {
        return true;
    }
    /**
     * Returns the child at given position.
     *
     * @param childIndex            index of the child that needs to be returned
     * @return                      child with given index
     */
    public TreeNode getChildAt(int childIndex) {
        if (m_children==null || m_children==NODE_IS_LOADING)
            return null;
        return m_children[childIndex];
    }
    /**
     * Returns the number of children of this node.
     *
     * @return                      number of children of this node
     */
    public int getChildCount() {
        if (m_children==null || m_children==NODE_IS_LOADING)
            return 0;
        return m_children.length;
    }
    /**
     * Returns the index of given child (or -1 if supplied node is not a child of this node)
     *
     * @param node                  node whose index is requested
     * @return                      index of given node (or -1 if node is not a child of this node)
     */
    public int getIndex(TreeNode node) {
        if (m_children==null || m_children==NODE_IS_LOADING)
            return -1;
        for (int i=m_children.length-1;i>=0;--i)
            if (node.equals(m_children[i]))
                return i;
        return -1;
    }
    /**
     * Returns the parent of this node.
     *
     * @return                      parent of this node
     */
    public TreeNode getParent() {
        return m_parent;
    }
    /**
     * Returns the state of this node.
     *
     * @return                      state of this node
     */
    public int getNodeState() {
        if (m_children==null)
            return NOT_LOADED;
        else if (m_children==NODE_IS_LOADING)
            return LOADING;
        else
            return LOADED;
    }
    /**
     * Changes the state of this node to <code>LOADING</code>.
     */
    public void setNodeLoading() {
        m_children=NODE_IS_LOADING;
    }
    /**
     * Unloads children of a node.
     */
    public void unloadChildren() {
        m_children=null;
    }
    /**
     * Notifies this node that its children have been loaded.
     *
     * @param children              array of children of this node
     */
    public void setChildrenLoaded(TreeNode[] children) {
        m_children=children;
    }
    /**
     * Retruns the array of children of this node. If node is not loaded, <code>null</code> is returned.
     *
     * @return                      array of children of this node
     */
    public TreeNode[] getChildren() {
        if (getNodeState()==LOADED)
            return m_children;
        else
            return null;
    }
    /**
     * Returns <code>true</code> if this node is a leaf node in the tree.
     *
     * @return                      <code>true</code> if this node is a leaf node
     */
    public boolean isLeaf() {
        if (getNodeState()==LOADED)
            return getChildCount()==0;
        return false;
    }

    /**
     * Adapter that turns an array into an enumeration.
     */
    protected static class ArrayEnumeration implements Enumeration {
        /** Array being enumerated. */
        protected Object[] m_elements;
        /** Index of the element that will be returned next. */
        protected int m_index;

        /**
         * Creates an enumeration adapter.
         *
         * @param elements              elements being adapted
         */
        public ArrayEnumeration(Object[] elements) {
            m_elements=elements;
        }
        /**
         * Tests if there are more elements.
         *
         * @return                      <code>true</code> if there are more elements
         */
        public boolean hasMoreElements() {
            return m_index<m_elements.length;
        }
        /**
         * Retrns next element.
         *
         * @return                      next element
         * @exception NoSuchElementException    thrown if there are no more elements
         */
        public Object nextElement() throws NoSuchElementException {
            if (!hasMoreElements())
                throw new NoSuchElementException();
            return m_elements[m_index++];
        }
    }
}
