package de.fzi.wim.guibase.graphview.view;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Font;
import java.awt.FontMetrics;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.controller.*;

/**
 * The painter drawing the node as a rectangle.
 */
public class RectangleNodePainter implements NodePainter {
    /** An instance. */
    public static final NodePainter INSTANCE=new RectangleNodePainter();

    /**
     * Paints the supplied node.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param node                  the node to paint
     */
    public void paintNode(JGraphPane graphPane,Graphics2D g,Node node) {
        HighlightingManipulator highlightingManipulator=(HighlightingManipulator)graphPane.getManipulator(HighlightingManipulator.NAME);
        boolean isHighlighted=highlightingManipulator!=null && highlightingManipulator.getHighlightedNode()==node;
        SelectionManipulator selectionManipulator=(SelectionManipulator)graphPane.getManipulator(SelectionManipulator.NAME);
        boolean isSelected=selectionManipulator!=null && selectionManipulator.getNodeSelectionModel().isNodeSelected(node);
        DraggingManipulator draggingManipulator=(DraggingManipulator)graphPane.getManipulator(DraggingManipulator.NAME);
        boolean isDragging=draggingManipulator!=null && draggingManipulator.getDraggedNode()==node;
        Point nodePoint=graphPane.getScreenPointForNode(node);
        int width=1;
        int height=1;
        int textX=0;
        int textY=0;
        String label=node.getLabel();
        if (label!=null) {
            FontMetrics fontMetrics=graphPane.getFontMetrics(graphPane.getFont());
            int stringWidth=fontMetrics.stringWidth(label);
            width+=stringWidth+6;
            height+=fontMetrics.getAscent()+fontMetrics.getDescent()+4;
            textX=nodePoint.x-stringWidth/2;
            textY=nodePoint.y+(fontMetrics.getAscent()-fontMetrics.getDescent())/2;
        }
        else {
            width+=40;
            height+=20;
        }
        Color oldColor=g.getColor();
        g.setColor(getBackgroundColor(isHighlighted,isSelected,isDragging));
        g.fillRect(nodePoint.x-width/2,nodePoint.y-height/2,width,height);
        if (label!=null) {
            Font oldFont=g.getFont();
            g.setFont(graphPane.getFont());
            g.setColor(getTextColor(isHighlighted,isSelected,isDragging));
            g.drawString(label,textX,textY);
            g.setFont(oldFont);
        }
        g.setColor(getBorderColor(isHighlighted,isSelected,isDragging));
        g.drawRect(nodePoint.x-width/2,nodePoint.y-height/2,width,height);
        g.setColor(oldColor);
    }
    /**
     * Returns the border color of the node.
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @return                      the border color
     */
    protected Color getBorderColor(boolean isHighlighted,boolean isSelected,boolean isDragging) {
        return isHighlighted || isDragging ? Color.red : Color.black;
    }
    /**
     * Returns the background color of the node.
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @return                      the background color
     */
    protected Color getBackgroundColor(boolean isHighlighted,boolean isSelected,boolean isDragging) {
        if (isSelected)
            return Color.cyan;
        else
            return Color.yellow;
    }
    /**
     * Returns the text color of the node
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @return                      the background color
     */
    protected Color getTextColor(boolean isHighlighted,boolean isSelected,boolean isDragging) {
        return Color.black;
    }
    /**
     * Checks whether given point is inside the node.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      <code>true</code> if the point is in the node
     */
    public boolean isInNode(JGraphPane graphPane,Node node,Point point) {
        Rectangle nodeScreenRectangle=new Rectangle();
        getNodeScreenBounds(graphPane,node,nodeScreenRectangle);
        return nodeScreenRectangle.contains(point);
    }
    /**
     * Returns the outer rectangle of the node on screen.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param nodeScreenRectangle   the rectangle receiving the node's coordinates
     */
    public void getNodeScreenBounds(JGraphPane graphPane,Node node,Rectangle nodeScreenRectangle) {
        Point nodePoint=graphPane.getScreenPointForNode(node);
        String label=node.getLabel();
        int width=1;
        int height=1;
        if (label!=null) {
            FontMetrics fontMetrics=graphPane.getFontMetrics(graphPane.getFont());
            width+=fontMetrics.stringWidth(label)+6;
            height+=fontMetrics.getAscent()+fontMetrics.getDescent()+4;
        }
        else {
            width+=40;
            height+=20;
        }
        nodeScreenRectangle.setBounds(nodePoint.x-width/2,nodePoint.y-height/2,width,height);
    }
    /**
     * Retruns the tool-tip for given point.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      the tool-tip at given point (or <code>null</code>)
     */
    public String getToolTipText(JGraphPane graphPane,Node node,Point point) {
        return null;
    }
}
