package de.fzi.wim.guibase.graphview.view;

import java.util.Iterator;
import java.util.ListIterator;
import java.util.Collection;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.awt.event.MouseEvent;
import java.awt.event.KeyEvent;
import java.awt.event.FocusEvent;
import java.awt.event.ComponentEvent;
import javax.swing.JPanel;
import javax.swing.ToolTipManager;
import javax.swing.SwingUtilities;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.lens.*;

/**
 * The panel capable of visualizing the graph.
 */
public class JGraphPane extends JPanel {
    /** The graph being visualized. */
    protected Graph m_graph;
    /** The listener for the graph. */
    protected GraphListener m_graphListner;
    /** The listener for the lens. */
    protected LensListener m_lensListener;
    /** The currently active lens. */
    protected Lens m_lens;
    /** The map of node positions. */
    protected Map m_nodePositions;
    /** The array of registered manipulators in the order they were registered. */
    protected List m_manipulators;
    /** The map of rigestered manipulators keyed by their name. */
    protected Map m_manipulatorsByName;

   /**
    * Creates a graph pane.
    *
    * @param graph                      the graph
    */
    public JGraphPane(Graph graph) {
        super(null);
        m_graphListner=new GraphHandler();
        m_lensListener=new LensHandler();
        m_nodePositions=new HashMap();
        m_manipulators=new ArrayList();
        m_manipulatorsByName=new HashMap();
        setGraph(graph);
        enableEvents(MouseEvent.MOUSE_EVENT_MASK | MouseEvent.MOUSE_MOTION_EVENT_MASK | KeyEvent.KEY_EVENT_MASK | FocusEvent.FOCUS_EVENT_MASK | ComponentEvent.COMPONENT_EVENT_MASK);
        setBackground(Color.white);
        ToolTipManager.sharedInstance().registerComponent(this);
    }
    /**
     * Adds a manipulator to this pane. If a manipulator with the same name if registered, if is rist removed.
     *
     * @param manipulator               the manipulator
     */
    public void addManipulator(Manipulator manipulator) {
        removeManipulator(manipulator.getName());
        m_manipulators.add(manipulator);
        m_manipulatorsByName.put(manipulator.getName(),manipulator);
        manipulator.setGraphPane(this);
    }
    /**
     * Returns a manipulator with given name.
     *
     * @param name                      the name of the manpulator
     * @return                          the manipulator with given name (or <code>null</code> if the manipualtor with given name is not registered)
     */
    public Manipulator getManipulator(String name) {
        return (Manipulator)m_manipulatorsByName.get(name);
    }
    /**
     * Removes a manipulator with given name.
     *
     * @param name                      the name of the manpulator
     */
    public void removeManipulator(String name) {
        Manipulator manipulator=(Manipulator)m_manipulatorsByName.remove(name);
        if (manipulator!=null) {
            m_manipulators.remove(manipulator);
            manipulator.setGraphPane(null);
        }
    }
    /**
     * Returns the current lens.
     *
     * @return                          the current lens (may be <code>null</code>)
     */
    public Lens getLens() {
        return m_lens;
    }
    /**
     * Sets the new lens.
     *
     * @param lens                      the new lens
     */
    public void setLens(Lens lens) {
        Lens oldLens=m_lens;
        if (m_lens!=null)
            m_lens.removeLensListener(m_lensListener);
        m_lens=lens;
        if (m_lens!=null)
            m_lens.addLensListener(m_lensListener);
        m_nodePositions.clear();
        repaint();
        firePropertyChange("lens",oldLens,m_lens);
    }
    /**
     * Returns the current graph.
     *
     * @return                          the current graph
     */
    public Graph getGraph() {
        return m_graph;
    }
    /**
     * Sets the current graph.
     *
     * @param graph                     the new graph
     */
    public void setGraph(Graph graph) {
        Graph oldGraph=m_graph;
        if (m_graph!=null)
            m_graph.removeGraphListener(m_graphListner);
        m_graph=graph;
        if (m_graph!=null)
            m_graph.addGraphListener(m_graphListner);
        m_nodePositions.clear();
        repaint();
        firePropertyChange("graph",oldGraph,m_graph);
    }
    /**
     * Updates the component.
     *
     * @param g                         the graphics
     */
    public void paintComponent(Graphics g) {
        Graphics2D g2d=(Graphics2D)g;
        Rectangle clipRectangle=g.getClipBounds();
        Color oldColor=g.getColor();
        g.setColor(getBackground());
        g.fillRect(clipRectangle.x,clipRectangle.y,clipRectangle.width,clipRectangle.height);
        g.setColor(oldColor);
        if (m_graph!=null)
            synchronized (m_graph) {
                Rectangle bounds=new Rectangle();
                Iterator iterator=m_graph.getEdges().iterator();
                while (iterator.hasNext()) {
                    Edge edge=(Edge)iterator.next();
                    getEdgeScreenBounds(edge,bounds);
                    if (clipRectangle.intersects(bounds))
                        paintEdge(g2d,edge);
                }
                iterator=m_graph.getNodes().iterator();
                while (iterator.hasNext()) {
                    Node node=(Node)iterator.next();
                    getNodeScreenBounds(node,bounds);
                    if (clipRectangle.intersects(bounds))
                        paintNode(g2d,node);
                }
                for (int i=0;i<m_manipulators.size();i++)
                    ((Manipulator)m_manipulators.get(i)).paint(g2d);
            }
    }
    /**
     * Paints the edge.
     *
     * @param g                         the graphics
     * @param edge                      the edge
     */
    protected void paintEdge(Graphics2D g,Edge edge) {
        EdgePainter edgePainter=getPainterForEdge(edge);
        edgePainter.paintEdge(this,g,edge);
    }
    /**
     * Returns the painter for the edge.
     *
     * @param edge                      the edge
     * @return                          the painter for the edge
     */
    public EdgePainter getPainterForEdge(Edge edge) {
        return ArrowEdgePainter.INSTANCE;
    }
    /**
     * Paints the node.
     *
     * @param g                         the graphics
     * @param node                      the node
     */
    protected void paintNode(Graphics2D g,Node node) {
        NodePainter nodePainter=getPainterForNode(node);
        nodePainter.paintNode(this,g,node);
    }
    /**
     * Returns the painter for the node.
     *
     * @param node                      the node
     * @return                          the painter for the node
     */
    public NodePainter getPainterForNode(Node node) {
        return RectangleNodePainter.INSTANCE;
    }
    /**
     * Converts a given screen point into a point on the graph.
     *
     * @param point                     the sreen point
     * @param graphPoint                the calculatedpoint in the graph
     */
    public void screenToGraphPoint(Point point,Point2D graphPoint) {
        graphPoint.setLocation(point.x-getWidth()/2,point.y-getHeight()/2);
        if (m_lens!=null)
            m_lens.undoLens(graphPoint);
    }
    /**
     * Converts a given graph point into a point on the screen.
     *
     * @param graphPoint                the point in the graph
     * @param point                     the calculated screen point
     */
    public void graphToScreenPoint(Point2D graphPoint,Point point) {
        double oldX=graphPoint.getX();
        double oldY=graphPoint.getY();
        if (m_lens!=null)
            m_lens.applyLens(graphPoint);
        point.setLocation((int)graphPoint.getX()+getWidth()/2,(int)graphPoint.getY()+getHeight()/2);
        graphPoint.setLocation(oldX,oldY);
    }
    /**
     * Returns the node at given point, or <code>null</code> if there is no such node.
     *
     * @param point                     the screen point
     * @return                          the node at given point (or <code>null</code> if there is no such node)
     */
    public Node getNodeAtPoint(Point point) {
        synchronized (m_graph) {
            List nodes=m_graph.getNodes();
            ListIterator iterator=nodes.listIterator(nodes.size());
            while (iterator.hasPrevious()) {
                Node node=(Node)iterator.previous();
                NodePainter nodePainter=getPainterForNode(node);
                if (nodePainter.isInNode(this,node,point))
                    return node;
            }
            return null;
        }
    }
    /**
     * Returns the set of nodes in given rectangle.
     *
     * @param rectangle                 the rectangle in which the nodes must be located
     * @return                          the set of nodes in the region
     */
    public Set getNodesInRectangle(Rectangle rectangle) {
        synchronized (m_graph) {
            Set nodesInRectangle=new HashSet();
            Rectangle nodeRectangle=new Rectangle();
            Iterator nodes=m_graph.getNodes().iterator();
            while (nodes.hasNext()) {
                Node node=(Node)nodes.next();
                getNodeScreenBounds(node,nodeRectangle);
                if (rectangle.contains(nodeRectangle))
                    nodesInRectangle.add(node);
            }
            return nodesInRectangle;
        }
    }
    /**
     * Returns the edge at given point, or <code>null</code> if there is no such edge.
     *
     * @param point                     the screen point
     * @return                          the edge at given point (or <code>null</code> if there is no such edge)
     */
    public Edge getNearestEdge(Point point) {
        Edge nearestEdge=null;
        double minDistance=4;
        synchronized (m_graph) {
            List edges=m_graph.getEdges();
            ListIterator iterator=edges.listIterator(edges.size());
            while (iterator.hasPrevious()) {
                Edge edge=(Edge)iterator.previous();
                EdgePainter edgePainter=getPainterForEdge(edge);
                double distance=edgePainter.screenDistanceFromEdge(this,edge,point);
                if (distance<minDistance) {
                    minDistance=distance;
                    nearestEdge=edge;
                }
            }
            return nearestEdge;
        }
    }
    /**
     * Returns the position of the node on the screen.
     *
     * @param node                      the node whose on-screen position is required
     * @return                          the position of the node on screen
     */
    public Point getScreenPointForNode(Node node) {
        Point point=(Point)m_nodePositions.get(node);
        if (point==null) {
            point=new Point();
            graphToScreenPoint(new Point2D.Double(node.getX(),node.getY()),point);
            m_nodePositions.put(node,point);
        }
        return point;
    }
    /**
     * Updates the map of screen positions of nodes.
     */
    protected void updateNodeScreenPositions() {
        synchronized (m_graph) {
            Point2D graphPoint=new Point2D.Double();
            Iterator nodes=m_graph.getNodes().iterator();
            while (nodes.hasNext()) {
                Node node=(Node)nodes.next();
                Point point=(Point)m_nodePositions.get(node);
                if (point==null) {
                    point=new Point();
                    m_nodePositions.put(node,point);
                }
                graphPoint.setLocation(node.getX(),node.getY());
                graphToScreenPoint(graphPoint,point);
            }
        }
    }
    /**
     * Returns the screen bounds of given node.
     *
     * @param node                      the node for which the bounds must be returned
     * @param nodeScreenRectangle       the rectangle receiving the node's coordinates
     */
    public void getNodeScreenBounds(Node node,Rectangle nodeScreenRectangle) {
        NodePainter nodePainter=getPainterForNode(node);
        nodePainter.getNodeScreenBounds(this,node,nodeScreenRectangle);
    }
    /**
     * Repaints the given node.
     *
     * @param node                      the node that needs to be repainted
     */
    public void repaintNode(Node node) {
        Rectangle nodeScreenRectangle=new Rectangle();
        getNodeScreenBounds(node,nodeScreenRectangle);
        repaint(nodeScreenRectangle);
    }
    /**
     * Returns the screen bounds of given edge.
     *
     * @param edge                      the edge for which the bounds must be returned
     * @param edgeScreenRectangle       the rectangle receiving the edge's coordinates
     */
    public void getEdgeScreenBounds(Edge edge,Rectangle edgeScreenRectangle) {
        EdgePainter edgePainter=getPainterForEdge(edge);
        edgePainter.getEdgeScreenBounds(this,edge,edgeScreenRectangle);
    }
    /**
     * Repaints the given edge.
     *
     * @param edge                      the edge that needs to be repainted
     */
    public void repaintEdge(Edge edge) {
        Rectangle edgeScreenRectangle=new Rectangle();
        getEdgeScreenBounds(edge,edgeScreenRectangle);
        edgeScreenRectangle.grow(5,5);
        repaint(edgeScreenRectangle);
    }
    /**
     * Returns the text for the tool-tip.
     *
     * @param event                     the mouse event
     * @return                          the text for the tool-tip
     */
    public String getToolTipText(MouseEvent event) {
        Point point=event.getPoint();
        Node node=getNodeAtPoint(point);
        if (node!=null) {
            NodePainter nodePainter=getPainterForNode(node);
            String toolTipText=nodePainter.getToolTipText(this,node,point);
            if (toolTipText!=null)
                return toolTipText;
        }
        return super.getToolTipText(event);
    }
    /**
     * Processes the component event.
     *
     * @param e                         the event
     */
    protected void processComponentEvent(ComponentEvent e) {
        super.processComponentEvent(e);
        updateNodeScreenPositions();
        repaint();
    }
    /**
     * Processes the mouse event.
     *
     * @param e                         the event
     */
    protected void processMouseEvent(MouseEvent e) {
        super.processMouseEvent(e);
        switch (e.getID()) {
            case MouseEvent.MOUSE_PRESSED:
            if (!hasFocus() && isRequestFocusEnabled())
                requestFocus();
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mousePressed(e);
            break;
        case MouseEvent.MOUSE_RELEASED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseReleased(e);
            break;
        case MouseEvent.MOUSE_CLICKED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseClicked(e);
            break;
        case MouseEvent.MOUSE_ENTERED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseEntered(e);
            break;
        case MouseEvent.MOUSE_EXITED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseExited(e);
            break;
        }
    }
    /**
     * Processes the mouse motion events.
     *
     * @param e                         mouse event
     */
    protected void processMouseMotionEvent(MouseEvent e) {
        super.processMouseMotionEvent(e);
        switch (e.getID()) {
        case MouseEvent.MOUSE_MOVED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseMoved(e);
            break;
        case MouseEvent.MOUSE_DRAGGED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).mouseDragged(e);
            break;
        }
    }
    /**
     * Processes the key event.
     *
     * @param e                         the event
     */
    protected void processKeyEvent(KeyEvent e) {
        super.processKeyEvent(e);
        switch (e.getID()) {
        case KeyEvent.KEY_TYPED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).keyTyped(e);
            break;
        case KeyEvent.KEY_PRESSED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).keyPressed(e);
            break;
        case KeyEvent.KEY_RELEASED:
            for (int i=0;i<m_manipulators.size() && !e.isConsumed();i++)
                ((Manipulator)m_manipulators.get(i)).keyReleased(e);
            break;
        }
    }
    /**
     * Processes the focus event.
     *
     * @param e                         the event
     */
    protected void processFocusEvent(FocusEvent e) {
        super.processFocusEvent(e);
        switch (e.getID()) {
        case FocusEvent.FOCUS_GAINED:
            for (int i=0;i<m_manipulators.size();i++)
                ((Manipulator)m_manipulators.get(i)).focusGained(e);
            break;
        case FocusEvent.FOCUS_LOST:
            for (int i=0;i<m_manipulators.size();i++)
                ((Manipulator)m_manipulators.get(i)).focusLost(e);
            break;
        }
    }
    /**
     * Overridden to notify the manipulators that the scroll position has changed.
     *
     * @param rectangle                 the rectangle
     */
    public void scrollRectToVisible(Rectangle rectangle) {
        super.scrollRectToVisible(rectangle);
        for (int i=0;i<m_manipulators.size();i++)
            ((Manipulator)m_manipulators.get(i)).notifyGraphPaneScrolled();
    }

    /**
     * The handler of graph events.
     */
    protected class GraphHandler implements GraphListener {
        public void graphLayoutUpdated(Graph graph) {
            if (SwingUtilities.isEventDispatchThread()) {
                updateNodeScreenPositions();
                repaint();
            }
            else
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        updateNodeScreenPositions();
                        repaint();
                    }
                });
        }
        public void graphUpdated(Graph graph) {
            if (SwingUtilities.isEventDispatchThread())
                repaint();
            else
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        repaint();
                    }
                });
        }
        public void graphContentsChanged(Graph graph) {
            m_nodePositions.clear();
            repaint();
        }
        public void elementsAdded(Graph graph,Collection nodes,Collection edges) {
            repaint();
        }
        public void elementsRemoved(Graph graph,Collection nodes,Collection edges) {
            if (nodes!=null) {
                Iterator iterator=nodes.iterator();
                while (iterator.hasNext()) {
                    Node node=(Node)iterator.next();
                    m_nodePositions.remove(node);
                }
            }
            repaint();
        }
    }

    /**
     * The handler of lens events.
     */
    protected class LensHandler implements LensListener {
        public void lensUpdated(Lens lens) {
            updateNodeScreenPositions();
            repaint();
        }
    }
}
