package de.fzi.wim.guibase.graphview.view;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.controller.*;

/**
 * The painter that paints the edge as the arrow.
 */
public class ArrowEdgePainter extends AbstractEdgePainter {
    /** The length of the arrow base. */
    protected static final double ARROW_BASE_LENGTH=3.0;
    /** An instance. */
    public static final EdgePainter INSTANCE=new ArrowEdgePainter();

    /**
     * Paints the supplied edge.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param edge                  the edge to paint
     */
    public void paintEdge(JGraphPane graphPane,Graphics2D g,Edge edge) {
        HighlightingManipulator highlightingManipulator=(HighlightingManipulator)graphPane.getManipulator(HighlightingManipulator.NAME);
        boolean isHighlighted=highlightingManipulator!=null && highlightingManipulator.getHighlightedEdge()==edge;
        DraggingManipulator draggingManipulator=(DraggingManipulator)graphPane.getManipulator(DraggingManipulator.NAME);
        boolean isDragging=draggingManipulator!=null && draggingManipulator.getDraggedEdge()==edge;
        Point from=graphPane.getScreenPointForNode(edge.getFrom());
        Point to=graphPane.getScreenPointForNode(edge.getTo());
        Color color=g.getColor();
        g.setColor(getEdgeColor(edge,isHighlighted,isDragging));
        paintArrow(g,from.x,from.y,to.x,to.y);
        g.setColor(color);
    }
    /**
     * Returns the color for the edge.
     *
     * @param edge                  the edge to be painted
     * @param isHighlighted         <code>true</code> if the edge is highlighted
     * @param isDragging            <code>true</code> if the edge is being dragged
     * @return                      the color for the edge
     */
    protected Color getEdgeColor(Edge edge,boolean isHighlighted,boolean isDragging) {
        if (isHighlighted || isDragging)
            return Color.red;
        else
            return Color.gray;
    }
    /**
     * Paints the arrow.
     *
     * @param g                     the graphics
     * @param x1                    the source x coordinate
     * @param y1                    the source y coordinate
     * @param x2                    the target x coordinate
     * @param y2                    the target y coordinate
     */
    public static void paintArrow(Graphics2D g,int x1,int y1,int x2,int y2) {
        double dx;
        double dy;
        double deltaX=x1-x2;
        double deltaY=y1-y2;
        if (Math.abs(deltaY)>Math.abs(deltaX)) {
            double slope=Math.abs(deltaX/deltaY);
            dx=ARROW_BASE_LENGTH/Math.sqrt(1+slope*slope);
            dy=dx*slope;
        }
        else {
            double slope=Math.abs(deltaY/deltaX);
            dy=ARROW_BASE_LENGTH/Math.sqrt(1+slope*slope);
            dx=dy*slope;
        }
        if (deltaY>0)
            dx*=-1;
        if (deltaX<0)
            dy*=-1;
        int[] pointsX=new int[] { x2,(int)(x1-dx),(int)(x1+dx) };
        int[] pointsY=new int[] { y2,(int)(y1-dy),(int)(y1+dy) };
        g.fillPolygon(pointsX,pointsY,3);
    }
    /**
     * Returns the distance of the point to the edge.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics object
     * @param edge                  the edge
     * @param point                 the point
     * @return                      the distance of the point from the edge
     */
    public double screenDistanceFromEdge(JGraphPane graphPane,Graphics2D g,Edge edge,Point point) {
        double px=point.x;
        double py=point.y;
        Point from=graphPane.getScreenPointForNode(edge.getFrom());
        Point to=graphPane.getScreenPointForNode(edge.getTo());
        double x1=from.x;
        double y1=from.y;
        double x2=to.x;
        double y2=to.y;
        if (px<Math.min(x1,x2)-8 || px>Math.max(x1,x2)+8 || py<Math.min(y1,y2)-8 || py>Math.max(y1,y2)+8)
            return 1000;
        double dist=1000;
        if (x1-x2!=0)
            dist=Math.abs((y2-y1)/(x2-x1)*(px-x1)+(y1-py));
        if (y1-y2!=0)
            dist=Math.min(dist,Math.abs((x2-x1)/(y2-y1)*(py-y1)+(x1-px)));
        return dist;
    }
    /**
     * Returns the outer rectangle of the edge on screen.
     *
     * @param graphPane             the graph pane
     * @param edge                  the edge
     * @param edgeScreenRectangle   the rectangle receiving the edge's coordinates
     */
    public void getEdgeScreenBounds(JGraphPane graphPane,Edge edge,Rectangle edgeScreenRectangle) {
        Point from=graphPane.getScreenPointForNode(edge.getFrom());
        Point to=graphPane.getScreenPointForNode(edge.getTo());
        edgeScreenRectangle.setBounds(Math.min(from.x,to.x),Math.min(from.y,to.y),Math.abs(to.x-from.x)+1,Math.abs(to.y-from.y)+1);
    }
}
