package de.fzi.wim.guibase.graphview.layout;

import java.util.Collection;

import de.fzi.wim.guibase.graphview.graph.*;

/**
 * The layouter applies the layout strategy to the graph in a separate thread,
 */
public class Layouter {
    /** The graph that this layouter takes care of. */
    protected Graph m_graph;
    /** The layout strategy for this layouter. */
    protected LayoutStrategy m_layoutStrategy;
    /** The thread of the layouter. */
    protected Thread m_layouterThread;

    /**
     * Creates the layouter.
     *
     * @param layoutStrategy            the layout strategy
     */
    public Layouter(LayoutStrategy layoutStrategy) {
        m_layoutStrategy=layoutStrategy;
        m_graph=m_layoutStrategy.getGraph();
        m_graph.addGraphListener(new GraphHandler());
    }
    /**
     * Starts the layouter thread.
     */
    public synchronized void start() {
        if (m_layouterThread==null) {
            m_layouterThread=new LayouterThread();
            m_layouterThread.start();
        }
    }
    /**
     * Stops the layouter thread.
     */
    public void stop() {
        Thread layouterThread;
        synchronized (this) {
            layouterThread=m_layouterThread;
            m_layouterThread=null;
        }
        if (layouterThread!=null) {
            layouterThread.interrupt();
            try {
                layouterThread.join();
            }
            catch (InterruptedException ignored) {
            }
        }
    }

    /**
     * The layouter thread.
     */
    protected class LayouterThread extends Thread {
        public LayouterThread() {
            super("GraphLayouter");
            setDaemon(true);
        }
        public void run() {
            synchronized (m_graph) {
                m_layoutStrategy.graphContentsChanged();
                m_layoutStrategy.notifyGraphLayoutUpdated();
            }
            while (true) {
                try {
                    boolean hasMoreSteps=false;
                    synchronized (m_graph) {
                        hasMoreSteps=m_layoutStrategy.shouldExecuteStep();
                        if (hasMoreSteps) {
                            m_layoutStrategy.executeGraphLayoutStep();
                            m_graph.notifyLayoutUpdated();
                        }
                    }
                    if (hasMoreSteps)
                        Thread.sleep(20);
                    else
                        synchronized (Layouter.this) {
                            Layouter.this.wait();
                        }
                }
                catch (InterruptedException e) {
                    break;
                }
                catch (Throwable shouldntHappen) {
                    shouldntHappen.printStackTrace();
                }
            }
        }
    }

    /**
     * The handler for graph events.
     */
    protected class GraphHandler implements GraphListener {
        public void graphLayoutUpdated(Graph graph) {
            synchronized (m_graph) {
                synchronized (Layouter.this) {
                    if (m_layouterThread!=Thread.currentThread()) {
                        m_layoutStrategy.notifyGraphLayoutUpdated();
                        Layouter.this.notifyAll();
                    }
                }
            }
        }
        public void graphUpdated(Graph graph) {
        }
        public void graphContentsChanged(Graph graph) {
            synchronized (m_graph) {
                m_layoutStrategy.graphContentsChanged();
                notifyGraphChanged();
            }
        }
        public void elementsAdded(Graph graph,Collection nodes,Collection edges) {
            synchronized (m_graph) {
                m_layoutStrategy.elementsAdded(nodes,edges);
                notifyGraphChanged();
            }
        }
        public void elementsRemoved(Graph graph,Collection nodes,Collection edges) {
            synchronized (m_graph) {
                m_layoutStrategy.elementsRemoved(nodes,edges);
                notifyGraphChanged();
            }
        }
        protected void notifyGraphChanged() {
            synchronized (Layouter.this) {
                m_layoutStrategy.notifyGraphLayoutUpdated();
                Layouter.this.notifyAll();
            }
        }
    }
}
