package de.fzi.wim.guibase.graphview.inplaceedit;

import java.util.Collection;
import java.util.Iterator;
import java.awt.Rectangle;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

/**
 * Provides in-place editing of nodes.
 */
public abstract class InPlaceNodeEditor implements GraphListener {
    /** The graph pane. */
    protected JGraphPane m_graphPane;
    /** The node being edited. */
    protected Node m_node;
    /** The editing controller. */
    protected NodeEditingController m_nodeEditingController;
    /** The component for the editor. */
    protected JComponent m_component;
    /** Set to <code>true</code> if editing is in progress. */
    protected boolean m_editingInProgress;
    /** Set to <code>true</code> if editing is in the process of being stopped. */
    protected boolean m_editingBeingStopped;
    /** The initial value. */
    protected Object m_initialValue;

    public InPlaceNodeEditor(JGraphPane graphPane,Node node,NodeEditingController nodeEditingController) {
        m_graphPane=graphPane;
        m_node=node;
        m_nodeEditingController=nodeEditingController;
        m_component=createEditorComponent();
    }
    /**
     * Returns the graph pane.
     *
     * @return                                      the graph pane
     */
    public JGraphPane getGraphPane() {
        return m_graphPane;
    }
    /**
     * Returns the node editing controller.
     *
     * @return                                      the node editing controller
     */
    public NodeEditingController getNodeEditingController() {
        return m_nodeEditingController;
    }
    /**
     * Returns the node being edited.
     *
     * @return                                      the node being edited
     */
    public Node getNode() {
        return m_node;
    }
    /**
     * Returns <code>true</code> if editing is in progress.
     *
     * @return                                       <code>true</code> if editing is in progress
     */
    public boolean getEditingInProgress() {
        return m_editingInProgress;
    }
    /**
     * Called when graph has changed, but no nodes have changed their position.
     *
     * @param graph                     the graph
     */
    public void graphUpdated(Graph graph) {
    }
    /**
     * Called when graph layout is updated.
     *
     * @param graph                                 the graph
     */
    public void graphLayoutUpdated(Graph graph) {
        if (SwingUtilities.isEventDispatchThread())
            updatePosition();
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    updatePosition();
                }
            });
    }
    /**
     * Called when graph contents is radically changed.
     *
     * @param graph                                 the graph
     */
    public void graphContentsChanged(Graph graph) {
        if (SwingUtilities.isEventDispatchThread())
            abortEditing();
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    abortEditing();
                }
            });
    }
    /**
     * Called when elements are added to the graph.
     *
     * @param graph                                 the graph
     * @param nodes                                 the nodes added (may be <code>null</code>)
     * @param edges                                 the edges added (may be <code>null</code>)
     */
    public void elementsAdded(Graph graph,Collection nodes,Collection edges) {
    }
    /**
     * Called when elements are removed from the graph.
     *
     * @param graph                                 the graph
     * @param nodes                                 the nodes added (may be <code>null</code>)
     * @param edges                                 the edges added (may be <code>null</code>)
     */
    public void elementsRemoved(final Graph graph,final Collection nodes,final Collection edges) {
        if (SwingUtilities.isEventDispatchThread())
            elementsRemovedInternal(graph,nodes,edges);
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    elementsRemovedInternal(graph,nodes,edges);
                }
            });
    }
    /**
     * Processes the elements removed event.
     *
     * @param graph                                 the graph
     * @param nodes                                 the nodes added (may be <code>null</code>)
     * @param edges                                 the edges added (may be <code>null</code>)
     */
    protected void elementsRemovedInternal(Graph graph,Collection nodes,Collection edges) {
        if (nodes!=null) {
            Iterator iterator=nodes.iterator();
            while (iterator.hasNext()) {
                Node node=(Node)iterator.next();
                if (node.equals(m_node))
                    abortEditing();
            }
        }
    }
    /**
     * Called to start the editing process.
     */
    public void startEditing() {
        if (!m_editingInProgress) {
            m_initialValue=m_nodeEditingController.editingStarted(m_node);
            setEditorComponentValue(m_component,m_initialValue);
            updatePosition();
            m_graphPane.add(m_component);
            editingStarting(m_component);
            m_graphPane.getGraph().addGraphListener(this);
            m_editingInProgress=true;
        }
    }
    /**
     * Called to start the editing process.
     *
     * @return                                      <code>true</code> if editing was really stopped
     */
    public boolean stopEditing() {
        if (m_editingInProgress && !m_editingBeingStopped) {
            m_editingBeingStopped=true;
            try {
                Object value=getEditorComponentValue(m_component);
                if (!m_nodeEditingController.editingFinished(m_node,value))
                    return false;
                m_graphPane.remove(m_component);
                m_graphPane.getGraph().removeGraphListener(this);
                m_graphPane.getGraph().notifyLayoutUpdated();
                m_editingInProgress=false;
            }
            finally {
                m_editingBeingStopped=false;
            }
        }
        return true;
    }
    /**
     * Called to abort the editing process.
     */
    public void abortEditing() {
        if (m_editingInProgress && !m_editingBeingStopped) {
            m_editingBeingStopped=true;
            try {
                m_nodeEditingController.editingAborted(m_node);
                m_graphPane.remove(m_component);
                m_graphPane.getGraph().removeGraphListener(this);
                m_graphPane.getGraph().notifyLayoutUpdated();
                m_editingInProgress=false;
            }
            finally {
                m_editingBeingStopped=false;
            }
        }
    }
    /**
     * Called to stop or abort editing, depending on whether the value has change.
     */
    public void stopOrAbortEditing() {
        if (getEditingInProgress()) {
            Object value=getEditorComponentValue(m_component);
            if (m_initialValue==value || (m_initialValue!=null && m_initialValue.equals(value)))
                abortEditing();
            else {
                if (!stopEditing())
                    abortEditing();
            }
        }
    }
    /**
     * Updates the position of the node.
     */
    protected void updatePosition() {
        Rectangle nodeBounds=new Rectangle();
        m_graphPane.getNodeScreenBounds(m_node,nodeBounds);
        m_component.setLocation(nodeBounds.x,nodeBounds.y);
        m_component.setSize(nodeBounds.width,nodeBounds.height);
    }
    /**
     * Creates the editor component.
     *
     * @return                                      the editor component
     */
    protected abstract JComponent createEditorComponent();
    /**
     * Notifies the component that editing is starting..
     *
     * @param component                             the editor component
     */
    protected abstract void editingStarting(JComponent component);
    /**
     * Sets the value into the component.
     *
     * @param component                             the editor component
     * @param value                                 the value
     */
    protected abstract void setEditorComponentValue(JComponent component,Object value);
    /**
     * Returns the current value of the component.
     *
     * @param component                             the editor component
     * @return                                      the value
     */
    protected abstract Object getEditorComponentValue(JComponent component);
}
