package de.fzi.wim.guibase.graphview.controller;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.awt.Point;
import java.awt.Stroke;
import java.awt.Rectangle;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.event.MouseEvent;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;
import de.fzi.wim.guibase.graphview.selection.*;

/**
 * A manipulator that provides the selection user interface.
 */
public class SelectionManipulator extends AbstractManipulator {
    /** The name of this manipulator. */
    public static final String NAME="SelectionManipulator";
    /** The dashed stroke. */
    protected static final Stroke DASHED_STROKE=new BasicStroke(1.0f,BasicStroke.CAP_SQUARE,BasicStroke.JOIN_BEVEL,2.0f,new float[] { 15.0f,10.0f },0.0f);

    /** The node selection model.*/
    protected NodeSelectionModel m_nodeSelectionModel;
    /** The graph position of the fixed point. */
    protected Point2D m_selectionRectangleFixedGraph;
    /** The fixed point of the selection rectangle. */
    protected Point m_selectionRectangleFixed;
    /** The moving point of the selection rectangle. */
    protected Point m_selectionRectangleMoving;

    /**
     * Creates an instance of this class.
     *
     * @param nodeSelectionModel                the model for node selection
     */
    public SelectionManipulator(NodeSelectionModel nodeSelectionModel) {
        m_nodeSelectionModel=nodeSelectionModel;
        m_nodeSelectionModel.addNodeSelectionListener(new NodeSelectionHandler());
        m_selectionRectangleFixedGraph=new Point2D.Double();
    }
    protected boolean shouldSelectNode(Node node) {
        return true;
    }
    public String getName() {
        return NAME;
    }
    public NodeSelectionModel getNodeSelectionModel() {
        return m_nodeSelectionModel;
    }
    public void paint(Graphics2D g) {
        if (m_selectionRectangleFixed!=null) {
            Color oldColor=g.getColor();
            g.setColor(Color.gray);
            Stroke oldStroke=g.getStroke();
            g.setStroke(DASHED_STROKE);
            int x=Math.min(m_selectionRectangleFixed.x,m_selectionRectangleMoving.x);
            int y=Math.min(m_selectionRectangleFixed.y,m_selectionRectangleMoving.y);
            int width=Math.abs(m_selectionRectangleFixed.x-m_selectionRectangleMoving.x);
            int height=Math.abs(m_selectionRectangleFixed.y-m_selectionRectangleMoving.y);
            g.drawRect(x,y,width,height);
            g.setStroke(oldStroke);
            g.setColor(oldColor);
        }
    }
    public void mousePressed(MouseEvent e) {
        if (m_graphPane.isEnabled() && (e.getModifiers() & MouseEvent.BUTTON1_MASK)!=0 && isStartSelectionEvent(e)) {
            Node node=m_graphPane.getNodeAtPoint(e.getPoint());
            if (node!=null) {
                Collection rawSelection=Collections.singletonList(node);
                changeSelection(rawSelection,e);
            }
            else {
                Edge edge=m_graphPane.getNearestEdge(e.getPoint());
                if (edge==null) {
                    updateLastMouseEventScreenPoint(e);
                    m_selectionRectangleFixed=e.getPoint();
                    m_graphPane.screenToGraphPoint(m_selectionRectangleFixed,m_selectionRectangleFixedGraph);
                    m_selectionRectangleMoving=m_selectionRectangleFixed;
                    e.consume();
                }
            }
        }
    }
    public void mouseReleased(MouseEvent e) {
        if (m_selectionRectangleFixed!=null && (e.getModifiers() & MouseEvent.BUTTON1_MASK)!=0) {
            m_selectionRectangleMoving=e.getPoint();
            int x=Math.min(m_selectionRectangleFixed.x,m_selectionRectangleMoving.x);
            int y=Math.min(m_selectionRectangleFixed.y,m_selectionRectangleMoving.y);
            int width=Math.abs(m_selectionRectangleFixed.x-m_selectionRectangleMoving.x);
            int height=Math.abs(m_selectionRectangleFixed.y-m_selectionRectangleMoving.y);
            Rectangle rectangle=new Rectangle(x,y,width,height);
            Collection nodesInRectangle=m_graphPane.getNodesInRectangle(rectangle);
            changeSelection(nodesInRectangle,e);
            m_selectionRectangleFixed=null;
            m_selectionRectangleMoving=null;
            m_lastMouseEventScreenPoint=null;
            m_graphPane.repaint();
            e.consume();
        }
    }
    public void mouseDragged(MouseEvent e) {
        if (m_selectionRectangleFixed!=null) {
            autoscroll(e);
            updateLastMouseEventScreenPoint(e);
            m_selectionRectangleMoving=e.getPoint();
            m_graphPane.repaint();
            e.consume();
        }
    }
    public void notifyGraphPaneScrolled() {
        if (m_selectionRectangleFixed!=null) {
            m_graphPane.graphToScreenPoint(m_selectionRectangleFixedGraph,m_selectionRectangleFixed);
            m_selectionRectangleMoving=getLastMouseEventPoint();
            m_graphPane.repaint();
        }
    }
    protected void changeSelection(Collection rawSelection,MouseEvent e) {
        Collection newSelection=new HashSet();
        Iterator iterator=rawSelection.iterator();
        while (iterator.hasNext()) {
            Node node=(Node)iterator.next();
            if (shouldSelectNode(node))
                newSelection.add(node);
        }
        if (!newSelection.isEmpty()) {
            Collection existingSelection=m_nodeSelectionModel.getSelectedNodes();
            if (isToggleSelectionEvent(e)) {
                Collection addElements=new HashSet();
                Collection removeElements=new HashSet();
                iterator=newSelection.iterator();
                while (iterator.hasNext()) {
                    Node node=(Node)iterator.next();
                    if (existingSelection.contains(node))
                        removeElements.add(node);
                    else
                        addElements.add(node);
                }
                if (!removeElements.isEmpty())
                    m_nodeSelectionModel.removeNodes(removeElements);
                if (!addElements.isEmpty())
                    m_nodeSelectionModel.addNodes(addElements);
            }
            else {
                if (!existingSelection.containsAll(newSelection) || existingSelection.size()!=newSelection.size()) {
                    m_nodeSelectionModel.clear();
                    m_nodeSelectionModel.addNodes(newSelection);
                }
            }
        }
    }
    protected boolean isStartSelectionEvent(MouseEvent e) {
        return (e.getModifiers() & MouseEvent.SHIFT_MASK)==0;
    }
    protected boolean isToggleSelectionEvent(MouseEvent e) {
        return (e.getModifiers() & MouseEvent.CTRL_MASK)!=0;
    }

    /**
     * The handles of the node selection events.
     */
    protected class NodeSelectionHandler implements NodeSelectionListener {
        public void nodesAddedToSelection(NodeSelectionModel nodeSelectionModel,Collection nodes) {
            m_graphPane.repaint();
        }
        public void nodesRemovedFromSelection(NodeSelectionModel nodeSelectionModel,Collection nodes) {
            m_graphPane.repaint();
        }
        public void selectionCleared(NodeSelectionModel nodeSelectionModel) {
            m_graphPane.repaint();
        }
    }
}
