package de.fzi.wim.guibase.graphview.controller;

import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Comparator;
import java.util.Collections;
import java.awt.Point;
import java.awt.Component;
import java.awt.Window;
import java.awt.Rectangle;
import java.awt.KeyboardFocusManager;
import java.awt.geom.Point2D;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.FocusEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentAdapter;
import javax.swing.JWindow;
import javax.swing.JToolTip;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

/**
 * A manipulator that provides incremental search functionality.
 */
public class IncrementalSearchManipulator extends AbstractManipulator {
    /** The name of this manipulator. */
    public static final String NAME="IncrementalSearchManipulator";

    /** The 'Search For:' label. */
    protected String m_searchForLabel;
    /** The popup showing the term. */
    protected PopupWindow m_searchTermPopup;
    /** The old forward focus traversal keys. */
    protected Set m_oldForwardTraversalKeys;
    /** The old backward focus traversal keys. */
    protected Set m_oldBackwardTraversalKeys;
    /** The component showing the search information. */
    protected JToolTip m_searchTermWindow;
    /** The search term. */
    protected String m_searchTerm;
    /** The set of highlighted nodes. */
    protected Set m_highlightedNodes;
    /** The list of nodes sorted by their position. */
    protected List m_orderedHighlightedNodes;
    /** The index of the current node. */
    protected int m_currentNode;

    /**
     * Creates an instance of this class.
     *
     * @param searchForLabel                    the label of for the 'Search For:'
     */
    public IncrementalSearchManipulator(String searchForLabel) {
        m_searchForLabel=searchForLabel;
        m_highlightedNodes=new HashSet();
        m_orderedHighlightedNodes=new ArrayList();
        m_searchTermWindow=new JToolTip();
        m_searchTerm="";
    }
    /**
     * Returns the name of the manipulator.
     *
     * @return                                  the name of the manipulator
     */
    public String getName() {
        return NAME;
    }
    /**
     * Returns <code>true</code> if the search is active.
     *
     * @return                                  <code>true</code> if the search is active
     */
    public boolean getSearchActive() {
        return m_searchTermPopup!=null;
    }
    /**
     * Returns the current search term.
     *
     * @return                                  the current search term
     */
    public String getSearchTerm() {
        return m_searchTerm;
    }
    /**
     * Sets the current search term.
     *
     * @param searchTerm                        the current search term
     */
    public void setSearchTerm(String searchTerm) {
        m_searchTerm=searchTerm;
        m_searchTermWindow.setTipText("<html><body style='margin: 1px 3px 2px 3px'><b>"+m_searchForLabel+"</b> "+m_searchTerm+"</body></html>");
        updateHighlightedNodes();
        m_graphPane.repaint();
        if (m_searchTermPopup!=null)
            m_searchTermPopup.pack();
    }
    /**
     * Stops the search.
     */
    public void stopSearch() {
        if (getSearchActive()) {
            m_searchTermPopup.dispose();
            m_searchTermPopup=null;
            setSearchTerm("");
            m_graphPane.setFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS,m_oldForwardTraversalKeys);
            m_graphPane.setFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS,m_oldBackwardTraversalKeys);
            m_oldForwardTraversalKeys=null;
            m_oldBackwardTraversalKeys=null;
            m_graphPane.repaint();
        }
    }
    /**
     * Starts the search.
     */
    public void startSearch() {
        if (!getSearchActive()) {
            Component owner=m_graphPane;
            while (owner!=null && m_searchTermPopup==null) {
                if (owner instanceof Window)
                    m_searchTermPopup=new PopupWindow((Window)owner);
                else
                    owner=owner.getParent();
            }
            if (m_searchTermPopup!=null) {
                m_searchTermPopup.setVisible(true);
                m_oldForwardTraversalKeys=m_graphPane.getFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS);
                m_oldBackwardTraversalKeys=m_graphPane.getFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS);
                m_graphPane.setFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS,Collections.EMPTY_SET);
                m_graphPane.setFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS,Collections.EMPTY_SET);
            }
            m_graphPane.repaint();
        }
    }
    /**
     * Returns the set of higlighted nodes.
     *
     * @return                                  the set of iglighted nodes
     */
    public Set getHighlightedNodes() {
        return m_highlightedNodes;
    }
    /**
     * Updates the set of highlighted nodes.
     */
    public void updateHighlightedNodes() {
        String searchTerm=m_searchTerm.toLowerCase();
        m_highlightedNodes.clear();
        Iterator nodes=m_graphPane.getGraph().getNodes().iterator();
        while (nodes.hasNext()) {
            Node node=(Node)nodes.next();
            String label=node.getLabel();
            if (label!=null) {
                label=label.toLowerCase();
                if (label.startsWith(searchTerm))
                    m_highlightedNodes.add(node);
            }
        }
        m_orderedHighlightedNodes.clear();
        m_currentNode=-1;
        if (!m_highlightedNodes.isEmpty()) {
            m_orderedHighlightedNodes.addAll(m_highlightedNodes);
            Collections.sort(m_orderedHighlightedNodes,NodeComparator.INSTANCE);
            Rectangle visibleRectangle=m_graphPane.getVisibleRect();
            Point screenCenter=new Point(visibleRectangle.x+visibleRectangle.width/2,visibleRectangle.y+visibleRectangle.height/2);
            Point2D graphPoint=new Point2D.Double();
            m_graphPane.screenToGraphPoint(screenCenter,graphPoint);
            double minDistanceSquared=Double.MAX_VALUE;
            for (int i=0;i<m_orderedHighlightedNodes.size();i++) {
                Node node=(Node)m_orderedHighlightedNodes.get(i);
                double dx=node.getX()-graphPoint.getX();
                double dy=node.getY()-graphPoint.getY();
                double distanceSquared=dx*dx+dy*dy;
                if (distanceSquared<minDistanceSquared) {
                    minDistanceSquared=distanceSquared;
                    m_currentNode=i;
                }
            }
            if (m_currentNode!=-1) {
                Node node=(Node)m_orderedHighlightedNodes.get(m_currentNode);
                centerNodeInView(node);
            }
        }
    }
    /**
     * Scroll node into view.
     *
     * @param node                              the node to scroll into view
     */
    public void centerNodeInView(Node node) {
        Point point=m_graphPane.getScreenPointForNode(node);
        Rectangle rectangle=m_graphPane.getVisibleRect();
        rectangle.setBounds(point.x-rectangle.width/2,point.y-rectangle.height/2,rectangle.width,rectangle.height);
        m_graphPane.scrollRectToVisible(rectangle);
    }
    /**
     * Highlights the next node.
     */
    public void highlightNextNode() {
        if (m_currentNode!=-1) {
            m_currentNode=(m_currentNode+1) % m_highlightedNodes.size();
            Node node=(Node)m_orderedHighlightedNodes.get(m_currentNode);
            centerNodeInView(node);
        }
    }
    /**
     * Highlights the next node.
     */
    public void highlightPreviousNode() {
        if (m_currentNode!=-1) {
            m_currentNode=m_currentNode-1;
            if (m_currentNode<0)
                m_currentNode=m_highlightedNodes.size()-1;
            Node node=(Node)m_orderedHighlightedNodes.get(m_currentNode);
            centerNodeInView(node);
        }
    }
    public void keyPressed(KeyEvent e) {
        if (!getSearchActive() || !m_graphPane.isEnabled())
            return;
        switch (e.getKeyCode()) {
        case KeyEvent.VK_TAB:
            if ((e.getModifiers() & KeyEvent.VK_SHIFT)!=0)
                highlightPreviousNode();
            else
                highlightNextNode();
            break;
        case KeyEvent.VK_ENTER:
            {
                SelectionManipulator selectionManipulator=(SelectionManipulator)m_graphPane.getManipulator(SelectionManipulator.NAME);
                if (selectionManipulator!=null) {
                    selectionManipulator.getNodeSelectionModel().clear();
                    selectionManipulator.getNodeSelectionModel().addNodes(m_highlightedNodes);
                }
                stopSearch();
            }
            break;
        case KeyEvent.VK_BACK_SPACE:
            if (m_searchTerm.length()>0) {
                m_searchTerm=m_searchTerm.substring(0,m_searchTerm.length()-1);
                setSearchTerm(m_searchTerm);
            }
            break;
        }
    }
    public void keyTyped(KeyEvent e) {
        char character=e.getKeyChar();
        if (character==KeyEvent.VK_ESCAPE)
            stopSearch();
        else if (character>=' ' && m_graphPane.isEnabled()) {
            if (!getSearchActive())
                startSearch();
            m_searchTerm=m_searchTerm+character;
            setSearchTerm(m_searchTerm);
        }
    }
    public void mousePressed(MouseEvent e) {
        if (getSearchActive())
            stopSearch();
    }
    public void mouseReleased(MouseEvent e) {
        if (getSearchActive())
            stopSearch();
    }
    public void focusLost(FocusEvent e) {
        if (getSearchActive())
            stopSearch();
    }

    public class PopupWindow extends JWindow {
        protected ComponentAdapter m_adapter;

        public PopupWindow(Window owner) {
            super(owner);
            m_adapter=new ComponentAdapter() {
                public void componentMoved(ComponentEvent e) {
                    updateLocation();
                }
            };
            getOwner().addComponentListener(m_adapter);
            updateLocation();
            setContentPane(m_searchTermWindow);
        }
        public void dispose() {
            super.dispose();
            getOwner().removeComponentListener(m_adapter);
        }
        public void updateLocation() {
            Point point=m_graphPane.getLocationOnScreen();
            Rectangle rectangle=m_graphPane.getVisibleRect();
            point.x+=rectangle.x;
            point.y+=rectangle.y;
            setLocation(point.x+50,point.y-30);
            pack();
        }
    }

    /**
     * The comparator for the nodes.
     */
    protected static class NodeComparator implements Comparator {
        public static final Comparator INSTANCE=new NodeComparator();
        public static final int STRIP_WIDTH=50;
        public int compare(Object o1,Object o2) {
            Node node1=(Node)o1;
            Node node2=(Node)o2;
            int row=(int)node1.getY();
            row=(row/STRIP_WIDTH)*STRIP_WIDTH;
            if (node2.getY()<row)
                return -1;
            if (row+STRIP_WIDTH<node2.getY())
                return 1;
            if (node1.getX()<node2.getX())
                return -1;
            else
                return 1;
        }
    };
}
