package de.fzi.wim.guibase.graphview.controller;

import java.awt.Cursor;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.awt.event.MouseEvent;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

/**
 * A manipulator enabling the dragging of the graph
 */
public class DraggingManipulator extends AbstractManipulator {
    /** The name of this manipulator. */
    public static final String NAME="DraggingManipulator";

    /** The position of the last mouse event. */
    protected Point m_lastPosition;
    /** The node that is being dragged. */
    protected Node m_draggedNode;
    /** The edge that is being dragged. */
    protected Edge m_draggedEdge;
    /** Point of the original grab. */
    protected Point m_grabPoint1;
    /** Point of the original grab. */
    protected Point m_grabPoint2;
    /** The old 'fixed' state of the node. */
    protected boolean m_oldFixedState1;
    /** The old 'fixed' state of the node. */
    protected boolean m_oldFixedState2;
    /** The original cursor of the graph. */
    protected Cursor m_graphCursor;

    /**
     * Creates an instance of this class.
     */
    public DraggingManipulator() {
    }
    public String getName() {
        return NAME;
    }
    public boolean isDragging() {
        return m_draggedNode!=null || m_draggedEdge!=null;
    }
    public Node getDraggedNode() {
        return m_draggedNode;
    }
    public Edge getDraggedEdge() {
        return m_draggedEdge;
    }
    public void mousePressed(MouseEvent e) {
        if (m_graphPane.isEnabled() && (e.getModifiers() & MouseEvent.BUTTON1_MASK)!=0 && isStartDraggingEvent(e)) {
            Point point=e.getPoint();
            m_draggedNode=m_graphPane.getNodeAtPoint(point);
            if (m_draggedNode!=null) {
                m_lastPosition=e.getPoint();
                Point nodeScreenPoint=m_graphPane.getScreenPointForNode(m_draggedNode);
                m_grabPoint1=new Point(point.x-nodeScreenPoint.x,point.y-nodeScreenPoint.y);
                m_oldFixedState1=m_draggedNode.isFixed();
                m_draggedNode.setFixed(true);
                m_graphCursor=m_graphPane.getCursor();
                m_graphPane.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                e.consume();
            }
            else {
                m_draggedEdge=m_graphPane.getNearestEdge(point);
                if (m_draggedEdge!=null) {
                    m_lastPosition=e.getPoint();
                    Point fromPoint=m_graphPane.getScreenPointForNode(m_draggedEdge.getFrom());
                    m_grabPoint1=new Point(point.x-fromPoint.x,point.y-fromPoint.y);
                    Point toPoint=m_graphPane.getScreenPointForNode(m_draggedEdge.getTo());
                    m_grabPoint2=new Point(point.x-toPoint.x,point.y-toPoint.y);
                    m_oldFixedState1=m_draggedEdge.getFrom().isFixed();
                    m_oldFixedState1=m_draggedEdge.getTo().isFixed();
                    m_draggedEdge.getFrom().setFixed(true);
                    m_draggedEdge.getTo().setFixed(true);
                    m_graphCursor=m_graphPane.getCursor();
                    m_graphPane.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                    e.consume();
                }
            }
        }
    }
    public void mouseReleased(MouseEvent e) {
        if (m_draggedNode!=null) {
            moveDraggedNode(e.getPoint());
            m_draggedNode.setFixed(m_oldFixedState1);
            m_graphPane.setCursor(m_graphCursor);
            m_graphCursor=null;
            m_grabPoint1=null;
            m_draggedNode=null;
            m_lastPosition=null;
            e.consume();
        }
        if (m_draggedEdge!=null) {
            moveDraggedEdge(e.getPoint());
            m_draggedEdge.getFrom().setFixed(m_oldFixedState1);
            m_draggedEdge.getTo().setFixed(m_oldFixedState2);
            m_graphPane.setCursor(m_graphCursor);
            m_graphCursor=null;
            m_grabPoint1=null;
            m_grabPoint2=null;
            m_draggedEdge=null;
            m_lastPosition=null;
            e.consume();
        }
    }
    public void mouseDragged(MouseEvent e) {
        if (m_draggedNode!=null) {
            autoscroll(e);
            moveDraggedNode(e.getPoint());
            e.consume();
        }
        if (m_draggedEdge!=null) {
            autoscroll(e);
            moveDraggedEdge(e.getPoint());
            e.consume();
        }
    }
    protected boolean isStartDraggingEvent(MouseEvent e) {
        return (e.getModifiers() & MouseEvent.SHIFT_MASK)!=0;
    }
    /**
     * Moves the dragged node to given point.
     *
     * @param point                         the screen point where the node should be dragged
     */
    protected void moveDraggedNode(Point point) {
        if (!point.equals(m_lastPosition)) {
            m_lastPosition.setLocation(point.x,point.y);
            point.x-=m_grabPoint1.x;
            point.y-=m_grabPoint1.y;
            Point2D graphPoint=new Point2D.Double();
            m_graphPane.screenToGraphPoint(point,graphPoint);
            m_draggedNode.setLocation(graphPoint.getX(),graphPoint.getY());
            m_graphPane.getGraph().notifyLayoutUpdated();
        }
    }
    /**
     * Moves the dragged edge to given point.
     *
     * @param point                         the screen point where the edge should be dragged
     */
    protected void moveDraggedEdge(Point point) {
        if (!point.equals(m_lastPosition)) {
            m_lastPosition.setLocation(point.x,point.y);
            Point2D graphPoint=new Point2D.Double();
            m_graphPane.screenToGraphPoint(new Point(point.x-m_grabPoint1.x,point.y-m_grabPoint1.y),graphPoint);
            m_draggedEdge.getFrom().setLocation(graphPoint.getX(),graphPoint.getY());
            m_graphPane.screenToGraphPoint(new Point(point.x-m_grabPoint2.x,point.y-m_grabPoint2.y),graphPoint);
            m_draggedEdge.getTo().setLocation(graphPoint.getX(),graphPoint.getY());
            m_graphPane.getGraph().notifyLayoutUpdated();
        }
    }
}
