package de.fzi.wim.guibase.dnd;

import java.awt.HeadlessException;
import java.awt.Toolkit;
import java.awt.Component;
import java.awt.Point;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetListener;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JComponent;
import javax.swing.Timer;
import javax.swing.SwingUtilities;

/**
 * The drop target that implements autoscrolling properly. The original drop target
 * doesn't check the visible rectangle of the drop target, but the actual size of the
 * drop target, thus rendering the whole autoscrolling mechanisms useless. Further,
 * Swing controls don't implement Autoscroll interface, which is annoying. This
 * drop target doesn't require that components implement Autoscroll.
 */
public class SmartDropTarget extends DropTarget {
    /** The autoscroller. */
    protected transient SmartDropTargetAutoScroller m_autoScroller;
    /** The insets around the scrolling region. */
    protected Insets m_scrollingInsets;

    /**
     * Creates an instance of this class.
     *
     * @param c                         the component
     * @param dtl                       the drop target listener
     * @throws HeadlessException        thrown if GraphicsEnvironment.isHeadless() returns true
     */
    public SmartDropTarget(Component c,DropTargetListener dtl) throws HeadlessException {
        super(c,dtl);
        setScrollingInsets(new Insets(3,3,3,3));
    }
    /**
     * Sets the srolling insets.
     *
     * @param insets                    the insets
     */
    public void setScrollingInsets(Insets insets) {
        m_scrollingInsets=insets;
    }
    /**
     * Returns the srolling insets.
     *
     * @return                          the insets
     */
    public Insets getScrollingInsets() {
        return m_scrollingInsets;
    }
    /**
     * Creates an autoscroller.
     *
     * @param component                 the component
     * @param point                     the point
     * @return                          the autoscroller
     */
    protected SmartDropTargetAutoScroller createSmartDropTargetAutoScroller(JComponent component,Point point) {
        return new SmartDropTargetAutoScroller(component,point,getScrollingInsets());
    }
    /**
     * Initializes autoscrolling.
     *
     * @param p                         the point
     */
    protected void initializeAutoscrolling(Point p) {
        if (getComponent() instanceof JComponent)
            m_autoScroller=createSmartDropTargetAutoScroller((JComponent)getComponent(),p);
    }
    /**
     * Updates the autoscroller when cursor is at given position.
     *
     * @param dragCursorPosition        the position of the drag cursor
     */
    protected void updateAutoscroll(Point dragCursorPosition) {
        if (m_autoScroller!=null)
            m_autoScroller.updateLocation(dragCursorPosition);
    }
    /**
     * Clears autoscrolling.
     */
    protected void clearAutoscroll() {
        if (m_autoScroller!=null) {
            m_autoScroller.stop();
            m_autoScroller=null;
        }
    }

    /**
     * The autoscroller that takes the visible region into account, for the difference of the original
     * autoscroller that works with the actual size of the component.
     */
    protected static class SmartDropTargetAutoScroller implements ActionListener,Runnable {
        protected JComponent m_component;
        protected Insets m_insets;
        protected Timer m_timer;
        protected Point m_location;
        protected int m_hysteresis;

        /**
         * Creates an instance of this class.
         *
         * @param component             the component
         * @param point                 the point
         * @param insets                the scrolling insets
         */
        protected SmartDropTargetAutoScroller(JComponent component,Point point,Insets insets) {
            m_component=component;
            m_insets=insets;
            Integer initial=new Integer(100);
            Integer interval=new Integer(100);
            Toolkit toolkit=Toolkit.getDefaultToolkit();
            try {
                initial=(Integer)toolkit.getDesktopProperty("DnD.Autoscroll.initialDelay");
            }
            catch (Exception ignored) {
            }
            try {
                interval=(Integer)toolkit.getDesktopProperty("DnD.Autoscroll.interval");
            }
            catch (Exception ignored) {
            }
            m_timer=new Timer(interval.intValue(),this);
            m_timer.setCoalesce(true);
            m_timer.setInitialDelay(initial.intValue());
            m_location=point;
            m_hysteresis=10;
            try {
                m_hysteresis=((Integer)toolkit.getDesktopProperty("DnD.Autoscroll.cursorHysteresis")).intValue();
            }
            catch (Exception ignored) {
            }
            m_timer.start();
        }
        /**
         * Notifies this class that mouse has been moved.
         *
         * @param newLocation           the new location
         */
        public synchronized void updateLocation(Point newLocation) {
            Point previousLocation=m_location;
            m_location=newLocation;
            if (Math.abs(m_location.x-previousLocation.x)>m_hysteresis || Math.abs(m_location.y-previousLocation.y)>m_hysteresis) {
                if (m_timer.isRunning())
                    m_timer.stop();
            } else {
                if (!m_timer.isRunning())
                    m_timer.start();
            }
        }
        /**
         * Stops autoscrolling.
         */
        public void stop() {
            m_timer.stop();
        }
        /**
         * Called by the timer to perform autoscrolling.
         *
         * @param e                     the event
         */
        public void actionPerformed(ActionEvent e) {
            SwingUtilities.invokeLater(this);
        }
        /**
         * Implements the Runnable interface.
         */
        public void run() {
            int x;
            int y;
            synchronized (this) {
                x=m_location.x;
                y=m_location.y;
            }
            performAutoscroll(x,y);
        }
        /**
         * Actually performs the autoscroll on the main thread.
         *
         * @param x                     the X position
         * @param y                     the Y position
         */
        protected void performAutoscroll(int x,int y) {
            Rectangle visibleRectangle=m_component.getVisibleRect();
            if (visibleRectangle.contains(x,y)) {
                if (y<visibleRectangle.y+m_insets.bottom)
                    scrollUp(visibleRectangle);
                if (visibleRectangle.y+visibleRectangle.height-m_insets.bottom<y)
                    scrollDown(visibleRectangle);
                if (x<visibleRectangle.x+m_insets.left)
                    scrollLeft(visibleRectangle);
                if (visibleRectangle.x+visibleRectangle.width-m_insets.right<y)
                    scrollRight(visibleRectangle);
            }
        }
        /**
         * Scrolls up.
         *
         * @param visibleRectangle      the visible rectangle
         */
        protected void scrollUp(Rectangle visibleRectangle) {
            Rectangle newRectangle=new Rectangle(visibleRectangle);
            newRectangle.y-=20;
            m_component.scrollRectToVisible(newRectangle);
        }
        /**
         * Scrolls down.
         *
         * @param visibleRectangle      the visible rectangle
         */
        protected void scrollDown(Rectangle visibleRectangle) {
            Rectangle newRectangle=new Rectangle(visibleRectangle);
            newRectangle.y+=20;
            m_component.scrollRectToVisible(newRectangle);
        }
        /**
         * Scrolls left.
         *
         * @param visibleRectangle      the visible rectangle
         */
        protected void scrollLeft(Rectangle visibleRectangle) {
            Rectangle newRectangle=new Rectangle(visibleRectangle);
            newRectangle.x-=20;
            m_component.scrollRectToVisible(newRectangle);
        }
        /**
         * Scrolls right.
         *
         * @param visibleRectangle      the visible rectangle
         */
        protected void scrollRight(Rectangle visibleRectangle) {
            Rectangle newRectangle=new Rectangle(visibleRectangle);
            newRectangle.x+=20;
            m_component.scrollRectToVisible(newRectangle);
        }
    }
}
