package de.fzi.wim.guibase.dnd;

import java.awt.Color;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetListener;
import javax.swing.JTree;
import javax.swing.JComponent;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreePath;

/**
 * Auxiliary class for drop targets based on a JTree.
 */
public class JTreeDropTarget implements DropTargetListener {
    /** The border for highlighting the active node. */
    protected static final Border HIGHLIGHTED_NODE_BORDER=new HighlightedNodeBorder();

    /** Drop target for the table. */
    protected DropTarget m_dropTarget;
    /** The tree. */
    protected JTree m_tree;
    /** The bordered viewport. */
    protected BorderedViewport m_borderedViewport;
    /** The cell renderer for the tree. */
    protected DraggingTreeCellRenderer m_cellRenderer;
    /** Set to the index of the row over which the dragging is performed. */
    protected int m_draggingRow;

    /**
     * Attachs this manputlator to a tree.
     *
     * @param tree                              the tree
     * @param borderedViewport                  the viewport being highlighted when the drag is over the tree, but not over an element (may be <code>null</code>)
     */
    public JTreeDropTarget(JTree tree,BorderedViewport borderedViewport) {
        m_draggingRow=-1;
        m_tree=tree;
        m_borderedViewport=borderedViewport;
        m_dropTarget=new SmartDropTarget(tree,this);
        m_cellRenderer=new DraggingTreeCellRenderer();
    }
    /**
     * Called when a drag operation has encountered the DropTarget.
     *
     * @param dtde                              the event
     */
    public void dragEnter(DropTargetDragEvent dtde) {
        m_cellRenderer.install();
        processDragEvent(dtde);
    }
    /**
     * The drag operation has departed the DropTarget without dropping.
     *
     * @param dte                               the event
     */
    public void dragExit(DropTargetEvent dte) {
        m_cellRenderer.uninstall();
        if (m_borderedViewport!=null)
            m_borderedViewport.setBorderShown(false);
    }
    /**
     * Called when a drag operation is ongoing on the DropTarget.
     *
     * @param dtde                              the event
     */
    public void dragOver(DropTargetDragEvent dtde) {
        processDragEvent(dtde);
    }
    /**
     * The drag operation has terminated with a drop on this DropTarget.
     *
     * @param dtde                              the event
     */
    public void drop(DropTargetDropEvent dtde) {
        m_cellRenderer.uninstall();
        if (m_borderedViewport!=null)
            m_borderedViewport.setBorderShown(false);
        Point location=dtde.getLocation();
        TreePath droppedPath=null;
        int droppedRow=m_tree.getRowForLocation(location.x,location.y);
        if (droppedRow!=-1)
            droppedPath=m_tree.getPathForRow(droppedRow);
        if (processDrop(dtde,droppedPath))
            dtde.acceptDrop(dtde.getSourceActions());
        else
            dtde.rejectDrop();
    }
    /**
     * Called if the user has modified the current drop gesture.
     *
     * @param dtde                              the event
     */
    public void dropActionChanged(DropTargetDragEvent dtde) {
    }
    /**
     * Processes a drag event.
     *
     * @param dtde                              the event
     */
    protected void processDragEvent(DropTargetDragEvent dtde) {
        Point point=dtde.getLocation();
        int newDraggingRow=m_tree.getRowForLocation(point.x,point.y);
        if (newDraggingRow==-1 && shouldSnapToRootIfNoNode() && m_tree.getRowCount()>0)
            newDraggingRow=0;
        if (!acceptsDrag(dtde,newDraggingRow)) {
            if (m_draggingRow!=-1) {
                m_draggingRow=-1;
                m_tree.repaint();
            }
            if (m_borderedViewport!=null)
                m_borderedViewport.setBorderShown(false);
            dtde.rejectDrag();
        }
        else {
            dtde.acceptDrag(dtde.getSourceActions());
            if (m_draggingRow!=newDraggingRow) {
                m_draggingRow=newDraggingRow;
                m_tree.repaint();
            }
            if (m_borderedViewport!=null && m_draggingRow==-1)
                m_borderedViewport.setBorderShown(true);
        }
    }
    /**
     * Determines whether operation can be accepted.
     *
     * @param dtde                              the event
     * @param draggingRow                       row over which the drag is performed
     * @return                                  <code>true</code> if the drag will be accepted
     */
    protected boolean acceptsDrag(DropTargetDragEvent dtde,int draggingRow) {
        return true;
    }
    /**
     * Determines whether dragging should snap to root if no node is under the cursor.
     *
     * @return                                  <code>true</code> if dragging should snap to root
     */
    protected boolean shouldSnapToRootIfNoNode() {
        return true;
    }
    /**
     * Processes the drop event.
     *
     * @param dtde                              the drop event
     * @param droppedPath                       the path over which the dragging was ended (may be <code>null</code>)
     * @return                                  <code>true</code> if drop was accepted
     */
    protected boolean processDrop(DropTargetDropEvent dtde,TreePath droppedPath) {
        return false;
    }

    /**
     * The cell renderer that highlights the item over which the mouse is positioned.
     */
    protected class DraggingTreeCellRenderer implements TreeCellRenderer {
        protected TreeCellRenderer m_originalRenderer;

        public void install() {
            m_originalRenderer=m_tree.getCellRenderer();
            m_tree.setCellRenderer(this);
            m_draggingRow=-1;
            m_tree.repaint();
        }
        public void uninstall() {
            m_tree.setCellRenderer(m_originalRenderer);
            m_draggingRow=-1;
            getTreeCellRendererComponent(m_tree,m_tree.getModel().getRoot(),false,false,true,0,false);     // to remove the border from the original component
            m_originalRenderer=null;
            m_tree.repaint();
        }
        public Component getTreeCellRendererComponent(JTree tree,Object value,boolean selected,boolean expanded,boolean leaf,int row,boolean hasFocus) {
            Component component=m_originalRenderer.getTreeCellRendererComponent(tree,value,selected,expanded,leaf,row,hasFocus);
            if (component instanceof JComponent)
                if (m_draggingRow==row)
                    ((JComponent)component).setBorder(HIGHLIGHTED_NODE_BORDER);
                else
                    ((JComponent)component).setBorder(null);

            return component;
        }
    }

    protected static class HighlightedNodeBorder implements Border {
        /** The border insets. */
        protected Insets m_insets=new Insets(0,0,0,0);

        public void paintBorder(Component c,Graphics g,int x,int y,int width,int height) {
            Color oldColor=g.getColor();
            g.setColor(UIManager.getColor("Table.gridColor"));
            g.drawRect(x,y,width-1,height-1);
            g.setColor(oldColor);
        }
        public Insets getBorderInsets(Component c) {
            return m_insets;
        }
        public boolean isBorderOpaque() {
            return true;
        }
    }
}
