package de.fzi.wim.guibase.dnd;

import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetListener;
import javax.swing.JTable;

/**
 * Auxiliary class for drop targets based on a JTable.
 */
public class JTableDropTarget implements DropTargetListener {
    /** Drop target for the table. */
    protected DropTarget m_dropTarget;
    /** Drop target for the viewport. */
    protected DropTarget m_viewportDropTarget;
    /** The table. */
    protected JTable m_table;
    /** The bordered viewport. */
    protected BorderedViewport m_borderedViewport;

    /**
     * Attachs this manputlator to a table.
     *
     * @param table                             the table
     * @param borderedViewport                  the viewport being highlighted when the drag is over the table, but not over an element (may be <code>null</code>)
     */
    public JTableDropTarget(JTable table,BorderedViewport borderedViewport) {
        m_table=table;
        m_borderedViewport=borderedViewport;
        m_dropTarget=new SmartDropTarget(table,this);
        setBorderedViewport(m_borderedViewport);
    }
    /**
     * Sets the bordered viewport.
     *
     * @param borderedViewport                  the viewport being highlighted when the drag is over the table, but not over an element (may be <code>null</code>)
     */
    public void setBorderedViewport(BorderedViewport borderedViewport) {
        if (borderedViewport!=m_borderedViewport) {
            if (m_borderedViewport!=null) {
                m_borderedViewport.setBorderShown(false);
                m_viewportDropTarget.setComponent(null);
            }
            m_borderedViewport=borderedViewport;
            if (m_borderedViewport!=null)
                m_viewportDropTarget=new DropTarget(m_borderedViewport,this);
            else
                m_viewportDropTarget=null;
        }
    }
    /**
     * Called when a drag operation has encountered the DropTarget.
     *
     * @param dtde                              the event
     */
    public void dragEnter(DropTargetDragEvent dtde) {
        processDragEvent(dtde);
    }
    /**
     * The drag operation has departed the DropTarget without dropping.
     *
     * @param dte                               the event
     */
    public void dragExit(DropTargetEvent dte) {
        if (m_borderedViewport!=null)
            m_borderedViewport.setBorderShown(false);
    }
    /**
     * Called when a drag operation is ongoing on the DropTarget.
     *
     * @param dtde                              the event
     */
    public void dragOver(DropTargetDragEvent dtde) {
        processDragEvent(dtde);
    }
    /**
     * The drag operation has terminated with a drop on this DropTarget.
     *
     * @param dtde                              the event
     */
    public void drop(DropTargetDropEvent dtde) {
        if (m_borderedViewport!=null)
            m_borderedViewport.setBorderShown(false);
        if (processDrop(dtde))
            dtde.acceptDrop(dtde.getSourceActions());
        else
            dtde.rejectDrop();
    }
    /**
     * Called if the user has modified the current drop gesture.
     *
     * @param dtde                              the event
     */
    public void dropActionChanged(DropTargetDragEvent dtde) {
    }
    /**
     * Processes a drag event.
     *
     * @param dtde                              the event
     */
    protected void processDragEvent(DropTargetDragEvent dtde) {
        if (!acceptsDrag(dtde)) {
            if (m_borderedViewport!=null)
                m_borderedViewport.setBorderShown(false);
            dtde.rejectDrag();
        }
        else {
            if (m_borderedViewport!=null)
                m_borderedViewport.setBorderShown(true);
            dtde.acceptDrag(dtde.getSourceActions());
        }
    }
    /**
     * Determines whether operation can be accepted.
     *
     * @param dtde                              the event
     * @return                                  <code>true</code> if the drag will be accepted
     */
    protected boolean acceptsDrag(DropTargetDragEvent dtde) {
        return true;
    }
    /**
     * Processes the drop event.
     *
     * @param dtde                              the drop event
     * @return                                  <code>true</code> if drop was accepted
     */
    protected boolean processDrop(DropTargetDropEvent dtde) {
        return false;
    }
}
