package de.fzi.wim.guibase.configuration;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.OutputStream;
import java.io.FileOutputStream;
import java.util.Properties;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.NoSuchElementException;

/**
 * Represents a configuration of the application.
 */
public class Configuration {
    /** The file of the configuration. */
    protected File m_configurationFile;
    /** The properties backing this configuration. */
    protected Properties m_properties;
    /** Prefix path. */
    protected String m_prefix;

    /**
     * Creates a configuration object and reads it from the the supplied location.
     *
     * @param configurationFile                     the file for the configuration
     */
    public Configuration(File configurationFile) {
        m_configurationFile=configurationFile;
        m_properties=new Properties();
        m_prefix="";
    }
    /**
     * Creates an inner configuration object.
     *
     * @param configurationFile                     the file for the configuration
     * @param properties                            the properties object
     * @param prefix                                the prefix
     */
    protected Configuration(File configurationFile,Properties properties,String prefix) {
        m_configurationFile=configurationFile;
        m_properties=properties;
        m_prefix=prefix;
    }
    /**
     * Loads the configuration from its location.
     *
     * @throws IOException                          thrown if the configuration cannot be read
     */
    public void load() throws IOException {
        m_properties.clear();
        InputStream inputStream=new FileInputStream(m_configurationFile);
        try {
            m_properties.load(inputStream);
        }
        finally {
            inputStream.close();
        }
    }
    /**
     * Saves the configuration.
     *
     * @throws IOException                          thrown if the configuration cannot be read
     */
    public void save() throws IOException {
        File parentFile=m_configurationFile.getParentFile();
        if (parentFile!=null)
            parentFile.mkdirs();
        OutputStream outputStream=new FileOutputStream(m_configurationFile);
        try {
            m_properties.store(outputStream,null);
        }
        finally {
            outputStream.close();
        }
    }
    /**
     * Returns the string value of the key in the configuration.
     *
     * @param key                                   the key
     * @param defaultValue                          the default
     * @return                                      the value (<code>defaultValue</code> if key doesn't exist)
     */
    public String getString(String key,String defaultValue) {
        String result=(String)m_properties.get(m_prefix+key);
        if (result==null)
            return defaultValue;
        else
            return result;
    }
    /**
     * Returns the string value of the key in the configuration.
     *
     * @param key                                   the key
     * @return                                      the value (<code>null</code> if key doesn't exist)
     */
    public String getString(String key) {
        return (String)m_properties.get(m_prefix+key);
    }
    /**
     * Returns the integer value of the key in the configuration.
     *
     * @param key                                   the key
     * @param defaultValue                          the default value
     * @return                                      the value (the default value is returned if key doesn't exist of if the value is misformatted)
     */
    public int getInt(String key,int defaultValue) {
        String value=getString(key);
        if (value!=null)
            try {
                return Integer.parseInt(value);
            }
            catch (NumberFormatException ignored) {
            }
        return defaultValue;
    }
    /**
     * Returns the double value of the key in the configuration.
     *
     * @param key                                   the key
     * @param defaultValue                          the default value
     * @return                                      the value (the default value is returned if key doesn't exist of if the value is misformatted)
     */
    public double getDouble(String key,double defaultValue) {
        String value=getString(key);
        if (value!=null)
            try {
                return Double.parseDouble(value);
            }
            catch (NumberFormatException ignored) {
            }
        return defaultValue;
    }
    /**
     * Returns the boolean value of the key in the configuration.
     *
     * @param key                                   the key
     * @param defaultValue                          the default value
     * @return                                      the value (the default value is returned if key doesn't exist of if the value is misformatted)
     */
    public boolean getBoolean(String key,boolean defaultValue) {
        String value=getString(key);
        if (value!=null)
            return Boolean.valueOf(value).booleanValue();
        return defaultValue;
    }
    /**
     * Sets the string value of the key in the configuration.
     *
     * @param key                                   the key
     * @param value                                 the value (<code>null</code> if key doesn't exist)
     */
    public void setString(String key,String value) {
        m_properties.put(m_prefix+key,value);
    }
    /**
     * Sets the integer value of the key in the configuration.
     *
     * @param key                                   the key
     * @param value                                 the default value
     */
    public void setInt(String key,int value) {
        setString(key,String.valueOf(value));
    }
    /**
     * Sets the double value of the key in the configuration.
     *
     * @param key                                   the key
     * @param value                                 the default value
     */
    public void setDouble(String key,double value) {
        setString(key,String.valueOf(value));
    }
    /**
     * Sets the boolean value of the key in the configuration.
     *
     * @param key                                   the key
     * @param value                                 the default value
     */
    public void setBoolean(String key,boolean value) {
        setString(key,String.valueOf(value));
    }
    /**
     * Removes given key.
     *
     * @param key                                   the key
     */
    public void removeKey(String key) {
        m_properties.remove(m_prefix+key);
    }
    /**
     * Clears all keys from this configuration.
     */
    public void clear() {
        Iterator keys=m_properties.keySet().iterator();
        while (keys.hasNext()) {
            String key=(String)keys.next();
            if (key.startsWith(m_prefix))
                keys.remove();
        }
    }
    /**
     * Returns the names of the keys in this configuration.
     *
     * @return                                      the names of the keys
     */
    public Iterator getKeyNames() {
        return new KeyNamesIterator(m_prefix,m_properties.keys());
    }
    /**
     * Returns the subconfiguration with given prefix.
     *
     * @param prefix                                the prefix of the subconfiguration
     * @return                                      the subconfiguration for given prefix
     */
    public Configuration getSubConfiguration(String prefix) {
        return new Configuration(m_configurationFile,m_properties,m_prefix+prefix+".");
    }
    /**
     * Stores the map of string keys and values under given key.
     *
     * @param key                                   the key
     * @param map                                   the map of string keys and values
     */
    public void setMap(String key,Map map) {
        int index=0;
        Iterator mapKeys=map.keySet().iterator();
        while (mapKeys.hasNext()) {
            String mapKey=(String)mapKeys.next();
            String value=(String)map.get(mapKey);
            setString(key+"."+index+".key",mapKey);
            setString(key+"."+index+".value",value);
            index++;
        }
        setInt(key+".count",index);
    }
    /**
     * Loads the map of string keys and values under given key.
     *
     * @param key                                   the key
     * @return                                      the map of string keys and values
     */
    public Map getMap(String key) {
        int count=getInt(key+".count",-1);
        if (count==-1)
            return null;
        else {
            Map result=new HashMap();
            for (int index=0;index<count;index++) {
                String mapKey=getString(key+"."+index+".key");
                String value=getString(key+"."+index+".value");
                if (mapKey!=null && value!=null)
                    result.put(mapKey,value);
            }
            return result;
        }
    }

    /**
     * The iterator for key names.
     */
    protected class KeyNamesIterator implements Iterator {
        /** The prefix. */
        protected String m_prefix;
        /** The enumeration of the properties object. */
        protected Enumeration m_keys;
        /** The current key. */
        protected String m_key;

        public KeyNamesIterator(String prefix,Enumeration keys) {
            m_prefix=prefix;
            m_keys=keys;
            advance();
        }
        public boolean hasNext() {
            return m_key!=null;
        }
        public Object next() {
            if (m_key==null)
                throw new NoSuchElementException();
            String key=m_key;
            advance();
            return key;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
        protected void advance() {
            m_key=null;
            while (m_keys.hasMoreElements()) {
                String key=(String)m_keys.nextElement();
                if (key.startsWith(m_prefix)) {
                    m_key=key;
                    break;
                }
            }
        }
    }
}
