package de.fzi.wim.guibase.command;

import java.beans.PropertyChangeSupport;
import java.util.List;
import java.util.LinkedList;
import javax.swing.SwingUtilities;
import javax.swing.event.EventListenerList;

import de.fzi.wim.guibase.util.*;

/**
 * A manager for commands that handles undo-redo.
 */
public class CommandManager {
    /** The thread pool for executing asynchronous commands. */
    protected ThreadPool m_threadPool;
    /** Support class for property change events. */
    protected PropertyChangeSupport m_propertyChangeSupport;
    /** List of recently executed commands. */
    protected List m_executedCommands;
    /** List of undoed command. */
    protected List m_undoedCommands;
    /** Set to <code>true</code> if an asynchronous command is being executed. */
    protected boolean m_executingCommand;
    /** The list of listeners. */
    protected EventListenerList m_listenerList;

    /**
     * Creates an instance of this class.
     *
     * @param threadPool                the pool of threads for executing asynchronous commands
     */
    public CommandManager(ThreadPool threadPool) {
        m_threadPool=threadPool;
        m_propertyChangeSupport=new PropertyChangeSupport(this);
        m_executedCommands=new LinkedList();
        m_undoedCommands=new LinkedList();
        m_listenerList=new EventListenerList();
    }
    /**
     * Adds a command manager listener.
     *
     * @param listener                  the listener to add
     */
    public void addCommandManagerListener(CommandManagerListener listener) {
        m_listenerList.add(CommandManagerListener.class,listener);
    }
    /**
     * Removes a command manager listener.
     *
     * @param listener                  the listener to remove
     */
    public void removeCommandManagerListener(CommandManagerListener listener) {
        m_listenerList.remove(CommandManagerListener.class,listener);
    }
    /**
     * Executes given command (synchronously or asynchronously, depending on the command).
     *
     * @param command                   command to execute
     * @throws CommandException         thrown if the command cannot be executed
     */
    public void execute(Command command) throws CommandException {
        if (command.executeAsynchronously())
            executeAsynchronously(command);
        else
            executeSynchronously(command);
    }
    /**
     * Executes a command synchronously. If command is not undoable, the undo queue is deleted.
     *
     * @param command                   command to execute
     * @throws CommandException         thrown if the command cannot be executed
     */
    public void executeSynchronously(Command command) throws CommandException {
        assertNotExecutingCommand();
        command.execute();
        commandExecuted(command);
    }
    /**
     * Executes a command asynchronously. If command is not undoable, the undo queue is deleted.
     * If a command is currently being executed, this method waits until the command execution thread terminates.
     *
     * @param command                   command to execute
     * @throws CommandException         thrown if the command cannot be executed
     */
    public void executeAsynchronously(Command command) throws CommandException {
        assertNotExecutingCommand();
        startCommandCapsule(command,1);
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Called after supplied command has been executed. This method must be called on the main thread.
     *
     * @param command                   command that was executed
     */
    protected void commandExecuted(Command command) {
        if (command instanceof UndoableCommand)
            m_executedCommands.add(0,command);
        else
            m_executedCommands.clear();
        m_undoedCommands.clear();
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Undoes a recently executed (synchronously or asynchronously, depending on the command).
     *
     * @throws CommandException         thrown if the command cannot be undoed
     */
    public void undo() throws CommandException {
        UndoableCommand undoableCommand=(UndoableCommand)m_executedCommands.get(0);
        if (undoableCommand.executeAsynchronously())
            undoAsynchronously();
        else
            undoSynchronously();
    }
    /**
     * Undoes a recently executed command synchronously.
     *
     * @throws CommandException         thrown if the command cannot be undoed
     */
    public void undoSynchronously() throws CommandException {
        assertNotExecutingCommand();
        UndoableCommand undoableCommand=(UndoableCommand)m_executedCommands.get(0);
        undoableCommand.undo();
        commandUndone(undoableCommand);
    }
    /**
     * Undoes a recently executed command asynchronously.
     *
     * @throws CommandException         thrown if the command cannot be undoed
     */
    public void undoAsynchronously() throws CommandException {
        assertNotExecutingCommand();
        UndoableCommand undoableCommand=(UndoableCommand)m_executedCommands.get(0);
        startCommandCapsule(undoableCommand,2);
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Called after supplied command has been undone. This method must be called on the main thread.
     *
     * @param undoableCommand           command that was undone
     */
    protected void commandUndone(UndoableCommand undoableCommand) {
        m_executedCommands.remove(0);
        m_undoedCommands.add(0,undoableCommand);
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Redoes a recently executed (synchronously or asynchronously, depending on the command).
     *
     * @throws CommandException         thrown if the command cannot be redoed
     */
    public void redo() throws CommandException {
        UndoableCommand undoableCommand=(UndoableCommand)m_undoedCommands.get(0);
        if (undoableCommand.executeAsynchronously())
            redoAsynchronously();
        else
            redoSynchronously();
    }
    /**
     * Redoes a recently undoed command synchronously.
     *
     * @throws CommandException         thrown if the command cannot be redoed
     */
    public void redoSynchronously() throws CommandException {
        assertNotExecutingCommand();
        UndoableCommand undoableCommand=(UndoableCommand)m_undoedCommands.get(0);
        undoableCommand.execute();
        commandRedoed(undoableCommand);
    }
    /**
     * Redoes a recently undoed command asynchronously.
     *
     * @throws CommandException         thrown if the command cannot be redoed
     */
    public void redoAsynchronously() throws CommandException {
        assertNotExecutingCommand();
        UndoableCommand undoableCommand=(UndoableCommand)m_undoedCommands.get(0);
        startCommandCapsule(undoableCommand,3);
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Called after supplied command has been redoed. This method must be called on the main thread.
     *
     * @param undoableCommand           command that was redoed
     */
    protected void commandRedoed(UndoableCommand undoableCommand) {
        m_undoedCommands.remove(0);
        m_executedCommands.add(0,undoableCommand);
        notifyUndoRedoCapabilityChanged();
    }
    /**
     * Returns <code>true</code> if a command is currently being executed.
     *
     * @return                          <code>true</code> if a command is currently being executed
     */
    public boolean isCommandBeingExecuted() {
        return m_executingCommand;
    }
    /**
     * Checks whether undo can  be performed.
     *
     * @return                          <code>true</code> if undo can be performed
     */
    public boolean getCanUndo() {
        return !m_executedCommands.isEmpty() && !isCommandBeingExecuted();
    }
    /**
     * Checks whether redo can  be performed.
     *
     * @return                          <code>true</code> if redo can be performed
     */
    public boolean getCanRedo() {
        return !m_undoedCommands.isEmpty() && !isCommandBeingExecuted();
    }
    /**
     * Called on the main thread to notify listeres that command execution will start.
     *
     * @param command                   the command
     * @param executeType               the type of execution
     */
    protected void notifyCommandWillExecute(Command command,int executeType) {
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==CommandManagerListener.class)
                ((CommandManagerListener)listeners[i+1]).commandWillExecute(this,command,executeType);
    }
    /**
     * Called on the main thread to notify listeres that command has been executed.
     *
     * @param command                   the command
     * @param executeType               the type of execution
     */
    protected void notifyCommandExecuted(Command command,int executeType) {
        m_executingCommand=false;
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==CommandManagerListener.class)
                ((CommandManagerListener)listeners[i+1]).commandExecuted(this,command,executeType);
    }
    /**
     * Called on the main thread when asynchronous command invocation throws an error.
     *
     * @param command                   the command
     * @param commandException          the exection that the command threw
     */
    protected void notifyCommandException(Command command,CommandException commandException) {
        m_executingCommand=false;
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==CommandManagerListener.class)
                ((CommandManagerListener)listeners[i+1]).commandException(this,command,commandException);
    }
    /**
     * Called when the undo and redo options may have changed.
     */
    protected void notifyUndoRedoCapabilityChanged() {
        boolean canUndo=getCanUndo();
        boolean canRedo=getCanRedo();
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==CommandManagerListener.class)
                ((CommandManagerListener)listeners[i+1]).undoRedoAvailabilityChanged(this,canUndo,canRedo);
    }
    /**
     * Throws an exception if a command is currently in progress.
     *
     * @throws CommandException         thrown if command is currently in progress
     */
    protected void assertNotExecutingCommand() throws CommandException {
        if (m_executingCommand)
            throw new CommandException("An asynchronous command is currently being executed.");
    }
    /**
     * Executes a command on the thread pool.
     *
     * @param command                   the command
     * @param executeType               the type of execution
     * @throws CommandException         thrown if there is an error
     */
    protected void startCommandCapsule(Command command,int executeType) throws CommandException {
        try {
            m_executingCommand=true;
            m_threadPool.executeTask(new CommandCapsule(command,executeType));
        }
        catch (InterruptedException e) {
            m_executingCommand=false;
            throw new CommandException("Calling thread interrupted.",e);
        }
    }

    /**
     * The capsule for executing a command.
     */
    protected class CommandCapsule implements Runnable {
        /** The current command being dispatched. */
        protected Command m_command;
        /** Determines what kind of execution this is. */
        protected int m_executeType;

        public CommandCapsule(Command command,int executeType) {
            m_command=command;
            m_executeType=executeType;
        }
        public void run() {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    notifyCommandWillExecute(m_command,m_executeType);
                }
            });
            try {
                switch (m_executeType) {
                case 1:
                case 3:
                    m_command.execute();
                    break;
                case 2:
                    ((UndoableCommand)m_command).undo();
                    break;
                }
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        notifyCommandExecuted(m_command,m_executeType);
                        switch (m_executeType) {
                        case 1:
                            commandExecuted(m_command);
                            break;
                        case 2:
                            commandUndone((UndoableCommand)m_command);
                            break;
                        case 3:
                            commandRedoed((UndoableCommand)m_command);
                            break;
                        }
                    }
                });
            }
            catch (final CommandException error) {
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        notifyCommandException(m_command,error);
                    }
                });
            }
        }
    }
}
