package de.fzi.wim.guibase.appdriver;

import javax.swing.JComponent;

import de.fzi.wim.guibase.configuration.Configuration;
import de.fzi.wim.guibase.actions.SmartActionMap;

/**
 * Interface implemented by all components that can be shown in a separate view in the main window.
 */
public interface Viewable {
    /**
     * Retruns the name of this viewable.
     *
     * @return                              the name of this viewable
     */
    String getViewableName();
    /**
     * Returns the module to which this object belongs to.
     *
     * @return                              the module of this object
     */
    Module getModule();
    /**
     * Returns the smart action map that should be used when this viewable is active.
     *
     * @return                              the smart action map of this viewable
     */
    SmartActionMap getSmartActionMap();
    /**
     * Attaches this object to an anchor.
     *
     * @param anchor                        anchor to which this object  needs to be attached
     */
    void setViewableAnchor(ViewableAnchor anchor);
    /**
     * Returns the current anchor of this viewable.
     *
     * @return                              current anchor of this viewable
     */
    ViewableAnchor getViewableAnchor();
    /**
     * Called when anchor is closing.
     */
    void closing();
    /**
     * Called when view will be activated. Should return <code>true</code> if view should be activated.
     *
     * @return                              <code>true</code> if view should be activated
     */
    boolean activating();
    /**
     * Called when view is activated.
     */
    void activated();
    /**
     * Called when view will be deactivated. Should return <code>true</code> if view should be deactivated.
     *
     * @return                              <code>true</code> if view should be deactivated
     */
    boolean deactivating();
    /**
     * Called when view is deactivated.
     */
    void deactivated();
    /**
     * Returns the component of this viewable.
     *
     * @return                                  component of this viewable
     */
    JComponent getComponent();
    /**
     * Disposes of this viewable.
     */
    void dispose();
    /**
     * Checks whether the application can terminate.
     *
     * @return                              <code>true</code> if application can terminate
     */
    boolean canExitApplication();
    /**
     * Notifies the module that the application is exiting.
     */
    void applicationExiting();
    /**
     * Called to save the UI state of the viewable.
     *
     * @param configuration                 the configuration
     * @return                              <code>true</code> if the configuration saved
     */
    boolean saveUIState(Configuration configuration);
    /**
     * Undoes an action.
     */
    void undo();
    /**
     * Returns <code>true</code> if undo can be performed.
     *
     * @return                              <code>true</code> if undo can be performed
     */
    boolean canUndo();
    /**
     * Redoes an action.
     */
    void redo();
    /**
     * Returns <code>true</code> if redo can be performed.
     *
     * @return                              <code>true</code> if redo can be performed
     */
    boolean canRedo();
}
