package de.fzi.wim.guibase.appdriver;

import java.beans.PropertyChangeEvent;
import java.beans.VetoableChangeListener;
import java.beans.PropertyVetoException;
import java.util.List;
import java.util.LinkedList;
import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Component;
import java.awt.Toolkit;
import javax.swing.JDesktopPane;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JInternalFrame;
import javax.swing.JToolBar;
import javax.swing.JRootPane;
import javax.swing.BorderFactory;
import javax.swing.border.BevelBorder;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import de.fzi.wim.guibase.menus.*;
import de.fzi.wim.guibase.configuration.*;

/**
 * Main frame for application with many floating windows.
 */
public class MultiMainFrame extends JFrame implements LookAndFeelUpdateListener.UIUpdateNotificationTarget {
    /** Selected viewable property. */
    public static final String SELECTED_VIEWABLE_PROPERTY="selectedViewable";
    /** Specifies how much to push each new viewable anchor to the right. */
    protected static final int ANCHOR_FRAME_OFFSET=15;

    /** Application driver of this frame. */
    protected AppDriver m_appDriver;
    /** Desktop pane of main frame to allow MDI look&feel. */
    protected JDesktopPane m_desktopPane;
    /** Position of the last viewable anchor. */
    protected Point m_lastViewableAnchorPosition;
    /** Listener of L&F notification events. */
    protected LookAndFeelUpdateListener m_lfUpdateListener;
    /** Label of the status bar. */
    protected JLabel m_statusBar;
    /** Menu help tracked used to display help in the status bar. */
    protected MenuHelpTracker m_menuHelpTracker;

    /**
     * Creates an instance of this class.
     *
     * @param appDriver                         application driver of the frame
     */
    public MultiMainFrame(AppDriver appDriver) {
        super("<name not set>");
        m_appDriver=appDriver;
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        m_lfUpdateListener=new LookAndFeelUpdateListener(this);
        m_statusBar=new JLabel(m_appDriver.getLocalizationManager().getPhrase("STATUS_READY"),JLabel.LEFT);
        m_statusBar.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED),BorderFactory.createEmptyBorder(2,2,2,2)));
        m_menuHelpTracker=new MenuHelpTracker(m_statusBar);
        m_desktopPane=new JDesktopPane();
        JPanel mainPane=new JPanel(new BorderLayout());
        mainPane.add(m_desktopPane,BorderLayout.CENTER);
        mainPane.add(m_statusBar,BorderLayout.SOUTH);
        setContentPane(mainPane);
        pack();
        loadDefaultSizes();
    }
    /**
     * Loads the default sizes of the window from the configuration.
     */
    protected void loadDefaultSizes() {
        Configuration configuration=m_appDriver.getConfiguration().getSubConfiguration("multiMainFrame");
        int x=configuration.getInt("x",0);
        int y=configuration.getInt("y",0);
        int width=configuration.getInt("width",10000000);
        int height=configuration.getInt("height",10000000);
        int extendedState=configuration.getInt("extendedState",JFrame.NORMAL);
        Dimension screenSize=Toolkit.getDefaultToolkit().getScreenSize();
        if (x<10 || x>screenSize.width)
            x=0;
        if (y<10 || y>screenSize.height)
            y=0;
        if (width<0 || x+width>screenSize.width)
            width=screenSize.width-x;
        if (height<0 || y+height>screenSize.height)
            height=screenSize.height-y;
        setLocation(x,y);
        setSize(width,height);
        setExtendedState(extendedState);
    }
    /**
     * Called when UI of the window is updated.
     */
    public void notifyUpdateUI() {
        validate();
    }
    /**
     * Creates new window and displays supplied viewable pane in it.
     *
     * @param viewable                          viewable to be displayed
     * @return                                  viewable anchor of the new window
     */
    public ViewableAnchor newViewableAncor(Viewable viewable) {
        ViewableFrame frame=new ViewableFrame(viewable);
        m_desktopPane.add(frame);
        frame.pack();
        Dimension availableSize=m_desktopPane.getSize();
        frame.setSize(2*availableSize.width/3,2*availableSize.height/3);
        if (m_lastViewableAnchorPosition!=null) {
            if (m_lastViewableAnchorPosition.x+ANCHOR_FRAME_OFFSET+frame.getWidth()<availableSize.width && m_lastViewableAnchorPosition.y+ANCHOR_FRAME_OFFSET+frame.getWidth()<availableSize.width)
                frame.setLocation(m_lastViewableAnchorPosition.x+ANCHOR_FRAME_OFFSET,m_lastViewableAnchorPosition.y+ANCHOR_FRAME_OFFSET);
        }
        m_lastViewableAnchorPosition=frame.getLocation();
        return frame;
    }
    /**
     * Called to dispose of this window.
     */
    public void dispose() {
        m_lfUpdateListener.dispose();
        m_menuHelpTracker.trackMenuBar(null);
        super.dispose();
    }
    /**
     * Sets the bar into this frame.
     *
     * @param menuBar                           menu bar that this frame should use
     */
    public void setJMenuBar(JMenuBar menuBar) {
        m_menuHelpTracker.trackMenuBar(menuBar);
        super.setJMenuBar(menuBar);
        updateMenuBarVisibility();
    }
    /**
     * Sets the toolbar into this frame.
     *
     * @param toolBar                           the toolbar to be set
     */
    public void setToolBar(JToolBar toolBar) {
        JPanel panel=(JPanel)getContentPane();
        panel.add(toolBar,BorderLayout.NORTH);
        updateToolBarVisibility();
    }
    /**
     * Returns the currently selected viewable.
     *
     * @return                                  currently selected viewable
     */
    public Viewable getSelectedViewable() {
        ViewableFrame viewableFrame=(ViewableFrame)m_desktopPane.getSelectedFrame();
        if (viewableFrame==null)
            return null;
        else
            return viewableFrame.getViewable();
    }
    /**
     * Returns the current list of viewables.
     *
     * @return                                  the current list of viewables
     */
    public List getViewables() {
        JInternalFrame[] frames=m_desktopPane.getAllFrames();
        List result=new LinkedList();
        for (int i=0;i<frames.length;i++) {
            ViewableFrame viewableFrame=(ViewableFrame)frames[i];
            result.add(viewableFrame.getViewable());
        }
        return result;
    }
    /**
     * Fires a property change event for the selectedViewable property.
     *
     * @param oldValue                          old selected viewable
     * @param newValue                          new selected viewable
     */
    protected void fireSelectedViewableChageEvent(Viewable oldValue,Viewable newValue) {
        firePropertyChange(SELECTED_VIEWABLE_PROPERTY,oldValue,newValue);
        updateToolBarVisibility();
        updateMenuBarVisibility();
    }
    /**
     * Updates the visible parts of the toolbars.
     */
    protected void updateToolBarVisibility() {
        JPanel panel=(JPanel)getContentPane();
        int components=panel.getComponentCount();
        String currentViewable;
        if (getSelectedViewable()!=null)
            currentViewable=getSelectedViewable().getViewableName();
        else
            currentViewable=null;
        for (int i=0;i<components;i++) {
            Component component=panel.getComponent(i);
            if (component instanceof JToolBar)
                AppDriver.updateVisibleComponents(component,currentViewable);
        }
    }
    /**
     * Updates the visible parts of the menu bar.
     */
    protected void updateMenuBarVisibility() {
        JMenuBar menuBar=getJMenuBar();
        if (menuBar!=null) {
            String currentViewable;
            if (getSelectedViewable()!=null)
                currentViewable=getSelectedViewable().getViewableName();
            else
                currentViewable=null;
            AppDriver.updateVisibleComponents(menuBar,currentViewable);
        }
    }
    /**
     * Called when the application is exiting.
     */
    public void applicationExiting() {
        Configuration configuration=m_appDriver.getConfiguration().getSubConfiguration("multiMainFrame");
        configuration.setInt("x",getX());
        configuration.setInt("y",getY());
        configuration.setInt("width",getWidth());
        configuration.setInt("height",getHeight());
        configuration.setInt("extendedState",getExtendedState());
    }

    /**
     * Internal frame window for displaying viewables.
     */
    public class ViewableFrame extends JInternalFrame implements ViewableAnchor {
        /** Display pane that this frame manages. */
        protected Viewable m_viewable;

        /**
         * Creates an internal frame display window.
         *
         * @param viewable                      viewable for this window
         */
        protected ViewableFrame(Viewable viewable) {
            super("",true,true,true,true);
            setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
            m_viewable=viewable;
            m_viewable.setViewableAnchor(this);
            addInternalFrameListener(new InternalFrameAdapter() {
                public void internalFrameActivated(InternalFrameEvent e) {
                    m_viewable.activated();
                    fireSelectedViewableChageEvent(null,m_viewable);
                }
                public void internalFrameDeactivated(InternalFrameEvent e) {
                    m_viewable.deactivated();
                    fireSelectedViewableChageEvent(m_viewable,null);
                }
                public void internalFrameClosing(InternalFrameEvent e) {
                    m_viewable.closing();
                }
            });
            addVetoableChangeListener(new VetoableChangeListener() {
                public void vetoableChange(PropertyChangeEvent evt) throws PropertyVetoException {
                    if (evt.getPropertyName()==IS_SELECTED_PROPERTY && ((Boolean)evt.getNewValue()).booleanValue())
                        if (!m_viewable.activating())
                            throw new PropertyVetoException("Selection of viewable was vetoed.",evt);
                }
            });
            setContentPane(m_viewable.getComponent());
            pack();
        }
        /**
         * Disposes of this frame.
         */
        public void dispose() {
            if (m_viewable!=null) {
                int index=m_desktopPane.getIndexOf(this);
                // if there are more viewables, then select one of them
                int scanIndex=index+1;
                while (scanIndex!=index) {
                    if (scanIndex>=m_desktopPane.getComponentCount()) {
                        if (index==-1)
                            break;
                        scanIndex=0;
                    }
                    Component component=m_desktopPane.getComponent(scanIndex);
                    if (component instanceof ViewableFrame) {
                        ((ViewableFrame)component).activate();
                        break;
                    }
                    scanIndex++;
                }
                super.dispose();
                m_viewable.dispose();
                m_appDriver.viewableDisposed(m_viewable);
                removeAll();
                m_viewable=null;
                rootPane=null;
                setRootPane(new JRootPane());   // this is used to avoid an obscure memory leak in Java - the JRootPane remains referenced by the keyboard manager etc.
            }
        }
        /**
         * Activates this frame.
         */
        public void activate() {
            try {
                setSelected(true);
            }
            catch (PropertyVetoException ignored) {
            }
        }
        /**
         * Returns the viewable.
         *
         * @return                              the viewable of this anchor
         */
        public Viewable getViewable() {
            return m_viewable;
        }
    }
}
