package de.fzi.wim.guibase.appdriver;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.awt.Component;
import java.awt.Container;
import javax.swing.JComponent;
import javax.swing.UIManager;
import javax.swing.JMenu;

/**
 * Listens for notifications that look and feel was updated and updates the UI of components in the tree rooted at the component
 * passed in the constructor.
 */
public class LookAndFeelUpdateListener implements PropertyChangeListener {
    /** Root of the tree to notify about UI update. */
    protected Component m_root;

    /**
     * Creates a listener and attaches it to given root.
     *
     * @param root                      the root of the tree
     */
    public LookAndFeelUpdateListener(Component root) {
        m_root=root;
        UIManager.addPropertyChangeListener(this);
    }
    /**
     * Called when a property of <code>UIManager</code> changes.
     *
     * @param event                     notification about the type of change
     */
    public void propertyChange(PropertyChangeEvent event) {
        if ("lookAndFeel".equals(event.getPropertyName()))
            uiUpdated();
    }
    /**
     * Destroys this listener object. This method must be called when object is no longer needed.
     */
    public void dispose() {
        UIManager.removePropertyChangeListener(this);
    }
    /**
     * Called when UI is updated.
     */
    public void uiUpdated() {
        updateComponentUI(m_root);
    }
    /**
     * Updates the UI of the supplied component tree.
     *
     * @param root                      root of the tree to update
     */
    public static void updateComponentUI(Component root) {
        if (root instanceof JMenu)
            updateComponentUI(((JMenu)root).getPopupMenu());
        if (root instanceof JComponent)
            ((JComponent)root).updateUI();
        if (root instanceof UIUpdateNotificationTarget)
            ((UIUpdateNotificationTarget)root).notifyUpdateUI();
        if (root instanceof Container) {
            Container container=(Container)root;
            int numberOfChildren=container.getComponentCount();
            for (int i=0;i<numberOfChildren;i++)
                updateComponentUI(container.getComponent(i));
        }
    }

    /**
     * Components implementing this interface will receive notifications about look&feel changes.
     */
    public static interface UIUpdateNotificationTarget {
        /**
         * Called when UI was updated.
         */
        void notifyUpdateUI();
    }
}
