package de.fzi.wim.guibase.appdriver;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Locale;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.io.IOException;
import java.io.InputStream;
import java.io.File;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Window;
import java.awt.event.ActionEvent;
import javax.swing.RootPaneContainer;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JMenu;
import javax.swing.JFrame;
import javax.swing.JComponent;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;
import org.w3c.dom.NodeList;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.toolbars.*;
import de.fzi.wim.guibase.menus.*;
import de.fzi.wim.guibase.actions.*;
import de.fzi.wim.guibase.configuration.*;
import de.fzi.wim.guibase.classfactory.*;
import de.fzi.wim.guibase.util.*;

/**
 * This is the main class that start the application.
 */
public abstract class AppDriver implements SmartActionManager {
    /** Selected viewable property. */
    public static final String SELECTED_VIEWABLE_PROPERTY="selectedViewable";
    /** Viewable set property. */
    public static final String VIEWABLES_PROPERTY="viewables";
    /** Available look & feels. */
    protected static final String[][] s_lookAndFeels;
    static {
        UIManager.LookAndFeelInfo[] lookAndFeels=UIManager.getInstalledLookAndFeels();
        s_lookAndFeels=new String[lookAndFeels.length][2];
        for (int i=0;i<lookAndFeels.length;i++) {
            s_lookAndFeels[i][0]=lookAndFeels[i].getClassName();
            s_lookAndFeels[i][1]=lookAndFeels[i].getName();
        }
    }

    /** Property change support for this class. */
    protected PropertyChangeSupport m_propertyChangeSupport;
    /** The thread pool. */
    protected ThreadPool m_threadPool;
    /** Localization manager to use for the application. */
    protected LocalizationManager m_localizationManager;
    /** Map of all actions in the application. */
    protected SmartActionMap m_actions;
    /** Map of installed modules. */
    protected Map m_modules;
    /** The document for the GUI configuration. */
    protected Document m_guiConfiguration;
    /** The class factory. */
    protected ClassFactory m_classFactory;
    /** Currently selected viewable. */
    protected Viewable m_selectedViewable;
    /** The configuration. */
    protected Configuration m_configuration;
    /** The file in which the last workspace was saved. */
    protected File m_lastWorkspaceFile;

    /**
     * Creates an instance of this class.
     */
    public AppDriver() {
    }
    /**
     * Initializes the application.
     *
     * @param guiConfigurationResource          the resource for the GUI configuration
     * @param locale                            locale for the application
     * @param configurationName                 the name of the configuration
     */
    public void initialize(final String guiConfigurationResource,final Locale locale,final String configurationName) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                initializeInternal(guiConfigurationResource,locale,configurationName);
            }
        });
    }
    /**
     * Initializes the application.
     *
     * @param guiConfigurationResource          the resource for the GUI configuration
     * @param locale                            locale for the application
     * @param configurationName                 the name of the configuration
     */
    protected void initializeInternal(String guiConfigurationResource,Locale locale,String configurationName) {
        // some basic initialization
        m_propertyChangeSupport=new PropertyChangeSupport(this);
        m_modules=new HashMap();
        m_threadPool=new ThreadPool(1,10);
        // load the configuration
        File userHome=new File(System.getProperty("user.home",""));
        File configurationFile=new File(userHome,configurationName);
        m_configuration=new Configuration(configurationFile);
        try {
            m_configuration.load();
        }
        catch (IOException ignored) {
        }
        restoreConfiguration();
        // parse the configuration resource
        try {
            setGUIConfigurationResource(guiConfigurationResource);
        }
        catch (SAXException e) {
            e.printStackTrace();
            return;
        }
        // create the class factory
        m_classFactory=new ClassFactory(m_guiConfiguration);
        // load the modules
        Module[] modules=loadModules();
        // load localization manager and setup the locale
        LocalizationManagerLoader loader=new LocalizationManagerLoader();
        try {
            loader.loadLocalizationManagers("de/fzi/wim/guibase/res/guibase.xml");
            for (int i=0;i<modules.length;i++)
                if (modules[i].getLocalizationResourceName()!=null)
                    loader.loadLocalizationManagers(modules[i].getLocalizationResourceName());
            NodeList nodeList=m_guiConfiguration.getElementsByTagName("localization");
            for (int i=0;i<nodeList.getLength();i++) {
                Element localizationNode=(Element)nodeList.item(i);
                String resourceName=localizationNode.getAttribute("href");
                loader.loadLocalizationManagers(resourceName);
            }
        }
        catch (SAXException e) {
            IllegalArgumentException error=new IllegalArgumentException("Cannot parse localization resource.");
            error.initCause(e);
            throw error;
        }
        // select the desired localization manager
        m_localizationManager=loader.getLocalizationManager(locale);
        if (m_localizationManager==null)
            throw new IllegalArgumentException("Locale "+locale+" cannot be loaded from the localization manager.");
        Locale.setDefault(m_localizationManager.getLocale());
        JOptionPane.setDefaultLocale(m_localizationManager.getLocale());
        // create the action map of this module
        m_actions=new SmartActionHashMap(null);
        addAction(new Exit());
        addAction(new Undo());
        addAction(new Redo());
        addAction(new SaveWorkspace());
        addAction(new LoadWorkspace());
        addAction(new SetLookAndFeel());
        // add all modules to the driver
        for (int i=0;i<modules.length;i++)
            addModule(modules[i]);
        // create the main window
        createMainFrameWindow();
        JOptionPane.setRootFrame(getMainFrameWindow());
        // load the menu bar and the toolbar
        Element application=getApplicationNode();
        String menuBarID=application.getAttribute("menuBarID");
        if (menuBarID.length()!=0)
            loadMenuBar(menuBarID);
        String toolBarID=application.getAttribute("toolBarID");
        if (toolBarID.length()!=0)
            loadToolBar(toolBarID);
        String mainFrameID=application.getAttribute("mainFrameID");
        showMainWindow(mainFrameID);
    }
    /**
     * Returns the thread pool.
     *
     * @return                          the thread pool
     */
    public ThreadPool getThreadPool() {
        return m_threadPool;
    }
    /**
     * Returns the configuration of the application.
     *
     * @return                          the configuration
     */
    public Configuration getConfiguration() {
        return m_configuration;
    }
    /**
     * Saves the configuration.
     */
    public void saveConfiguration() {
        try {
            m_configuration.save();
        }
        catch (IOException error) {
            displayErrorNotification(error);
        }
    }
    /**
     * Returns the application node.
     *
     * @return                          the application node
     */
    protected Element getApplicationNode() {
        NodeList nodeList=m_guiConfiguration.getElementsByTagName("application");
        if (nodeList.getLength()!=1)
            throw new IllegalArgumentException("GUI configuration must have exactly one 'application' node.");
        return (Element)nodeList.item(0);
    }
    /**
     * Adds a property change listener.
     *
     * @param listener                  the listener to add
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        m_propertyChangeSupport.addPropertyChangeListener(listener);
    }
    /**
     * Removes a property change listener.
     *
     * @param listener                  the listener to remove
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        m_propertyChangeSupport.removePropertyChangeListener(listener);
    }
    /**
     * Returns the current localization manager.
     *
     * @return                                  current localization manager
     */
    public LocalizationManager getLocalizationManager() {
        return m_localizationManager;
    }
    /**
     * Returns the class factory.
     *
     * @return                                  the class factory
     */
    public ClassFactory getClassFactory() {
        return m_classFactory;
    }
    /**
     * Shows the main window.
     *
     * @param titleID                           ID of the title of the main frame
     */
    public void showMainWindow(String titleID) {
        JFrame frame=getMainFrameWindow();
        frame.setTitle(m_localizationManager.getPhrase(titleID));
        frame.setVisible(true);
    }
    /**
     * Returns the GUI configuration of this object.
     *
     * @return                                  the GUI configuration
     */
    public Document getGUIConfiguration() {
        return m_guiConfiguration;
    }
    /**
     * Returns the element node of the configuration with given element and attribute values.
     *
     * @param elementName                       the name of the element
     * @param attributeName                     the name of the attribute
     * @param attributeValue                    the value of the attribute
     * @return                                  configuration element or <code>null</code> if no element matches
     */
    public Element getConfigurationNode(String elementName,String attributeName,String attributeValue) {
        NodeList elementList=m_guiConfiguration.getElementsByTagName(elementName);
        for (int i=0;i<elementList.getLength();i++) {
            Element element=(Element)elementList.item(i);
            if (attributeValue.equals(element.getAttribute(attributeName)))
                return element;
        }
        return null;
    }
    /**
     * Sets the document for the GUI configuration.
     *
     * @param guiConfigurationResource          the resource for the GUI configuration
     * @throws SAXException                     thrown if GUI configuration cannot be parsed
     */
    protected void setGUIConfigurationResource(String guiConfigurationResource) throws SAXException {
        try {
            DocumentBuilder builder=DocumentBuilderFactory.newInstance().newDocumentBuilder();
            InputStream inputStream=AppDriver.class.getClassLoader().getResourceAsStream(guiConfigurationResource);
            if (inputStream==null)
    			throw new SAXException("Configuration resource '"+guiConfigurationResource+"' could not be found on the classpath.");
            try {
                m_guiConfiguration=builder.parse(new InputSource(inputStream));
            }
            finally {
                inputStream.close();
            }
        }
        catch (ParserConfigurationException e) {
            throw new SAXException("Parser configuration problem",e);
        }
        catch (IOException e) {
            throw new SAXException("I/O problem",e);
        }
    }
    /**
     * Loads a menu bar from given resource.
     *
     * @param menuBarID                         the name of the menu bar key
     */
    public void loadMenuBar(String menuBarID) {
        Element element=getConfigurationNode("menubar","key",menuBarID);
        LocalizedMenuBuilder menuBuilder=new LocalizedMenuBuilder(m_localizationManager,this,m_actions);
        getMainFrameWindow().setJMenuBar(menuBuilder.createMenuBar(element));
        m_actions.updateActions();
    }
    /**
     * Loads a toolbar from given resource.
     *
     * @param toolBarID                         the name of the node that contains the toolbar
     */
    public void loadToolBar(String toolBarID) {
        Element element=getConfigurationNode("toolbar","key",toolBarID);
        LocalizedToolBarBuilder toolBarBuilder=new LocalizedToolBarBuilder(m_localizationManager,this,m_actions);
        setMainFrameToolBar(toolBarBuilder.createToolBar(JToolBar.HORIZONTAL,element));
        m_actions.updateActions();
    }
    /**
     * Creates a popup menu from the GUI specification with given key.
     *
     * @param popupMenuID                       the ID of the popup menu in the specification
     * @return                                  popup menu
     */
    public JPopupMenu createPopupMenu(String popupMenuID) {
        Element element=getConfigurationNode("popupmenu","key",popupMenuID);
        LocalizedMenuBuilder menuBuilder=new LocalizedMenuBuilder(m_localizationManager,this,m_actions);
        return menuBuilder.createPopupMenu(element);
    }
    /**
     * Initializes all loaded modules.
     *
     * @param module                            the module being added
     */
    public void addModule(Module module) {
        module.setAppDriver(this);
        m_modules.put(module.getModuleName(),module);
        m_actions.putAll(module.getSmartActionMap());
    }
    /**
     * Returns the module with given name.
     *
     * @param moduleName                        the name of the module
     * @return                                  the module with given name (or <code>null</code> if the module isn't registerd)
     */
    public Module getModule(String moduleName) {
        return (Module)m_modules.get(moduleName);
    }
   /**
     * Returns an iterator of registered module names (as <code>String</code> objects).
     *
     * @return                                  an iterator of registered module names
     */
    public Iterator getModuleNames() {
        return m_modules.keySet().iterator();
    }
    /**
     * Adds an action to the action map.
     *
     * @param action                            action to add
     */
    protected void addAction(SmartAction action) {
        m_actions.put(action.getActionID(),action);
    }
    /**
     * Returns the actions of this object.
     *
     * @return                                  the actions of this object
     */
    public SmartActionMap getSmartActionMap() {
        return m_actions;
    }
    /**
     * Exits the current application.
     */
    public void exitApplication() {
        Iterator iterator=m_modules.values().iterator();
        while (iterator.hasNext()) {
            Module module=(Module)iterator.next();
            if (!module.canExitApplication())
                return;
        }
        iterator=getViewables().iterator();
        while (iterator.hasNext()) {
            Viewable viewable=(Viewable)iterator.next();
            if (!viewable.canExitApplication())
                return;
        }
        if (!canExitApplication())
            return;
        m_threadPool.interrupt();
        iterator=m_modules.values().iterator();
        while (iterator.hasNext()) {
            Module module=(Module)iterator.next();
            module.applicationExiting();
        }
        iterator=getViewables().iterator();
        while (iterator.hasNext()) {
            Viewable viewable=(Viewable)iterator.next();
            viewable.applicationExiting();
        }
        applicationExiting();
        saveConfiguration();
        System.exit(0);
    }
    /**
     * Called to check whether application can exit.
     *
     * @return                                  <code>true</code> if the application can exit
     */
    protected boolean canExitApplication() {
        if (m_lastWorkspaceFile!=null) {
            String message=m_localizationManager.getPhrase("guibase.saveWorkspace");
            String title=m_localizationManager.getPhrase("guibase.title");
            int result=JOptionPane.showConfirmDialog(getMainFrameWindow(),message,title,JOptionPane.YES_NO_CANCEL_OPTION);
            if (result==JOptionPane.YES_OPTION)
                return saveWorkspace(m_lastWorkspaceFile);
            else if (result==JOptionPane.CANCEL_OPTION)
                return false;
        }
        return true;
    }
    /**
     * Called when the application is exiting.
     */
    protected void applicationExiting() {
    }
    /**
     * Called to restore the configuration.
     */
    protected void restoreConfiguration() {
        String lookAndFeelClassName=m_configuration.getString("lookAndFeel");
        if (lookAndFeelClassName!=null)
            try {
                UIManager.setLookAndFeel(lookAndFeelClassName);
            }
            catch (Exception error) {
                displayErrorNotification(error);
            }
    }
    /**
     * Displays a notification about given error.
     *
     * @param error                             error
     */
    public void displayErrorNotification(Throwable error) {
        error.printStackTrace();
        String errorText=error.getMessage();
        error=error.getCause();
        while (error!=null) {
            errorText+="\n"+error.getMessage();
            error=error.getCause();
        }
        JOptionPane.showMessageDialog(getMainFrameWindow(),errorText,m_localizationManager.getPhrase("error_notification_title"),JOptionPane.ERROR_MESSAGE);
    }
    /**
     * Displays a notification with given ID and with the title equals to the name of the current viewable.
     *
     * @param notificationID                    the ID of the notification
     */
    public void displayNotification(String notificationID) {
        if (getSelectedViewable()!=null)
            displayNotification(notificationID,getSelectedViewable().getViewableName()+".title");
        else
            displayNotification(notificationID,"error_notification_title");
    }
    /**
     * Displays a notification with given ID from the localization manager.
     *
     * @param notificationID                    the ID of the notification
     * @param titleID                           the ID of the title
     */
    public void displayNotification(String notificationID,String titleID) {
        JOptionPane.showMessageDialog(getMainFrameWindow(),m_localizationManager.getPhrase(notificationID),m_localizationManager.getPhrase(titleID),JOptionPane.ERROR_MESSAGE);
    }
    /**
     * Returns the current action map.
     *
     * @return                              current action map
     */
    public SmartActionMap getCurrentAggregateActionMap() {
        if (getSelectedViewable()==null)
            return null;
        else
            return getSelectedViewable().getSmartActionMap();
    }
    /**
     * Returns the name of the currently selected context.
     *
     * @return                              the name of the currently selected context
     */
    public String getCurrentContextName() {
        if (getSelectedViewable()==null)
            return null;
        else
            return getSelectedViewable().getViewableName();
    }
    /**
     * Loads all modules from the configuration file.
     *
     * @return                              the array of loaded modules
     */
    protected Module[] loadModules() {
        NodeList nodeList=m_guiConfiguration.getElementsByTagName("modules");
        if (nodeList.getLength()!=1)
            throw new IllegalArgumentException("Configuration resource must have exactly one application.");
        Element modulesNode=(Element)nodeList.item(0);
        NodeList modules=modulesNode.getElementsByTagName("module");
        Module[] result=new Module[modules.getLength()];
        for (int i=0;i<modules.getLength();i++) {
            Element moduleNode=(Element)modules.item(i);
            String moduleClassName=moduleNode.getAttribute("name");
            try {
                Class moduleClass=Class.forName(moduleClassName);
                result[i]=(Module)moduleClass.newInstance();
            }
            catch (ClassNotFoundException e) {
                IllegalArgumentException error=new IllegalArgumentException("Module class '"+moduleClassName+"' not found.");
                error.initCause(e);
                throw error;
            }
            catch (InstantiationException e) {
                IllegalArgumentException error=new IllegalArgumentException("Cannot create module class '"+moduleClassName+"'.");
                error.initCause(e);
                throw error;
            }
            catch (IllegalAccessException e) {
                IllegalArgumentException error=new IllegalArgumentException("Cannot create module class '"+moduleClassName+"'.");
                error.initCause(e);
                throw error;
            }
        }
        return result;
    }
    /**
     * Creates a new viewable window.
     *
     * @param viewable                          viewable pane for which the window is creataed
     * @return                                  anchor within which the viewable was placed
     */
    public ViewableAnchor newViewableAncor(Viewable viewable) {
        return newViewableAncor(viewable,null);
    }
    /**
     * Creates a new viewable window.
     *
     * @param viewable                          viewable pane for which the window is creataed
     * @param configuration                     the configuration specifying the position of the anchor
     * @return                                  anchor within which the viewable was placed
     */
    public ViewableAnchor newViewableAncor(final Viewable viewable,Configuration configuration) {
        ViewableAnchor result=createViewableAncor(viewable);
        if (configuration!=null) {
            int x=configuration.getInt("anchor.x",-100);
            int y=configuration.getInt("anchor.y",-100);
            if (x!=-100 && y!=-100)
                result.setLocation(x,y);
            int width=configuration.getInt("anchor.width",-100);
            int height=configuration.getInt("anchor.height",-100);
            if (width!=-100 && height!=-100)
                result.setSize(width,height);
        }
        result.setVisible(true);
        List viewables=getViewables();
        m_propertyChangeSupport.firePropertyChange(VIEWABLES_PROPERTY,null,viewables);
        return result;
    }
    /**
     * Called by the frame when the viewable is disposed.
     *
     * @param viewable                          viewable that is closing
     */
    void viewableDisposed(Viewable viewable) {
        List viewables=getViewables();
        m_propertyChangeSupport.firePropertyChange(VIEWABLES_PROPERTY,null,viewables);
    }
    /**
     * Returns all open viewables of given module.
     *
     * @param module                            module whose viewables are open
     * @return                                  the set of all open viewables of given module
     */
    public List getViewables(Module module) {
        List result=new LinkedList();
        Iterator iterator=getViewables().iterator();
        while (iterator.hasNext()) {
            Viewable viewable=(Viewable)iterator.next();
            if (viewable.getModule()==module)
                result.add(viewable);
        }
        return result;
    }
    /**
     * Closes all open viewables.
     */
    public void closeAllViewables() {
        Iterator viewables=getViewables().iterator();
        while (viewables.hasNext()) {
            Viewable viweable=(Viewable)viewables.next();
            viweable.closing();
        }
    }
    /**
     * Puts the application in the wait-state (displays the wait cursor).
     */
    public void startWaitState() {
        addWaitState(getMainFrameWindow());
    }
    /**
     * Adds the wait-state to the supplied root pane container.
     *
     * @param rootPaneContainer                 the component being processed
     */
    public void addWaitState(RootPaneContainer rootPaneContainer) {
        rootPaneContainer.getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        rootPaneContainer.getGlassPane().setVisible(true);
        if (rootPaneContainer instanceof Window) {
            Window[] owned=((Window)rootPaneContainer).getOwnedWindows();
            for (int i=0;i<owned.length;i++)
                if (owned[i] instanceof RootPaneContainer)
                    addWaitState((RootPaneContainer)owned[i]);
        }
    }
    /**
     * Puts the application out of the wait-state (restores the original cursor).
     */
    public void endWaitState() {
        removeWaitState(getMainFrameWindow());
    }
    /**
     * Removes the wait-state from the supplied root pane container.
     *
     * @param rootPaneContainer                 the component being processed
     */
    public void removeWaitState(RootPaneContainer rootPaneContainer) {
        rootPaneContainer.getGlassPane().setCursor(null);
        rootPaneContainer.getGlassPane().setVisible(false);
        if (rootPaneContainer instanceof java.awt.Window) {
            java.awt.Window[] owned=((java.awt.Window)rootPaneContainer).getOwnedWindows();
            for (int i=0;i<owned.length;i++)
                if (owned[i] instanceof RootPaneContainer)
                    removeWaitState((RootPaneContainer)owned[i]);
        }
    }
    /**
     * Saves the UI state of all modules.
     *
     * @param configuration                     the configuration receiving the UI state
     */
    public void saveUIState(Configuration configuration) {
        configuration.clear();
        List viewables=getViewables();
        int index=0;
        Iterator iterator=viewables.iterator();
        while (iterator.hasNext()) {
            Viewable viewable=(Viewable)iterator.next();
            Configuration subConfiguration=configuration.getSubConfiguration(String.valueOf(index));
            if (viewable.saveUIState(subConfiguration)) {
                subConfiguration.setString("moduleName",viewable.getModule().getModuleName());
                ViewableAnchor viewableAnchor=viewable.getViewableAnchor();
                subConfiguration.setInt("anchor.x",viewableAnchor.getLocation().x);
                subConfiguration.setInt("anchor.y",viewableAnchor.getLocation().y);
                subConfiguration.setInt("anchor.width",viewableAnchor.getSize().width);
                subConfiguration.setInt("anchor.height",viewableAnchor.getSize().height);
                index++;
            }
        }
        configuration.setInt("count",index);
    }
    /**
     * Restores the UI state.
     *
     * @param configuration                     the configuration containing the UI state
     */
    public void restoreUIState(Configuration configuration) {
        int count=configuration.getInt("count",0);
        for (int index=count-1;index>=0;index--) {
            Configuration subConfiguration=configuration.getSubConfiguration(String.valueOf(index));
            String moduleName=subConfiguration.getString("moduleName","");
            Module module=getModule(moduleName);
            if (module!=null)
                module.restoreUIState(subConfiguration);
        }
    }
    /**
     * Saves the UI state into a workspace file.
     *
     * @param file                              the workspace file
     * @return                                  <code>true</code> if workspace was saved correctly
     */
    public boolean saveWorkspace(File file) {
        Configuration configuration=new Configuration(file);
        saveUIState(configuration);
        try {
            configuration.save();
            m_lastWorkspaceFile=file;
            return true;
        }
        catch (IOException e) {
            displayErrorNotification(e);
            return false;
        }
    }
    /**
     * Saves the workspace into a selected file
     *
     * @return                                  <code>true</code> if workspace was saved correctly
     */
    public boolean saveWorkspace() {
        JFileChooserEx fileChooser=new JFileChooserEx();
        ExtensionFileFilter defaultFilter=new ExtensionFileFilter(m_localizationManager,"guibase.kaonWorkspaceFiles",new String[] { "kws" });
        fileChooser.addChoosableFileFilter(defaultFilter);
        fileChooser.setFileFilter(defaultFilter);
        String workspaceFileName=getConfiguration().getString("guibase.workspaceFile");
        if (workspaceFileName!=null)
            fileChooser.setSelectedFile(new File(workspaceFileName));
        String workspaceChooserCurrentDirectoryName=getConfiguration().getString("guibase.workspaceChooserCurrentDirectory");
        if (workspaceChooserCurrentDirectoryName!=null)
            fileChooser.setCurrentDirectory(new File(workspaceChooserCurrentDirectoryName));
        if (fileChooser.showSaveDialog(getMainFrameWindow())==JFileChooserEx.APPROVE_OPTION) {
            File workspaceFile=fileChooser.getSelectedFileEx();
            getConfiguration().setString("guibase.workspaceFile",workspaceFile.getAbsolutePath());
            getConfiguration().setString("guibase.workspaceChooserCurrentDirectory",fileChooser.getCurrentDirectory().getAbsolutePath());
            return saveWorkspace(workspaceFile);
        }
        return false;
    }
    /**
     * Loads the workspace into a selected file
     */
    public void loadWorkspace() {
        JFileChooserEx fileChooser=new JFileChooserEx();
        ExtensionFileFilter defaultFilter=new ExtensionFileFilter(m_localizationManager,"guibase.kaonWorkspaceFiles",new String[] { "kws" });
        fileChooser.addChoosableFileFilter(defaultFilter);
        fileChooser.setFileFilter(defaultFilter);
        String workspaceFileName=getConfiguration().getString("guibase.workspaceFile");
        if (workspaceFileName!=null)
            fileChooser.setSelectedFile(new File(workspaceFileName));
        String workspaceChooserCurrentDirectoryName=getConfiguration().getString("guibase.workspaceChooserCurrentDirectory");
        if (workspaceChooserCurrentDirectoryName!=null)
            fileChooser.setCurrentDirectory(new File(workspaceChooserCurrentDirectoryName));
        if (fileChooser.showOpenDialog(getMainFrameWindow())==JFileChooserEx.APPROVE_OPTION) {
            File workspaceFile=fileChooser.getSelectedFileEx();
            loadWorkspace(workspaceFile);
        }
    }
    /**
     * Loads the UI state from a workspace file.
     *
     * @param file                              the workspace file
     */
    public void loadWorkspace(File file) {
        Configuration configuration=new Configuration(file);
        try {
            configuration.load();
            m_lastWorkspaceFile=file;
        }
        catch (IOException e) {
            displayErrorNotification(e);
            return;
        }
        closeAllViewables();
        restoreUIState(configuration);
    }
    /**
     * Creates the main frame window.
     */
    protected abstract void createMainFrameWindow();
    /**
     * Sets the toolbar to the main window.
     *
     * @param toolBar                           the toolbar to set
     */
    protected abstract void setMainFrameToolBar(JToolBar toolBar);
    /**
     * Returns the main application frame window.
     *
     * @return                                  main application frame window
     */
    public abstract JFrame getMainFrameWindow();
    /**
     * Creates a new viewable window.
     *
     * @param viewable                          viewable pane for which the window is creataed
     * @return                                  anchor within which the viewable was placed
     */
    protected abstract ViewableAnchor createViewableAncor(Viewable viewable);
    /**
     * Returns the currently selected viewable.
     *
     * @return                                  currently selected viewable
     */
    public abstract Viewable getSelectedViewable();
    /**
     * Returns the current list of viewables.
     *
     * @return                                  the current list of viewables
     */
    public abstract List getViewables();

    /**
     * Updates the visibility status of components a tree of components accorsing to their 'modules' property.
     *
     * @param component                         component whose status is changed
     * @param viewableName                      the name of the current viewable
     */
    public static void updateVisibleComponents(Component component,String viewableName) {
        updateVisibleComponentsInternal(component,viewableName==null ? null : ","+viewableName+",");
    }
    /**
     * Implementation method for the <code>updateVisibleComponents</code> method.
     *
     * @param component                         component whose status is changed
     * @param viewableName                      the name of the current viewable
     */
    protected static void updateVisibleComponentsInternal(Component component,String viewableName) {
        if (component instanceof JComponent) {
            JComponent jComponent=(JComponent)component;
            String contexts=(String)jComponent.getClientProperty("guibase.contexts");
            if (contexts!=null)
                if (viewableName==null)
                    jComponent.setVisible(false);
                else
                    jComponent.setVisible(contexts.indexOf(viewableName)!=-1);
        }
        if (component instanceof JMenu) {
            JMenu menu=(JMenu)component;
            JPopupMenu popupMenu=menu.getPopupMenu();
            updateVisibleComponentsInternal(popupMenu,viewableName);
            boolean enabled;
            if (menu.getAction()!=null)
                enabled=menu.getAction().isEnabled();
            else {
                enabled=false;
                for (int i=0;i<popupMenu.getComponentCount();i++)
                    if (popupMenu.getComponent(i).isVisible()) {
                        enabled=true;
                        break;
                    }
            }
            menu.setEnabled(enabled);
        }
        if (component instanceof Container) {
            Container container=(Container)component;
    		int size=container.getComponentCount();
	        for (int i=0;i<size;i++)
                updateVisibleComponentsInternal(container.getComponent(i),viewableName);
        }
    }

    /**
     * Action that terminates the application.
     */
    protected class Exit extends AbstractSmartAction {
        public Exit() {
            super("action.guibase.exit",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            exitApplication();
        }
    }
    /**
     * Action that undoes a command.
     */
    protected class Undo extends AbstractSmartAction {
        public Undo() {
            super("action.guibase.undo",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            getSelectedViewable().undo();
        }
        public void updateAction() {
            setEnabled(getSelectedViewable()!=null && getSelectedViewable().canUndo());
        }
    }
    /**
     * Action that redoes a command.
     */
    protected class Redo extends AbstractSmartAction {
        public Redo() {
            super("action.guibase.redo",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            getSelectedViewable().redo();
        }
        public void updateAction() {
            setEnabled(getSelectedViewable()!=null && getSelectedViewable().canRedo());
        }
    }
    /**
     * Action that saves the state of the UI.
     */
    protected class SaveWorkspace extends AbstractSmartAction {
        public SaveWorkspace() {
            super("action.guibase.saveWorkspace",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            saveWorkspace();
        }
    }
    /**
     * Action that loads the state of the UI.
     */
    protected class LoadWorkspace extends AbstractSmartAction {
        public LoadWorkspace() {
            super("action.guibase.loadWorkspace",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            loadWorkspace();
        }
    }
    /**
     * Action that chooses the look&feel.
     */
    protected class SetLookAndFeel extends AbstractSmartAction implements ModalSmartAction {
        public SetLookAndFeel() {
            super("action.guibase.setLookAndFeel",m_localizationManager);
        }
        public void actionPerformed(ActionEvent e) {
            try {
                UIManager.setLookAndFeel(e.getActionCommand());
                m_configuration.setString("lookAndFeel",UIManager.getLookAndFeel().getClass().getName());
            }
            catch (Exception error) {
                displayErrorNotification(error);
            }
        }
        public String[][] getModalOptions() {
            return s_lookAndFeels;
        }
        public boolean isOptionSelected(int optionIndex) {
            return s_lookAndFeels[optionIndex][0].equals(UIManager.getLookAndFeel().getClass().getName());
        }
    }
}
