package de.fzi.wim.guibase.appdriver;

import javax.swing.JComponent;

import de.fzi.wim.guibase.actions.SmartActionMap;
import de.fzi.wim.guibase.actions.SmartActionHashMap;
import de.fzi.wim.guibase.actions.SmartAction;
import de.fzi.wim.guibase.configuration.Configuration;

import de.fzi.wim.guibase.command.*;

/**
 * Abstract base class for all viewables.
 */
public abstract class AbstractViewable implements Viewable {
    /** The module of this object. */
    protected Module m_module;
    /** Viewable anchor. */
    protected ViewableAnchor m_viewableAnchor;
    /** Component of this viewble. */
    protected JComponent m_component;
    /** The smart action map of this viewable. */
    protected SmartActionMap m_actionMap;
    /** The command manager. */
    protected CommandManager m_commandManager;

    /**
     * Creates an instance of this class.
     *
     * @param module                        the module of this object
     */
    public AbstractViewable(Module module) {
        m_module=module;
        m_actionMap=new SmartActionHashMap(m_module.getSmartActionMap());
        m_commandManager=new CommandManager(m_module.getAppDriver().getThreadPool());
        m_commandManager.addCommandManagerListener(new CommandManagerListener() {
            public void commandWillExecute(CommandManager commandManager,Command command,int executionType) {
                getSmartActionMap().updateActions();
            }
            public void commandExecuted(CommandManager commandManager,Command command,int executionType) {
                getSmartActionMap().updateActions();
            }
            public void commandException(CommandManager commandManager,Command command,CommandException commandException) {
                getSmartActionMap().updateActions();
                if (!(commandException instanceof CommandAbortedException))
                    m_module.getAppDriver().displayErrorNotification(commandException);
            }
            public void undoRedoAvailabilityChanged(CommandManager commandManager,boolean canUndo,boolean canRedo) {
                m_module.getAppDriver().getSmartActionMap().getAction("action.guibase.undo").updateAction();
                m_module.getAppDriver().getSmartActionMap().getAction("action.guibase.redo").updateAction();
            }
        });
    }
    /**
     * Retruns the name of this viewable.
     *
     * @return                              the name of this viewable
     */
    public String getViewableName() {
        return getModule().getModuleName();
    }
    /**
     * Returns the module to which this object belongs to.
     *
     * @return                              the module of this object
     */
    public Module getModule() {
        return m_module;
    }
    /**
     * Returns the smart action map that should be used when this viewable is active.
     *
     * @return                              the smart action map of this viewable
     */
    public SmartActionMap getSmartActionMap() {
        return m_actionMap;
    }
    /**
     * Attaches this object to an anchor.
     *
     * @param anchor                        anchor to which this object  needs to be attached
     */
    public void setViewableAnchor(ViewableAnchor anchor) {
        m_viewableAnchor=anchor;
    }
    /**
     * Returns the current anchor of this viewable.
     *
     * @return                              current anchor of this viewable
     */
    public ViewableAnchor getViewableAnchor() {
        return m_viewableAnchor;
    }
    /**
     * Called when anchor is closing.
     */
    public void closing() {
        m_viewableAnchor.dispose();
    }
    /**
     * Called when view will be activated. Should return <code>true</code> if view should be activated.
     *
     * @return                              <code>true</code> if view should be activated
     */
    public boolean activating() {
        return true;
    }
    /**
     * Called when view is activated.
     */
    public void activated() {
    }
    /**
     * Called when view will be deactivated. Should return <code>true</code> if view should be deactivated.
     *
     * @return                              <code>true</code> if view should be deactivated
     */
    public boolean deactivating() {
        return true;
    }
    /**
     * Called when view is deactivated.
     */
    public void deactivated() {
    }
    /**
     * Returns the component of this viewable.
     *
     * @return                                  component of this viewable
     */
    public JComponent getComponent() {
        return m_component;
    }
    /**
     * Disposes of this viewable.
     */
    public void dispose() {
    }
    /**
     * Checks whether the application can terminate.
     *
     * @return                              <code>true</code> if application can terminate
     */
    public boolean canExitApplication() {
        return true;
    }
    /**
     * Notifies the module that the application is exiting.
     */
    public void applicationExiting() {
        m_viewableAnchor.dispose();
    }
    /**
     * Adds a viewable-specific action to the action map.
     *
     * @param action                        action to add
     */
    protected void addAction(SmartAction action) {
        m_actionMap.put(action.getActionID(),action);
    }
    /**
     * Updates all actions of this viewable.
     */
    protected void updateActions() {
        getSmartActionMap().updateActions();
    }
    /**
     * Returns <code>true</code> if a command can be executed for this OI-model.
     *
     * @return                                  <code>true</code> if a command can be executed for this OI-model
     */
    public boolean commandsAreEnabled() {
        return !m_commandManager.isCommandBeingExecuted();
    }
    /**
     * Called to save the UI state of the viewable.
     *
     * @param configuration                 the configuration
     * @return                              <code>true</code> if the configuration saved
     */
    public boolean saveUIState(Configuration configuration) {
        return false;
    }
    /**
     * Undoes an action.
     */
    public void undo() {
        try {
            m_commandManager.undo();
        }
        catch (CommandAbortedException ignored) {
        }
        catch (CommandException error) {
            m_module.getAppDriver().displayErrorNotification(error);
        }
    }
    /**
     * Returns <code>true</code> if undo can be performed.
     *
     * @return                              <code>true</code> if undo can be performed
     */
    public boolean canUndo() {
        return m_commandManager.getCanUndo();
    }
    /**
     * Redoes an action.
     */
    public void redo() {
        try {
            m_commandManager.redo();
        }
        catch (CommandAbortedException ignored) {
        }
        catch (CommandException error) {
            m_module.getAppDriver().displayErrorNotification(error);
        }
    }
    /**
     * Returns <code>true</code> if redo can be performed.
     *
     * @return                              <code>true</code> if redo can be performed
     */
    public boolean canRedo() {
        return m_commandManager.getCanRedo();
    }
    /**
     * Executes a command with the context of the current viewable. If an exception
     * happens, it is reported using the default mechamism.
     *
     * @param command                           command to execute
     */
    public void executeCommandEx(Command command) {
        try {
            m_commandManager.execute(command);
        }
        catch (CommandAbortedException ignored) {
        }
        catch (CommandException e) {
            m_module.getAppDriver().displayErrorNotification(e);
        }
    }
}
