package de.fzi.wim.guibase.actions;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.awt.event.ActionEvent;
import javax.swing.event.PopupMenuEvent;

import de.fzi.wim.guibase.localization.LocalizationManager;

/**
 * An implementation of a SmartAction that decorates an existing action.
 */
public abstract class SmartActionDecorator extends AbstractSmartAction implements PropertyChangeListener {
    protected static final String[] s_properties={ SmartAction.DEFAULT,SmartAction.NAME,SmartAction.SHORT_DESCRIPTION,SmartAction.LONG_DESCRIPTION,SmartAction.SMALL_ICON,SmartAction.ACTION_COMMAND_KEY,SmartAction.ACCELERATOR_KEY,SmartAction.MNEMONIC_KEY,SmartAction.ACTION_ID,SmartAction.SELECTED };

    /** Currently selected action. */
    protected SmartAction m_currentAction;

    /**
     * Creates and initializes an instance of this class.
     *
     * @param actionID                          ID of this action
     * @param localizationManager               the localization manager
     */
    public SmartActionDecorator(String actionID,LocalizationManager localizationManager) {
        super(actionID,localizationManager);
    }
    /**
    * Invokes this action by invoking the target.
    *
    * @param event The action event that is passed to the target action.
    */
    public void actionPerformed(ActionEvent event) {
        if (m_currentAction!=null)
            m_currentAction.actionPerformed(event);
    }
    /**
     * Returns the value with given key.
     *
     * @param key                   key of the value to be retrieved
     * @return                      value with given key
     */
    public Object getValue(String key) {
        Object value=null;
        if (m_currentAction!=null)
            value=m_currentAction.getValue(key);
        if (value==null)
            value=super.getValue(key);
        return value;
    }
    /**
     * Sets the value with given key.
     *
     * @param key                   key of the value to be set
     * @param value                 value with given key
     */
    public void putValue(String key,Object value) {
        if (m_currentAction!=null)
            m_currentAction.putValue(key,value);
        else
            super.putValue(key,value);
    }
    /**
     * Returns <code>true</code> if this action is enabled.
     *
     * @return                      <code>true</code> if this action is enabled
     */
    public boolean isEnabled() {
        if (m_currentAction!=null)
            return m_currentAction.isEnabled();
        else
            return false;
    }
    /**
     * Sets the enabled state of this action.
     *
     * @param enabled               <code>true</code> if this action's state is enabled
     */
    public void setEnabled(boolean enabled) {
        if (m_currentAction!=null)
            m_currentAction.setEnabled(enabled);
    }
    /**
     * Returns whether this action is selected.
     *
     * @return                              <code>true</code> if this action is selected
     */
    public boolean isSelected() {
        if (m_currentAction!=null)
            return m_currentAction.isSelected();
        else
            return false;
    }
    /**
     * Sets the selected property of the action.
     *
     * @param selected                      selected property
     */
    public void setSelected(boolean selected) {
        if (m_currentAction!=null)
            m_currentAction.setSelected(selected);
    }
    /**
     * Updates this action by calling update on the target and then copying the target's enabled status.
     */
    public void updateAction() {
        updateCurrentAction();
        if (m_currentAction!=null)
            m_currentAction.updateAction();
    }
	/**
	 * An implementation side-effect since this class implements <code>PopupMenuListener</code> interface.
	 *
	 * @param e                             Swing event
	 */
	public void popupMenuCanceled(PopupMenuEvent e) {
        if (m_currentAction!=null)
            m_currentAction.popupMenuCanceled(e);
	}
	/**
	 * An implementation side-effect since this class implements <code>PopupMenuListener</code> interface.
	 *
	 * @param e Swing event.
	 */
	public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
        if (m_currentAction!=null)
            m_currentAction.popupMenuWillBecomeInvisible(e);
	}
	/**
	 * Calls <code>updateAction</code> so that action is update before menu is shown on screen.
	 *
	 * @param e                             Swing event
	 */
	public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
        if (m_currentAction!=null)
            m_currentAction.popupMenuWillBecomeVisible(e);
	}
    /**
     * Called when currnet actions's properties are updated.
     *
     * @param event                 event
     */
    public void propertyChange(PropertyChangeEvent event) {
        if (event.getSource()==m_currentAction)
            firePropertyChange(event.getPropertyName(),event.getOldValue(),event.getNewValue());
    }
    /**
     * Computes the current action of this aggregate.
     */
    protected void updateCurrentAction() {
        SmartAction oldCurrentAction=m_currentAction;
        m_currentAction=getDecoratedAction();
        if (oldCurrentAction!=m_currentAction) {
            if (oldCurrentAction!=null)
                oldCurrentAction.removePropertyChangeListener(this);
            if (m_currentAction!=null)
                m_currentAction.addPropertyChangeListener(this);
            for (int i=0;i<s_properties.length;i++)
                fireValueUpdate(oldCurrentAction,m_currentAction,s_properties[i]);
            boolean oldEnabled=oldCurrentAction==null ? false : oldCurrentAction.isEnabled();
            boolean newEnabled=m_currentAction==null ? false : m_currentAction.isEnabled();
            if (oldEnabled!=newEnabled)
                firePropertyChange(ENABLED,new Boolean(oldEnabled),new Boolean(newEnabled));
        }
    }
    /**
     * Fires the update event when current action is changed.
     *
     * @param oldAction                     old smart action
     * @param newAction                     new smart action
     * @param key                           key of the value
     */
    protected void fireValueUpdate(SmartAction oldAction,SmartAction newAction,String key) {
        Object oldValue=null;
        if (oldAction!=null)
            oldValue=oldAction.getValue(key);
        if (oldValue==null)
            oldValue=super.getValue(key);
        Object newValue=null;
        if (newAction!=null)
            newValue=newAction.getValue(key);
        if (newValue==null)
            newValue=super.getValue(key);
        if ((oldValue!=null && !oldValue.equals(newValue)) || (oldValue==null && newValue!=null))
            firePropertyChange(key,oldValue,newValue);
    }
    /**
     * Returns the decorated action.
     *
     * @return                              decorated action
     */
    protected abstract SmartAction getDecoratedAction();
}
