package de.fzi.wim.guibase.actions;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import javax.swing.AbstractAction;
import javax.swing.KeyStroke;
import javax.swing.Icon;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;

import de.fzi.wim.guibase.localization.LocalizationManager;

/**
 * Default implementation of {link SmartAction} interface.
 */
public abstract class AbstractSmartAction extends AbstractAction implements SmartAction,PopupMenuListener {
    /** Keys of all action names that are returned in an action map of anny action. */
    protected static final String[] s_actionKeys={ ACTION_ID,ACCELERATOR_KEY,ACTION_COMMAND_KEY,LONG_DESCRIPTION,MNEMONIC_KEY,NAME,SHORT_DESCRIPTION,SMALL_ICON };

    /**
     * Creates an action with default parameters.
     *
     * @param actionID                      ID of this action
     */
    public AbstractSmartAction(String actionID) {
        putValue(ACTION_ID,actionID);
    }
    /**
     * Initializes this action from the map of action properties.
     *
     * @param actionMap                     map of action properties
     */
    public AbstractSmartAction(Map actionMap) {
        setToActionMap(actionMap);
    }
    /**
     * Creates an action with given ID localized from the supplied localization manager.
     *
     * @param actionID                      ID of this action (specifies resources in the localization manager)
     * @param localizationManager           localization manager for the action
     */
    public AbstractSmartAction(String actionID,LocalizationManager localizationManager) {
        this(localizationManager.getActionMap(actionID));
    }
    /**
     * Returns the action ID of this action.
     *
     * @return                              action ID of this action
     */
    public String getActionID() {
        return (String)getValue(ACTION_ID);
    }
    /**
     * Returns the action map of this action.
     *
     * @return                              action map of this action
     */
    public Map getActionMap() {
        Map actionMap=new HashMap(11);
        Object[] keys=getKeys();
        for (int i=keys.length-1;i>=0;--i) {
            Object value=getValue((String)keys[i]);
            if (value!=null)
                actionMap.put(keys[i],value);
        }
        return actionMap;
    }
    /**
     * Updates the values of this action with values from the action map.
     *
     * @param actionMap                     values that this action needs to be updated with
     */
    public void setToActionMap(Map actionMap) {
        Iterator keys=actionMap.keySet().iterator();
        while (keys.hasNext()) {
            String key=(String)keys.next();
            Object value=actionMap.get(key);
            putValue(key,value);
        }
    }
    /**
     * Sets action name.
     *
     * @param name                          new name of this action
     */
    public void setName(String name) {
        putValue(NAME,name);
    }
    /**
     * Sets action icon.
     *
     * @param icon                          new icon of this action
     */
    public void setIcon(Icon icon) {
        putValue(SMALL_ICON,icon);
    }
    /**
     * Sets short description of this action (used to display tooltip text).
     *
     * @param shortDescription              new short description of this action
     */
    public void setShortDescription(String shortDescription) {
        putValue(SHORT_DESCRIPTION,shortDescription);
    }
    /**
     * Sets accelerator key of this action.
     *
     * @param acceleratorKey                new accelerator key of this action
     */
    public void setAcceleratorKey(KeyStroke acceleratorKey) {
        putValue(ACCELERATOR_KEY,acceleratorKey);
    }
    /**
     * Sets mnemonic key of this action.
     *
     * @param mnemonicKey                   new mnemonic key of this action
     */
    public void setMnemonicKey(int mnemonicKey) {
        putValue(MNEMONIC_KEY,new Integer(mnemonicKey));
    }
    /**
     * Sets long description of this action.
     *
     * @param longDescription               new long description of this action
     */
    public void setLongDescription(String longDescription) {
        putValue(LONG_DESCRIPTION,longDescription);
    }
    /**
     * Sets the action command key.
     *
     * @param commandKey                    string that is sent to action listeners when action is invoked
     */
    public void setCommandKey(String commandKey) {
        putValue(ACTION_COMMAND_KEY,commandKey);
    }
    /**
     * Should be overridden to provide the predicate for computing the state of the action. State might include updating
     * enabled state of the icon, but might also include update the name (e.g. based on the current selection).
     * Default implementation does nothing.
     */
    public void updateAction() {
    }
    /**
     * An implementation side-effect since this class implements <code>PopupMenuListener</code> interface.
     *
     * @param e                             Swing event
     */
    public void popupMenuCanceled(PopupMenuEvent e) {
    }
    /**
     * An implementation side-effect since this class implements <code>PopupMenuListener</code> interface.
     *
     * @param e Swing event.
     */
    public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
    }
    /**
     * Calls <code>updateAction</code> so that action is update before menu is shown on screen.
     *
     * @param e                             Swing event
     */
    public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
        updateAction();
    }
    /**
     * Sets a value under given name in this action. Overridden to not fire an event if new and old values are equal (this
     * reduces event storms generated in the course of updating an action).
     *
     * @param name                          name for the value being set
     * @param newValue                      value being set
     */
    public void putValue(String name,Object newValue) {
        Object oldValue=getValue(name);
        if (oldValue==null) {
            if (newValue==null)
                return;
        }
        else {
            if (oldValue.equals(newValue))
                return;
        }
        super.putValue(name,newValue);
    }
    /**
     * Designates this action as enabled or disabled. Overridden to not fire an event if new and old values are equal (this
     * reduces event storms generated in the course of updating an action).
     *
     * @param enabled                       <code>true</code> if action should be enabled
     */
    public void setEnabled(boolean enabled) {
        if (isEnabled()!=enabled)
            super.setEnabled(enabled);
    }
    /**
     * Returns whether this action is selected.
     *
     * @return                              <code>true</code> if this action is selected
     */
    public boolean isSelected() {
        Boolean selected=(Boolean)getValue(SELECTED);
        if (selected==null)
            return false;
        return selected.booleanValue();
    }
    /**
     * Sets the selected property of the action.
     *
     * @param selected                      selected property
     */
    public void setSelected(boolean selected) {
        Boolean newValue=new Boolean(selected);
        putValue(SELECTED,newValue);
    }
}
