package edu.unika.aifb.kaon.engineeringserver.util;

import java.util.Map;
import java.util.HashMap;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Utility class for writing models directly into the database. This class is particularly useful
 * for importing large amounts of data into an empty database, since the performance of the import
 * is much better than doing this through the KAON API. The writer can import several models at once.
 * It imposes the following limitation: the URI of each entity must be unique in all models being imported.
 */
public class DirectOIModelWriter {
    /** The connection for the model. */
    protected Connection m_connection;
    /** The map of model physical URIs to model IDs. */
    protected Map m_modelIDsByLogicalURIs;
    /** The map of URIs to information about entities. */
    protected Map m_entityInfosByURIs;
    /** The number of additions since the last flush. */
    protected int m_additionsSinceFlush;
    /** The last OIModelEntity ID. */
    protected int m_lastOIModelEntityID;
    /** The statement for adding OIModelEntity objects. */
    protected PreparedStatement m_addOIModelEntity;
    /** The statement for selecting OIModelEntity objects. */
    protected PreparedStatement m_selectOIModelEntity;
    /** The statement for adding InverseProperty objects. */
    protected PreparedStatement m_addInverseProperty;
    /** The statement for adding ConceptHierarchy objects. */
    protected PreparedStatement m_addConceptHierarchy;
    /** The statement for adding PropertyHierarchy objects. */
    protected PreparedStatement m_addPropertyHierarchy;
    /** The statement for adding PropertyDomain objects. */
    protected PreparedStatement m_addPropertyDomain;
    /** The statement for adding PropertyRange objects. */
    protected PreparedStatement m_addPropertyRange;
    /** The statement for adding ConceptInstance objects. */
    protected PreparedStatement m_addConceptInstance;
    /** The statement for adding RelationInstance objects. */
    protected PreparedStatement m_addRelationInstance;
    /** The statement for adding AttributeInstance objects. */
    protected PreparedStatement m_addAttributeInstance;

    /**
     * Creates an instance of this class. For performance, the connection should not have auto-commit turned on. We don't turn
     * it off in the constructor in order to allow using this class in an application server.
     *
     * @param connection                        the connection
     * @throws SQLException                     thrown if there is an error
     */
    public DirectOIModelWriter(Connection connection) throws SQLException {
        m_connection=connection;
        m_modelIDsByLogicalURIs=new HashMap();
        m_entityInfosByURIs=new HashMap();
        m_addOIModelEntity=m_connection.prepareStatement("INSERT INTO OIModelEntity (entityID,modelID,entityURI,conceptVersion,propertyVersion,isAttribute,isSymmetric,isTransitive,instanceVersion) VALUES (?,?,?,?,?,?,?,?,?)");
        m_selectOIModelEntity=m_connection.prepareStatement("SELECT entityID,conceptVersion,propertyVersion,instanceVersion FROM OIModelEntity WHERE entityURI=? AND modelID IN (SELECT includedModelID FROM AllIncludedOIModels WHERE includingModelID=?)");
        m_addInverseProperty=m_connection.prepareStatement("INSERT INTO InverseProperty (modelID,propertyID,inversePropertyID) VALUES (?,?,?)");
        m_addConceptHierarchy=m_connection.prepareStatement("INSERT INTO ConceptHierarchy (modelID,superConceptID,subConceptID) VALUES (?,?,?)");
        m_addPropertyHierarchy=m_connection.prepareStatement("INSERT INTO PropertyHierarchy (modelID,superPropertyID,subPropertyID) VALUES (?,?,?)");
        m_addPropertyDomain=m_connection.prepareStatement("INSERT INTO PropertyDomain (modelID,propertyID,conceptID,minimumCardinality,maximumCardinality) VALUES (?,?,?,?,?)");
        m_addPropertyRange=m_connection.prepareStatement("INSERT INTO PropertyRange (modelID,propertyID,conceptID) VALUES (?,?,?)");
        m_addConceptInstance=m_connection.prepareStatement("INSERT INTO ConceptInstance (modelID,conceptID,instanceID) VALUES (?,?,?)");
        m_addRelationInstance=m_connection.prepareStatement("INSERT INTO RelationInstance (modelID,propertyID,sourceInstanceID,targetInstanceID) VALUES (?,?,?,?)");
        m_addAttributeInstance=m_connection.prepareStatement("INSERT INTO AttributeInstance (modelID,propertyID,sourceInstanceID,textValue) VALUES (?,?,?,?)");
        m_lastOIModelEntityID=getLastOIModelEntityID();
    }
    /**
     * Flushes the contents of the writer.
     *
     * @throws SQLException                     thrown if there is an error
     */
    public void flush() throws SQLException {
        m_addOIModelEntity.executeBatch();
        m_addInverseProperty.executeBatch();
        m_addConceptHierarchy.executeBatch();
        m_addPropertyHierarchy.executeBatch();
        m_addPropertyDomain.executeBatch();
        m_addPropertyRange.executeBatch();
        m_addConceptInstance.executeBatch();
        m_addRelationInstance.executeBatch();
        m_addAttributeInstance.executeBatch();
        executeUpdate("UPDATE PKCounter SET counter="+m_lastOIModelEntityID+" WHERE type='oimodelentity'");
        m_connection.commit();
        m_additionsSinceFlush=0;
    }
    /**
     * Closes this writer.
     */
    public void close() {
        closeStatement(m_addOIModelEntity);
        closeStatement(m_selectOIModelEntity);
        closeStatement(m_addInverseProperty);
        closeStatement(m_addConceptHierarchy);
        closeStatement(m_addPropertyHierarchy);
        closeStatement(m_addPropertyDomain);
        closeStatement(m_addPropertyRange);
        closeStatement(m_addConceptInstance);
        closeStatement(m_addRelationInstance);
        closeStatement(m_addAttributeInstance);
        m_addOIModelEntity=null;
        m_selectOIModelEntity=null;
        m_addInverseProperty=null;
        m_addConceptHierarchy=null;
        m_addPropertyHierarchy=null;
        m_addPropertyDomain=null;
        m_addPropertyRange=null;
        m_addConceptInstance=null;
        m_addRelationInstance=null;
        m_addAttributeInstance=null;
        m_modelIDsByLogicalURIs=null;
        m_entityInfosByURIs=null;
        m_connection=null;
    }
    /**
     * Returns the ID of the OI-model with given logical URI.
     *
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the ID of the OI-model
     * @throws SQLException                     thrown if there is an error, or if the OI-model doesn't exist
     */
    protected int getModelID(String logicalURI) throws SQLException {
        Integer modelID=(Integer)m_modelIDsByLogicalURIs.get(logicalURI);
        if (modelID==null) {
            Statement statement=m_connection.createStatement();
            try {
                ResultSet resultSet=statement.executeQuery("SELECT modelID FROM OIModel WHERE logicalURI='"+logicalURI+"'");
                try {
                    if (!resultSet.next())
                        throw new SQLException("OI-model '"+logicalURI+"' doesn't exist.");
                    else {
                        modelID=new Integer(resultSet.getInt(1));
                        m_modelIDsByLogicalURIs.put(logicalURI,modelID);
                    }
                }
                finally {
                    resultSet.close();
                }
            }
            finally {
                statement.close();
            }
        }
        return modelID.intValue();
    }
    /**
     * Returns the ID of the entity with given URI. This method DOESN'T check the database.
     * The entity must have been created in this import session.
     *
     * @param modelLogicalURI                   the logical URI of the model in which the entity is looked up
     * @param entityURI                         the URI of the entity
     * @return                                  the ID of the entity
     * @throws SQLException                     thrown if there is an error, or if the entity doesn't exist
     */
    protected int getEntityID(String modelLogicalURI,String entityURI) throws SQLException {
        EntityInfo entityInfo=(EntityInfo)m_entityInfosByURIs.get(entityURI);
        if (entityInfo==null) {
            int modelID=getModelID(modelLogicalURI);
            m_selectOIModelEntity.setString(1,entityURI);
            m_selectOIModelEntity.setInt(2,modelID);
            ResultSet resultSet=m_selectOIModelEntity.executeQuery();
            try {
                if (!resultSet.next())
                    throw new SQLException("Entity '"+entityURI+"' doesn't exist in OI-model '"+modelLogicalURI+"'.");
                else {
                    entityInfo=new EntityInfo(resultSet.getInt(1),resultSet.getInt(2)!=Integer.MAX_VALUE,resultSet.getInt(3)!=Integer.MAX_VALUE,resultSet.getInt(4)!=Integer.MAX_VALUE);
                    m_entityInfosByURIs.put(entityURI,entityInfo);
                }
            }
            finally {
                resultSet.close();
            }
        }
        return entityInfo.m_entityID;
    }
    /**
     * Checks whether for given URI a concept has been added to th OI-model.
     *
     * @param entityURI                         the URI of the entity
     * @return                                  <code>true</code> if the entity is a concept
     */
    public boolean isConcept(String entityURI) {
        EntityInfo entityInfo=(EntityInfo)m_entityInfosByURIs.get(entityURI);
        if (entityInfo==null)
            return false;
        else
            return entityInfo.m_isConcept;
    }
    /**
     * Checks whether for given URI a property has been added to th OI-model.
     *
     * @param entityURI                         the URI of the entity
     * @return                                  <code>true</code> if the entity is a property
     */
    public boolean isProperty(String entityURI) {
        EntityInfo entityInfo=(EntityInfo)m_entityInfosByURIs.get(entityURI);
        if (entityInfo==null)
            return false;
        else
            return entityInfo.m_isProperty;
    }
    /**
     * Checks whether for given URI an instance has been added to th OI-model.
     *
     * @param entityURI                         the URI of the entity
     * @return                                  <code>true</code> if the entity is an instance
     */
    public boolean isInstance(String entityURI) {
        EntityInfo entityInfo=(EntityInfo)m_entityInfosByURIs.get(entityURI);
        if (entityInfo==null)
            return false;
        else
            return entityInfo.m_isInstance;
    }
    /**
     * Creates an entity in the model.
     *
     * @param modelLogicalURI                   the logical URI of the model to which the entity is added
     * @param entityURI                         the URI of the entity
     * @param isConcept                         <code>true</code> if entity is a concept
     * @param isProperty                        <code>true</code> if entity is a property
     * @param isAttribute                       <code>true</code> if property is an attribute
     * @param isSymmetric                       <code>true</code> if property is symmetric
     * @param isTransitive                      <code>true</code> if property is transitive
     * @param isInstance                        <code>true</code> if entity is an instance
     * @throws SQLException                     thrown if there is an error or if entity already exists
     */
    public void createEntity(String modelLogicalURI,String entityURI,boolean isConcept,boolean isProperty,boolean isAttribute,boolean isSymmetric,boolean isTransitive,boolean isInstance) throws SQLException {
        if (m_entityInfosByURIs.containsKey(entityURI))
            throw new SQLException("Entity with URI '"+entityURI+"' has already been added to the model.");
        EntityInfo entityInfo=new EntityInfo(++m_lastOIModelEntityID,isConcept,isProperty,isInstance);
        m_entityInfosByURIs.put(entityURI,entityInfo);
        m_addOIModelEntity.setInt(1,entityInfo.m_entityID);
        m_addOIModelEntity.setInt(2,getModelID(modelLogicalURI));
        m_addOIModelEntity.setString(3,entityURI);
        m_addOIModelEntity.setInt(4,isConcept ? 1 : Integer.MAX_VALUE);
        m_addOIModelEntity.setInt(5,isProperty ? 1 : Integer.MAX_VALUE);
        m_addOIModelEntity.setInt(6,isAttribute ? 1 : 0);
        m_addOIModelEntity.setInt(7,isSymmetric ? 1 : 0);
        m_addOIModelEntity.setInt(8,isTransitive ? 1 : 0);
        m_addOIModelEntity.setInt(9,isInstance ? 1 : Integer.MAX_VALUE);
        m_addOIModelEntity.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds an inverse property relation between two properties.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param propertyURI1                      the URI of the first property
     * @param propertyURI2                      the URI of the second property
     * @throws SQLException                     thrown if there is an error
     */
    public void addInverseProperty(String modelLogicalURI,String propertyURI1,String propertyURI2) throws SQLException {
        m_addInverseProperty.setInt(1,getModelID(modelLogicalURI));
        m_addInverseProperty.setInt(2,getEntityID(modelLogicalURI,propertyURI1));
        m_addInverseProperty.setInt(3,getEntityID(modelLogicalURI,propertyURI2));
        m_addInverseProperty.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds a concept hierarchy relation between two concepts.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param superConceptURI                   the URI of the superconcept
     * @param subConceptURI                     the URI of the subconcept
     * @throws SQLException                     thrown if there is an error
     */
    public void addConceptHierarchy(String modelLogicalURI,String superConceptURI,String subConceptURI) throws SQLException {
        m_addConceptHierarchy.setInt(1,getModelID(modelLogicalURI));
        m_addConceptHierarchy.setInt(2,getEntityID(modelLogicalURI,superConceptURI));
        m_addConceptHierarchy.setInt(3,getEntityID(modelLogicalURI,subConceptURI));
        m_addConceptHierarchy.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds a property hierarchy relation between two properties.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param superPropertyURI                  the URI of the superproperty
     * @param subPropertyURI                    the URI of the subproperty
     * @throws SQLException                     thrown if there is an error
     */
    public void addPropertyHierarchy(String modelLogicalURI,String superPropertyURI,String subPropertyURI) throws SQLException {
        m_addPropertyHierarchy.setInt(1,getModelID(modelLogicalURI));
        m_addPropertyHierarchy.setInt(2,getEntityID(modelLogicalURI,superPropertyURI));
        m_addPropertyHierarchy.setInt(3,getEntityID(modelLogicalURI,subPropertyURI));
        m_addPropertyHierarchy.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds a property domain relation between a property and a concept.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param propertyURI                       the URI of the property
     * @param conceptURI                        the URI of the concept
     * @param minimumCardinality                the minimum cardinality
     * @param maximumCardinality                the maximum cardinality
     * @throws SQLException                     thrown if there is an error
     */
    public void addPropertyDomain(String modelLogicalURI,String propertyURI,String conceptURI,int minimumCardinality,int maximumCardinality) throws SQLException {
        m_addPropertyDomain.setInt(1,getModelID(modelLogicalURI));
        m_addPropertyDomain.setInt(2,getEntityID(modelLogicalURI,propertyURI));
        m_addPropertyDomain.setInt(3,getEntityID(modelLogicalURI,conceptURI));
        m_addPropertyDomain.setInt(4,minimumCardinality);
        m_addPropertyDomain.setInt(5,maximumCardinality);
        m_addPropertyDomain.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds a property range relation between a property and a concept.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param propertyURI                       the URI of the property
     * @param conceptURI                        the URI of the concept
     * @throws SQLException                     thrown if there is an error
     */
    public void addPropertyRange(String modelLogicalURI,String propertyURI,String conceptURI) throws SQLException {
        m_addPropertyRange.setInt(1,getModelID(modelLogicalURI));
        m_addPropertyRange.setInt(2,getEntityID(modelLogicalURI,propertyURI));
        m_addPropertyRange.setInt(3,getEntityID(modelLogicalURI,conceptURI));
        m_addPropertyRange.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds an instance to a concept.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param conceptURI                        the URI of the concept
     * @param instanceURI                       the URI of the instance
     * @throws SQLException                     thrown if there is an error
     */
    public void addConceptInstance(String modelLogicalURI,String conceptURI,String instanceURI) throws SQLException {
        m_addConceptInstance.setInt(1,getModelID(modelLogicalURI));
        m_addConceptInstance.setInt(2,getEntityID(modelLogicalURI,conceptURI));
        m_addConceptInstance.setInt(3,getEntityID(modelLogicalURI,instanceURI));
        m_addConceptInstance.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds a relation instance.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param propertyURI                       the URI of the property
     * @param sourceInstanceURI                 the URI of the source instance
     * @param targetInstanceURI                 the URI of the target instance
     * @throws SQLException                     thrown if there is an error
     */
    public void addRelationInstance(String modelLogicalURI,String propertyURI,String sourceInstanceURI,String targetInstanceURI) throws SQLException {
        m_addRelationInstance.setInt(1,getModelID(modelLogicalURI));
        m_addRelationInstance.setInt(2,getEntityID(modelLogicalURI,propertyURI));
        m_addRelationInstance.setInt(3,getEntityID(modelLogicalURI,sourceInstanceURI));
        m_addRelationInstance.setInt(4,getEntityID(modelLogicalURI,targetInstanceURI));
        m_addRelationInstance.addBatch();
        notifyElementAdded();
    }
    /**
     * Adds an attribute instance.
     *
     * @param modelLogicalURI                   the logical URI of the model to in which this relation is created
     * @param propertyURI                       the URI of the property
     * @param sourceInstanceURI                 the URI of the source instance
     * @param targetValue                       the value of the attribute
     * @throws SQLException                     thrown if there is an error
     */
    public void addAttributeInstance(String modelLogicalURI,String propertyURI,String sourceInstanceURI,String targetValue) throws SQLException {
        m_addAttributeInstance.setInt(1,getModelID(modelLogicalURI));
        m_addAttributeInstance.setInt(2,getEntityID(modelLogicalURI,propertyURI));
        m_addAttributeInstance.setInt(3,getEntityID(modelLogicalURI,sourceInstanceURI));
        m_addAttributeInstance.setString(4,targetValue);
        m_addAttributeInstance.addBatch();
        notifyElementAdded();
    }
    /**
     * Notifies the writer that an element was added.
     *
     * @throws SQLException                     thrown if there is an error
     */
    protected void notifyElementAdded() throws SQLException {
        m_additionsSinceFlush++;
        if (m_additionsSinceFlush>5000)
            flush();
    }
    /**
     * Closes a statement.
     *
     * @param statement                         the statement to be closed
     */
    protected void closeStatement(Statement statement) {
        try {
            statement.close();
        }
        catch (SQLException ignored) {
        }
    }
    /**
     * Executes a SQL update.
     *
     * @param query                             the text of the update
     * @return                                  the result of the update
     * @throws SQLException                     thrown if there is an error
     */
    protected int executeUpdate(String query) throws SQLException {
        Statement statement=m_connection.createStatement();
        try {
            return statement.executeUpdate(query);
        }
        finally {
            statement.close();
        }
    }
    /**
     * Reads the last OIModelEntity ID.
     *
     * @return                                  the ID of the last entity written
     * @throws SQLException                     thrown if there is an error
     */
    protected int getLastOIModelEntityID() throws SQLException {
        Statement statement=m_connection.createStatement();
        try {
            ResultSet resultSet=statement.executeQuery("SELECT counter FROM PKCounter WHERE type='oimodelentity'");
            try {
                if (!resultSet.next()) {
                    executeUpdate("INSERT INTO PKCounter (type,counter) VALUES ('oimodelentity',0)");
                    return 0;
                }
                else
                    return resultSet.getInt(1);
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }

    /**
     * Information about entities.
     */
    protected static class EntityInfo {
        /** The ID of the entity. */
        public int m_entityID;
        /** True if the entity is a concept. */
        public boolean m_isConcept;
        /** True if the entity is a property. */
        public boolean m_isProperty;
        /** True if the entity is an instance. */
        public boolean m_isInstance;

        public EntityInfo(int entityID,boolean isConcept,boolean isProperty,boolean isInstance) {
            m_entityID=entityID;
            m_isConcept=isConcept;
            m_isProperty=isProperty;
            m_isInstance=isInstance;
        }
    }
}
