package edu.unika.aifb.kaon.engineeringserver.util;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collections;
import java.sql.Connection;
import java.sql.SQLException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import edu.unika.aifb.kaon.engineeringserver.client.*;

/**
 * A utility class for copying any model into a model in the direct engineering server.
 */
public class DirectOIModelReplicator extends OIModelProcessor {
    /** Phase of loading instances. */
    public static final int PHASE_LOAD_ENTITIES=1;
    /** Phase of copying instances. */
    public static final int PHASE_COPY_INSTANCES=2;
    /** Phase of copying concepts. */
    public static final int PHASE_COPY_CONCEPTS=3;
    /** Phase of copying properties. */
    public static final int PHASE_COPY_PROPERTIES=4;

    /** The source OI-model. */
    protected OIModel m_sourceOIModel;
    /** The target OI-model. */
    protected OIModel m_targetOIModel;
    /** The target KAON connection. */
    protected DirectKAONConnection m_targetKAONConnection;
    /** The model writer. */
    protected DirectOIModelWriter m_modelWriter;
    /** The map of source to target models. */
    protected Map m_sourceToTarget;

    /**
     * Creates an instance of this class.
     *
     * @param sourceOIModel                     the source OI-model
     * @param targetKAONConnection              the target KAON connection
     */
    public DirectOIModelReplicator(OIModel sourceOIModel,DirectKAONConnection targetKAONConnection) {
        m_sourceToTarget=new HashMap();
        m_sourceOIModel=sourceOIModel;
        m_targetKAONConnection=targetKAONConnection;
    }
    /**
     * Starts the replication of the source model to the target model
     *
     * @return                                  the target model
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    public OIModel doReplication() throws KAONException,InterruptedException {
        Connection connection=null;
        try {
            connection=m_targetKAONConnection.getDatabaseConnection();
            setUpModelMappings();
            m_modelWriter=new DirectOIModelWriter(connection);
            processInstances();
            processConcepts();
            processProperties();
            m_modelWriter.flush();
            return m_targetOIModel;
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
        finally {
            if (m_modelWriter!=null) {
                m_modelWriter.close();
                m_modelWriter=null;
            }
            if (connection!=null)
                m_targetKAONConnection.releaseDatabaseConnection(connection);
            m_sourceOIModel=null;
            m_targetOIModel=null;
            m_targetKAONConnection=null;
        }
    }
    /**
     * Establishes the map from source to target OI-models.
     *
     * @throws KAONException                    thrown it there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void setUpModelMappings() throws KAONException,InterruptedException {
        m_targetOIModel=createTargetOIModel(m_sourceOIModel);
        m_sourceToTarget.put(m_sourceOIModel,m_targetOIModel);
        Iterator iterator=OIModels.getAllIncludedOIModels(m_sourceOIModel).iterator();
        while (iterator.hasNext()) {
            OIModel sourceOIModel=(OIModel)iterator.next();
            try {
                m_targetKAONConnection.openOIModelLogical(sourceOIModel.getLogicalURI());
            }
            catch (KAONException modelShouldBeReplicated) {
                OIModel targetOIModel=createTargetOIModel(sourceOIModel);
                m_sourceToTarget.put(sourceOIModel,targetOIModel);
            }
            checkInterrupted();
        }
        iterator=m_sourceToTarget.keySet().iterator();
        while (iterator.hasNext()) {
            OIModel sourceOIModel=(OIModel)iterator.next();
            OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceOIModel);
            Iterator inner=sourceOIModel.getIncludedOIModels().iterator();
            while (inner.hasNext()) {
                OIModel sourceIncludedOIModel=(OIModel)inner.next();
                OIModel targetIncludedOIModel=m_targetKAONConnection.openOIModelLogical(sourceIncludedOIModel.getLogicalURI());
                if (!targetOIModel.getIncludedOIModels().contains(targetIncludedOIModel))
                    targetOIModel.applyChanges(Collections.singletonList(new AddIncludedOIModel(targetIncludedOIModel)));
                checkInterrupted();
            }
        }
    }
    /**
     * Creates a target model for given source OI-model.
     *
     * @param sourceOIModel                     the source OI-model
     * @return                                  the OI-model generated for given source OI-model
     * @throws KAONException                    thrown it there is an error
     */
    protected OIModel createTargetOIModel(OIModel sourceOIModel) throws KAONException {
        String logicalURI=sourceOIModel.getLogicalURI();
        OIModel targetOIModel=m_targetKAONConnection.createOIModel(null,logicalURI);
        Map attributes=sourceOIModel.getAttributes();
        Iterator keys=attributes.keySet().iterator();
        while (keys.hasNext()) {
            String key=(String)keys.next();
            String value=(String)attributes.get(key);
            targetOIModel.setAttribute(key,value);
        }
        targetOIModel.setAttribute("OIModel.replicatedFromPhysicalURI",sourceOIModel.getPhysicalURI());
        return targetOIModel;
    }
    /**
     * Processes all concepts.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processConcepts() throws KAONException,InterruptedException {
        Set concepts=m_sourceOIModel.getConcepts();
        copyConcepts(concepts);
        flushChanges();
    }
    /**
     * Processes all properties.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processProperties() throws KAONException,InterruptedException {
        Set properties=m_sourceOIModel.getProperties();
        copyProperties(properties);
        flushChanges();
    }
    /**
     * Processes all instances.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processInstances() throws KAONException,InterruptedException {
        Set instances=loadInstances();
        flushChanges();
        copyInstances(instances);
        flushChanges();
    }
    /**
     * Loads the instances. As a side-effect, this method creates all entities.
     *
     * @return                                  the set of instances
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected Set loadInstances() throws KAONException,InterruptedException {
        Set instances=m_sourceOIModel.getInstances();
        processElements(instances,m_sourceOIModel,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                try {
                    Iterator iterator=objects.iterator();
                    while (iterator.hasNext()) {
                        Instance instance=(Instance)iterator.next();
                        OIModel targetOIModel=(OIModel)m_sourceToTarget.get(instance.getSourceOIModel());
                        if (targetOIModel!=null) {
                            Concept concept=instance.getSpanningConcept();
                            Property property=instance.getSpanningProperty();
                            m_modelWriter.createEntity(targetOIModel.getLogicalURI(),instance.getURI(),concept.isInOIModel(),property.isInOIModel(),property.isAttribute(),property.isSymmetric(),property.isTransitive(),instance.isInOIModel());
                        }
                        checkInterrupted();
                    }
                    flushChanges();
                }
                catch (SQLException e) {
                    throw new KAONException("SQL error",e);
                }
            }
        },PHASE_LOAD_ENTITIES);
        return instances;
    }
    /**
     * Copies the information about the concepts.
     *
     * @param concepts                          the set of concepts
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyConcepts(Set concepts) throws KAONException,InterruptedException {
        processElements(concepts,m_sourceOIModel,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                try {
                    Iterator iterator=objects.iterator();
                    while (iterator.hasNext()) {
                        Concept concept=(Concept)iterator.next();
                        Iterator superConcepts=concept.getSuperConcepts().iterator();
                        while (superConcepts.hasNext()) {
                            Concept superConcept=(Concept)superConcepts.next();
                            OIModel targetOIModel=(OIModel)m_sourceToTarget.get(concept.getSuperSubConceptOIModel(superConcept));
                            if (targetOIModel!=null)
                                m_modelWriter.addConceptHierarchy(targetOIModel.getLogicalURI(),superConcept.getURI(),concept.getURI());
                            checkInterrupted();
                        }
                        checkInterrupted();
                    }
                    flushChanges();
                }
                catch (SQLException e) {
                    throw new KAONException("SQL error",e);
                }
            }
        },PHASE_COPY_CONCEPTS);
    }
    /**
     * Copies the information about the properties.
     *
     * @param properties                        the set of properties
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyProperties(Set properties) throws KAONException,InterruptedException {
        processElements(properties,m_sourceOIModel,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                try {
                    Iterator iterator=objects.iterator();
                    while (iterator.hasNext()) {
                        Property property=(Property)iterator.next();
                        OIModel targetOIModel=(OIModel)m_sourceToTarget.get(property.getInversePropertyOIModel());
                        if (targetOIModel!=null)
                            m_modelWriter.addInverseProperty(targetOIModel.getLogicalURI(),property.getURI(),property.getInverseProperty().getURI());
                        Iterator superProperties=property.getSuperProperties().iterator();
                        while (superProperties.hasNext()) {
                            Property superProperty=(Property)superProperties.next();
                            targetOIModel=(OIModel)m_sourceToTarget.get(property.getSuperSubPropertyOIModel(superProperty));
                            if (targetOIModel!=null)
                                m_modelWriter.addPropertyHierarchy(targetOIModel.getLogicalURI(),superProperty.getURI(),property.getURI());
                            checkInterrupted();
                        }
                        Iterator domainConcepts=property.getDomainConcepts().iterator();
                        while (domainConcepts.hasNext()) {
                            Concept domainConcept=(Concept)domainConcepts.next();
                            targetOIModel=(OIModel)m_sourceToTarget.get(property.getDomainConceptOIModel(domainConcept));
                            if (targetOIModel!=null)
                                m_modelWriter.addPropertyDomain(targetOIModel.getLogicalURI(),property.getURI(),domainConcept.getURI(),property.getMinimumCardinality(domainConcept),property.getMaximumCardinality(domainConcept));
                            checkInterrupted();
                        }
                        Iterator rangeConcepts=property.getRangeConcepts().iterator();
                        while (rangeConcepts.hasNext()) {
                            Concept rangeConcept=(Concept)rangeConcepts.next();
                            targetOIModel=(OIModel)m_sourceToTarget.get(property.getRangeConceptOIModel(rangeConcept));
                            if (targetOIModel!=null)
                                m_modelWriter.addPropertyRange(targetOIModel.getLogicalURI(),property.getURI(),rangeConcept.getURI());
                            checkInterrupted();
                        }
                        checkInterrupted();
                    }
                    flushChanges();
                }
                catch (SQLException e) {
                    throw new KAONException("SQL error",e);
                }
            }
        },PHASE_COPY_PROPERTIES);
    }
    /**
     * Copies the information about the instances.
     *
     * @param instances                         the set of instances
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyInstances(Set instances) throws KAONException,InterruptedException {
        processElements(instances,null,0,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                try {
                    Iterator iterator=objects.iterator();
                    while (iterator.hasNext()) {
                        Instance instance=(Instance)iterator.next();
                        Iterator parentConcepts=instance.getParentConcepts().iterator();
                        while (parentConcepts.hasNext()) {
                            Concept concept=(Concept)parentConcepts.next();
                            OIModel targetOIModel=(OIModel)m_sourceToTarget.get(instance.getConceptInstanceOIModel(concept));
                            if (targetOIModel!=null)
                                m_modelWriter.addConceptInstance(targetOIModel.getLogicalURI(),concept.getURI(),instance.getURI());
                            checkInterrupted();
                        }
                        Iterator propertyInstances=instance.getFromPropertyInstances().iterator();
                        while (propertyInstances.hasNext()) {
                            PropertyInstance propertyInstance=(PropertyInstance)propertyInstances.next();
                            OIModel targetOIModel=(OIModel)m_sourceToTarget.get(propertyInstance.getSourceOIModel());
                            if (targetOIModel!=null) {
                                Property property=propertyInstance.getProperty();
                                Instance sourceInstance=propertyInstance.getSourceInstance();
                                Object targetValue=propertyInstance.getTargetValue();
                                if (targetValue instanceof Instance)
                                    m_modelWriter.addRelationInstance(targetOIModel.getLogicalURI(),property.getURI(),sourceInstance.getURI(),((Instance)targetValue).getURI());
                                else
                                    m_modelWriter.addAttributeInstance(targetOIModel.getLogicalURI(),property.getURI(),sourceInstance.getURI(),(String)targetValue);
                            }
                            checkInterrupted();
                        }
                        checkInterrupted();
                    }
                    flushChanges();
                }
                catch (SQLException e) {
                    throw new KAONException("SQL error",e);
                }
            }
        },PHASE_COPY_INSTANCES);
    }
    /**
     * Flushes the changes to the OI-model.
     *
     * @throws KAONException                    thrown if there is an error
     */
    protected void flushChanges() throws KAONException {
        try {
            m_modelWriter.flush();
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
}
