package edu.unika.aifb.kaon.engineeringserver.query;

import java.util.List;
import java.util.ArrayList;

import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This static class contains a Datalog program that provides the standard axiomatization of the Engineering Server.
 * The consists of the following rules:
 * <p><code>
 * ConceptHierarchy_ti(X,Y) :- ConceptHierarchy_i(X,Y)<br>
 * ConceptHierarchy_ti(X,Y) :- ConceptHierarchy_i(X,Z),ConceptHierarchy_ti(Z,Y)<br>
 * <br>
 * PropertyHierarchy_ti(X,Y) :- PropertyHierarchy_i(X,Y)<br>
 * PropertyHierarchy_ti(X,Y) :- PropertyHierarchy_i(X,Z),PropertyHierarchy_ti(Z,Y)<br>
 * <br>
 * ConceptInstance_ai(C,I) :- ConceptInstance_i(C,I)<br>
 * ConceptInstance_ai(C,I) :- ConceptHierarchy_ti(C,SC),ConceptInstance_i(SC,I)<br>
 * <br>
 * AttributeInstance_ai(P,S,V) :- AttributeInstance_i(P,S,V)<br>
 * AttributeInstance_ai(P,S,V) :- PropertyHierarchy_ti(P,SP),AttributeInstance_i(SP,S,V)<br>
 * <br>
 * RelationInstance_ai(P,S,T) :- RelationInstance_i(P,S,T)<br>
 * RelationInstance_ai(P,S,T) :- PropertyHierarchy_t(P,SP),RelationInstance_ai(SP,S,T)<br>
 * RelationInstance_ai(P,S,T) :- OIModelEntity(URI,P,ATTR,SIM,TRANS),InverseProperty_i(P,IP),RelationInstance_ai(IP,T,S)<br>
 * RelationInstance_ai(P,S,T) :- OIModelEntity(URI,P,ATTR,true,TRANS),RelationInstance_i(P,T,S)<br>
 * RelationInstance_ai(P,S,T) :- OIModelEntity(URI,P,ATTR,SIM,true),RelationInstance_ai(P,S,M),RelationInstance_ai(P,M,T)<br>
 * <br>
 * ConceptHierarchy(XU,YU) :- ConceptHierarchy_i(X,Y),OIModelEntity(XU,X,ATTR1,SIM1,TRANS1),OIModelEntity(YU,Y,ATTR2,SIM2,TRANS2)<br>
 * PropertyHierarchy(XU,YU) :- PropertyHierarchy_t(X,Y),OIModelEntity(XU,X,ATTR1,SIM1,TRANS1),OIModelEntity(YU,Y,ATTR2,SIM2,TRANS2)<br>
 * ConceptInstance(CU,IU) :- ConceptInstance_i(C,I),OIModelEntity(CU,C,ATTR1,SIM1,TRANS1),OIModelEntity(IU,I,ATTR2,SIM2,TRANS2)<br>
 * AttributeInstance(PU,SU,V) :- AttributeInstance_i(P,S,V),OIModelEntity(PU,P,ATTR1,SIM1,TRANS1),OIModelEntity(SU,S,ATTR2,SIM2,TRANS2)<br>
 * RelationInstance(PU,SU,TU) :- RelationInstance_i(P,S,T),OIModelEntity(PU,P,ATTR1,SIM1,TRANS1),OIModelEntity(SU,S,ATTR2,SIM2,TRANS2),OIModelEntity(TU,T,ATTR3,SIM3,TRANS3)<br>
 * <br>
 * ConceptHierarchy_t(XU,YU) :- ConceptHierarchy_ti(X,Y),OIModelEntity(XU,X,ATTR1,SIM1,TRANS1),OIModelEntity(YU,Y,ATTR2,SIM2,TRANS2)<br>
 * PropertyHierarchy_t(XU,YU) :- PropertyHierarchy_ti(X,Y),OIModelEntity(XU,X,ATTR1,SIM1,TRANS1),OIModelEntity(YU,Y,ATTR2,SIM2,TRANS2)<br>
 * ConceptInstance_a(CU,IU) :- ConceptInstance_ai(C,I),OIModelEntity(CU,C,ATTR1,SIM1,TRANS1),OIModelEntity(IU,I,ATTR2,SIM2,TRANS2)<br>
 * AttributeInstance_a(PU,SU,V) :- AttributeInstance_ai(P,S,V),OIModelEntity(PU,P,ATTR1,SIM1,TRANS1),OIModelEntity(SU,S,ATTR2,SIM2,TRANS2)<br>
 * RelationInstance_a(PU,SU,TU) :- RelationInstance_ai(P,S,T),OIModelEntity(PU,P,ATTR1,SIM1,TRANS1),OIModelEntity(SU,S,ATTR2,SIM2,TRANS2),OIModelEntity(TU,T,ATTR3,SIM3,TRANS3)<br>
 * PropertyInstance_a(PU,SU,NULL,V) :- AttributeInstance_a(PU,SU,V)<br>
 * PropertyInstance_a(PU,SU,TU,NULL) :- RelationInstance_a(PU,SU,TU)<br>
 * </code></p>
 */
public class StandardAxiomatization {
    /** The predicate factory of the standard axiomatization. This object shouldn't be used to create new predicates. */
    public static final PredicateFactory PREDICATE_FACTORY;
    /** The program defining the standard axiomatization of the engineering server. */
    public static final Program PROGRAM;
    static {
        PredicateFactory predicateFactory=new PredicateFactory();
        List rules=new ArrayList();
        getAxiomatization(predicateFactory,rules);
        Rule[] rulesArray=new Rule[rules.size()];
        rules.toArray(rulesArray);
        PROGRAM=new Program(rulesArray);
        PREDICATE_FACTORY=new PredicateFactory(predicateFactory) {
            public Predicate getPredicate(String simpleName,int arity) {
                String fullName=simpleName+"/"+arity;
                Predicate predicate=(Predicate)m_predicatesByFullNames.get(fullName);
                if (predicate==null)
                    throw new IllegalArgumentException("Predicate with given name doesn't exist in the factory. This factory cannot create new predicates.");
                else
                    return predicate;
            }
        };
    }

    /**
     * Returns the axiomatization for given OI-model.
     *
     * @param predicateFactory                      the predicate factory to use
     * @param rules                                 the list receiving the rules
     */
    public static void getAxiomatization(PredicateFactory predicateFactory,List rules) {
        // EDB predicates
        Predicate oimodelEntity=predicateFactory.getPredicate("OIModelEntity",5);
        Predicate inverseProperty_i=predicateFactory.getPredicate("InverseProperty_i",2);
        Predicate conceptHierarchy_i=predicateFactory.getPredicate("ConceptHierarchy_i",2);
        Predicate propertyHierarchy_i=predicateFactory.getPredicate("PropertyHierarchy_i",2);
        Predicate conceptInstance_i=predicateFactory.getPredicate("ConceptInstance_i",2);
        Predicate attributeInstance_i=predicateFactory.getPredicate("AttributeInstance_i",3);
        Predicate relationInstance_i=predicateFactory.getPredicate("RelationInstance_i",3);
        // IDB predicates
        Predicate conceptHierarchy=predicateFactory.getPredicate("ConceptHierarchy",2);
        Predicate propertyHierarchy=predicateFactory.getPredicate("PropertyHierarchy",2);
        Predicate conceptInstance=predicateFactory.getPredicate("ConceptInstance",2);
        Predicate attributeInstance=predicateFactory.getPredicate("AttributeInstance",3);
        Predicate relationInstance=predicateFactory.getPredicate("RelationInstance",3);
        Predicate conceptHierarchy_ti=predicateFactory.getPredicate("ConceptHierarchy_ti",2);
        Predicate propertyHierarchy_ti=predicateFactory.getPredicate("PropertyHierarchy_ti",2);
        Predicate conceptInstance_ai=predicateFactory.getPredicate("ConceptInstance_ai",2);
        Predicate attributeInstance_ai=predicateFactory.getPredicate("AttributeInstance_ai",3);
        Predicate relationInstance_ai=predicateFactory.getPredicate("RelationInstance_ai",3);
        Predicate conceptHierarchy_t=predicateFactory.getPredicate("ConceptHierarchy_t",2);
        Predicate propertyHierarchy_t=predicateFactory.getPredicate("PropertyHierarchy_t",2);
        Predicate conceptInstance_a=predicateFactory.getPredicate("ConceptInstance_a",2);
        Predicate attributeInstance_a=predicateFactory.getPredicate("AttributeInstance_a",3);
        Predicate relationInstance_a=predicateFactory.getPredicate("RelationInstance_a",3);
        Predicate propertyInstance_a=predicateFactory.getPredicate("PropertyInstance_a",4);
        // utility variables
        Variable X=new Variable("X");
        Variable XU=new Variable("XU");
        Variable Y=new Variable("Y");
        Variable YU=new Variable("YU");
        Variable Z=new Variable("Z");
        Variable C=new Variable("C");
        Variable CU=new Variable("CU");
        Variable SC=new Variable("SC");
        Variable I=new Variable("I");
        Variable IU=new Variable("IU");
        Variable P=new Variable("P");
        Variable PU=new Variable("PU");
        Variable SP=new Variable("SP");
        Variable IP=new Variable("IP");
        Variable S=new Variable("S");
        Variable SU=new Variable("SU");
        Variable T=new Variable("T");
        Variable TU=new Variable("TU");
        Variable V=new Variable("V");
        Variable M=new Variable("M");
        // the program
        rules.add(new Rule("ch1",
            new Literal(conceptHierarchy_ti,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(conceptHierarchy_i,true,new Term[] { X,Y }),
            }
        ));
        rules.add(new Rule("ch2",
            new Literal(conceptHierarchy_ti,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(conceptHierarchy_i,true,new Term[] { X,Z }),
                new Literal(conceptHierarchy_ti,true,new Term[] { Z,Y }),
            }
        ));

        rules.add(new Rule("ph1",
            new Literal(propertyHierarchy_ti,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(propertyHierarchy_i,true,new Term[] { X,Y }),
            }
        ));
        rules.add(new Rule("ph2",
            new Literal(propertyHierarchy_ti,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(propertyHierarchy_i,true,new Term[] { X,Z }),
                new Literal(propertyHierarchy_ti,true,new Term[] { Z,Y }),
            }
        ));

        rules.add(new Rule("ci1",
            new Literal(conceptInstance_ai,true,new Term[] { C,I }),
            new Literal[] {
                new Literal(conceptInstance_i,true,new Term[] { C,I }),
            }
        ));
        rules.add(new Rule("ci2",
            new Literal(conceptInstance_ai,true,new Term[] { C,I }),
            new Literal[] {
                new Literal(conceptHierarchy_ti,true,new Term[] { C,SC }),
                new Literal(conceptInstance_i,true,new Term[] { SC,I }),
            }
        ));

        rules.add(new Rule("ai1",
            new Literal(attributeInstance_ai,true,new Term[] { P,S,V }),
            new Literal[] {
                new Literal(attributeInstance_i,true,new Term[] { P,S,V }),
            }
        ));
        rules.add(new Rule("ai2",
            new Literal(attributeInstance_ai,true,new Term[] { P,S,V }),
            new Literal[] {
                new Literal(propertyHierarchy_ti,true,new Term[] { P,SP }),
                new Literal(attributeInstance_i,true,new Term[] { SP,S,V }),
            }
        ));

        rules.add(new Rule("ri1",
            new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(relationInstance_i,true,new Term[] { P,S,T }),
            }
        ));
        rules.add(new Rule("ri2",
            new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(propertyHierarchy_ti,true,new Term[] { P,SP }),
                new Literal(relationInstance_ai,true,new Term[] { SP,S,T }),
            }
        ));
        rules.add(new Rule("ri3",
            new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(oimodelEntity,true,new Term[] { new Variable("_"),P,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(inverseProperty_i,true,new Term[] { P,IP }),
                new Literal(relationInstance_ai,true,new Term[] { IP,T,S }),
            }
        ));
        rules.add(new Rule("ri4",
            new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(oimodelEntity,true,new Term[] { new Variable("_"),P,new Variable("_"),new Constant(true),new Variable("_") }),
                new Literal(relationInstance_i,true,new Term[] { P,T,S }),
            }
        ));
        rules.add(new Rule("ri5",
            new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(oimodelEntity,true,new Term[] { new Variable("_"),P,new Variable("_"),new Variable("_"),new Constant(true) }),
                new Literal(relationInstance_ai,true,new Term[] { P,S,M }),
                new Literal(relationInstance_ai,true,new Term[] { P,M,T }),
            }
        ));

        rules.add(new Rule("chu",
            new Literal(conceptHierarchy_t,true,new Term[] { XU,YU }),
            new Literal[] {
                new Literal(conceptHierarchy_ti,true,new Term[] { X,Y }),
                new Literal(oimodelEntity,true,new Term[] { XU,X,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { YU,Y,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("phu",
            new Literal(propertyHierarchy_t,true,new Term[] { XU,YU }),
            new Literal[] {
                new Literal(propertyHierarchy_ti,true,new Term[] { X,Y }),
                new Literal(oimodelEntity,true,new Term[] { XU,X,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { YU,Y,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("ciu",
            new Literal(conceptInstance_a,true,new Term[] { CU,IU }),
            new Literal[] {
                new Literal(conceptInstance_ai,true,new Term[] { C,I }),
                new Literal(oimodelEntity,true,new Term[] { CU,C,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { IU,I,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("aiu",
            new Literal(attributeInstance_a,true,new Term[] { PU,SU,V }),
            new Literal[] {
                new Literal(attributeInstance_ai,true,new Term[] { P,S,V }),
                new Literal(oimodelEntity,true,new Term[] { PU,P,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { SU,S,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("riu",
            new Literal(relationInstance_a,true,new Term[] { PU,SU,TU }),
            new Literal[] {
                new Literal(relationInstance_ai,true,new Term[] { P,S,T }),
                new Literal(oimodelEntity,true,new Term[] { PU,P,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { SU,S,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { TU,T,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("piu1",
            new Literal(propertyInstance_a,true,new Term[] { PU,SU,Constant.NULL,V }),
            new Literal[] {
                new Literal(attributeInstance_a,true,new Term[] { PU,SU,V }),
            }
        ));
        rules.add(new Rule("piu2",
            new Literal(propertyInstance_a,true,new Term[] { PU,SU,TU,Constant.NULL }),
            new Literal[] {
                new Literal(relationInstance_a,true,new Term[] { PU,SU,TU }),
            }
        ));

        rules.add(new Rule("chuo",
            new Literal(conceptHierarchy,true,new Term[] { XU,YU }),
            new Literal[] {
                new Literal(conceptHierarchy_i,true,new Term[] { X,Y }),
                new Literal(oimodelEntity,true,new Term[] { XU,X,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { YU,Y,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("phuo",
            new Literal(propertyHierarchy,true,new Term[] { XU,YU }),
            new Literal[] {
                new Literal(propertyHierarchy_i,true,new Term[] { X,Y }),
                new Literal(oimodelEntity,true,new Term[] { XU,X,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { YU,Y,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("ciuo",
            new Literal(conceptInstance,true,new Term[] { CU,IU }),
            new Literal[] {
                new Literal(conceptInstance_i,true,new Term[] { C,I }),
                new Literal(oimodelEntity,true,new Term[] { CU,C,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { IU,I,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("aiuo",
            new Literal(attributeInstance,true,new Term[] { PU,SU,V }),
            new Literal[] {
                new Literal(attributeInstance_i,true,new Term[] { P,S,V }),
                new Literal(oimodelEntity,true,new Term[] { PU,P,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { SU,S,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
        rules.add(new Rule("riuo",
            new Literal(relationInstance,true,new Term[] { PU,SU,TU }),
            new Literal[] {
                new Literal(relationInstance_i,true,new Term[] { P,S,T }),
                new Literal(oimodelEntity,true,new Term[] { PU,P,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { SU,S,new Variable("_"),new Variable("_"),new Variable("_") }),
                new Literal(oimodelEntity,true,new Term[] { TU,T,new Variable("_"),new Variable("_"),new Variable("_") }),
            }
        ));
    }
}
