package edu.unika.aifb.kaon.engineeringserver.query;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import edu.unika.aifb.kaon.datalog.jdbc.*;

/**
 * The manager taking care of the type names and type conversions.
 */
public class ServerTypeManager extends TypeManager {
    /** The type of entity URI. */
    public static final int TYPE_ENTITY_URI=20000;

    /** The instance of this type manager. */
    public static final ServerTypeManager INSTANCE=new ServerTypeManager();

    /**
     * Sets the value of the object to the prepared statement.
     *
     * @param type                              the type
     * @param object                            the object being set
     * @param preparedStatement                 the prepared statement being set
     * @param index                             the index
     * @throws SQLException                     thrown if there is a problem
     */
    public void setObjectToStatement(int type,Object object,PreparedStatement preparedStatement,int index) throws SQLException {
        if (object==null) {
            if (type==TYPE_ENTITY_URI)
                type=Types.VARCHAR;
            preparedStatement.setNull(index,type);
        }
        else
            switch (type) {
            case TYPE_ENTITY_URI:
                preparedStatement.setString(index,((EntityURI)object).getEntityURI());
                break;
            case Types.INTEGER:
                preparedStatement.setInt(index,((Integer)object).intValue());
                break;
            case Types.BOOLEAN:
                preparedStatement.setBoolean(index,((Boolean)object).booleanValue());
                break;
            case Types.VARCHAR:
            case TYPE_REGULAR_EXPRESSION:
            case TYPE_LIKE:
            default:
                preparedStatement.setString(index,object.toString());
                break;
            }
    }
    /**
     * Returns the object from the result set.
     *
     * @param type                              the type
     * @param resultSet                         the result set
     * @param index                             the index
     * @return                                  the object
     * @throws SQLException                     thrown if there is a problem
     */
    public Object getObjectFromResultSet(int type,ResultSet resultSet,int index) throws SQLException {
        switch (type) {
        case TYPE_ENTITY_URI:
            return new EntityURI(resultSet.getString(index));
        case Types.BOOLEAN:
            return resultSet.getBoolean(index) ? Boolean.TRUE : Boolean.FALSE;
        case Types.INTEGER:
            return new Integer(resultSet.getInt(index));
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
        default:
            return resultSet.getString(index);
        }
    }
    /**
     * Returns the character representation of the type.
     *
     * @param type                              the type
     * @return                                  the character representation of the type
     */
    public String typeToString(int type) {
        switch (type) {
        case TYPE_ENTITY_URI:
            return "u";
        case Types.INTEGER:
            return "i";
        case Types.BOOLEAN:
            return "b";
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
        default:
            return "t";
        }
    }
    /**
     * Returns the SQL name of the type.
     *
     * @param type                              the type
     * @return                                  the SQL name of the type
     */
    public String typeToSQLName(int type) {
        switch (type) {
        case TYPE_ENTITY_URI:
            return "varchar(255)";
        case Types.INTEGER:
            return "int";
        case Types.BOOLEAN:
            return "smallint";
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
        default:
            return "varchar(255)";
        }
    }
}