package edu.unika.aifb.kaon.engineeringserver.loader;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Object loader for Oracle8i. From 9i version onwards Oracle supports the SQL2 sytax, so the ObjectLoaderSQL2 is used.
 */
public class ObjectLoaderOracle8i extends AbstractSQLObjectLoader {
    /** The IN condition restricting the model ID. */
    protected String m_modelIDRestriction;

    /**
     * Creates an instance of this class and attaches it to given connection.
     *
     * @param connection                    connection from which loading is performed
     * @param modelID                       the ID of the model for which loading is performed
     * @throws KAONException                thrown if there is an error
     */
    public ObjectLoaderOracle8i(Connection connection,int modelID) throws KAONException {
        super(connection,modelID);
        try {
            StringBuffer buffer=new StringBuffer();
            boolean first=true;
            Statement statement=m_connection.createStatement();
            try {
                ResultSet resultSet=statement.executeQuery("SELECT includedModelID FROM AllIncludedOIModels WHERE includingModelID="+m_modelID);
                try {
                    while (resultSet.next()) {
                        int includedModelID=resultSet.getInt(1);
                        if (first)
                            first=false;
                        else
                            buffer.append(",");
                        buffer.append(includedModelID);
                    }
                }
                finally {
                    resultSet.close();
                }
            }
            finally {
                statement.close();
            }
            m_modelIDRestriction=buffer.toString();
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    protected String getLoadBasicsQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ o.entityURI,o.modelID,o.conceptVersion,o.instanceVersion,o.propertyVersion,o.isAttribute,o.isSymmetric,o.isTransitive,i.entityURI,i.propertyVersion,ip.modelID "+
                      "FROM OIModelEntity o,(SELECT /*+ INDEX(InverseProperty InverseProperty_1) */ * FROM InverseProperty WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ip,OIModelEntity i "+
                      "WHERE o.entityID=ip.propertyID(+) AND ip.inversePropertyID=i.entityID(+) AND o.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(") AND o.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadSuperConceptsQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 1 AS rowType,ch.modelID AS modelID,c.entityURI AS objectURI,c.conceptVersion AS objectVersion,sc.entityURI AS superConceptURI,sc.conceptVersion AS superConceptVersion,0,0 "+
                      "FROM OIModelEntity c,(SELECT /*+ INDEX(ConceptHierarchy ConceptHierarchy_1) */ * FROM ConceptHierarchy WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ch,OIModelEntity sc "+
                      "WHERE c.entityID=ch.subConceptID(+) AND ch.superConceptID=sc.entityID(+) AND c.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND c.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadSubConceptsQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 2 AS rowType,ch.modelID AS modelID,c.entityURI AS objectURI,c.conceptVersion AS objectVersion,sc.entityURI AS subConceptURI,sc.conceptVersion AS subConceptVersion,0,0 "+
                      "FROM OIModelEntity c,(SELECT /*+ INDEX(ConceptHierarchy ConceptHierarchy_2) */ * FROM ConceptHierarchy WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ch,OIModelEntity sc "+
                      "WHERE c.entityID=ch.superConceptID(+) AND ch.subConceptID=sc.entityID(+) AND c.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND c.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertiesFromQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 3 AS rowType,pd.modelID AS modelID,c.entityURI AS objectURI,c.propertyVersion AS objectVersion,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,0,0 "+
                      "FROM OIModelEntity c,(SELECT /*+ INDEX(PropertyDomain PropertyDomain_1) */ * FROM PropertyDomain WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) pd,OIModelEntity p "+
                      "WHERE c.entityID=pd.conceptID(+) AND pd.propertyID=p.entityID(+) AND c.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND c.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertiesToQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 4 AS rowType,pr.modelID AS modelID,c.entityURI AS objectURI,c.propertyVersion AS objectVersion,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,0,0 "+
                      "FROM OIModelEntity c,(SELECT /*+ INDEX(PropertyRange PropertyRange_2) */ * FROM PropertyRange WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) pr,OIModelEntity p "+
                      "WHERE c.entityID=pr.conceptID(+) AND pr.propertyID=p.entityID(+) AND c.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND c.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadConceptInstancesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 5 AS rowType,ci.modelID AS modelID,c.entityURI AS objectURI,c.conceptVersion AS objectVersion,i.entityURI AS instanceURI,i.instanceVersion AS instanceVersion,0,0 "+
                      "FROM OIModelEntity c,(SELECT /*+ INDEX(ConceptInstance ConceptInstance_1) */ * FROM ConceptInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ci,OIModelEntity i "+
                      "WHERE c.entityID=ci.conceptID(+) AND ci.instanceID=i.entityID(+) AND c.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND c.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadSuperPropertiesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 6 AS rowType,ph.modelID AS modelID,p.entityURI AS objectURI,p.propertyVersion AS objectVersion,sp.entityURI AS superPropertyURI,sp.propertyVersion AS superPropertyVersion,0,0 "+
                      "FROM OIModelEntity p ,(SELECT /*+ INDEX(PropertyHierarchy PropHierarchy_1) */ * FROM PropertyHierarchy WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ph,OIModelEntity sp "+
                      "WHERE p.entityID=ph.subPropertyID(+) AND ph.superPropertyID=sp.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadSubPropertiesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 7 AS rowType,ph.modelID AS modelID,p.entityURI AS objectURI,p.propertyVersion AS objectVersion,sp.entityURI AS subPropertyURI,sp.propertyVersion AS subPropertyVersion,0,0 "+
                      "FROM OIModelEntity p,(SELECT /*+ INDEX(PropertyHierarchy PropHierarchy_2) */ * FROM PropertyHierarchy WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ph,OIModelEntity sp "+
                      "WHERE p.entityID=ph.superPropertyID(+) AND ph.subPropertyID=sp.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertyDomainsQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 8 AS rowType,pd.modelID AS modelID,p.entityURI AS objectURI,p.propertyVersion AS objectVersion,c.entityURI AS conceptURI,c.conceptVersion AS conceptVersion,pd.minimumCardinality,pd.maximumCardinality "+
                      "FROM OIModelEntity p,(SELECT /*+ INDEX(PropertyDomain PropertyDomain_1) */ * FROM PropertyDomain WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) pd,OIModelEntity c "+
                      "WHERE p.entityID=pd.propertyID(+) AND pd.conceptID=c.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertyRangesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 9 AS rowType,pr.modelID AS modelID,p.entityURI AS objectURI,p.propertyVersion AS objectVersion,c.entityURI AS conceptURI,c.conceptVersion AS conceptVersion,0,0 "+
                      "FROM OIModelEntity p,(SELECT /*+ INDEX(PropertyRange PropertyRange_1) */ * FROM PropertyRange WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) pr,OIModelEntity c "+
                      "WHERE p.entityID=pr.propertyID(+) AND pr.conceptID=c.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadParentConceptsQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 10 AS rowType,ci.modelID AS modelID,i.entityURI AS objectURI,i.instanceVersion AS objectVersion,c.entityURI AS conceptURI,c.conceptVersion AS conceptVersion,0,0 "+
                      "FROM OIModelEntity i,(SELECT /*+ INDEX(ConceptInstance ConceptInstance_2) */ * FROM ConceptInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ci,OIModelEntity c "+
                      "WHERE i.entityID=ci.instanceID(+) AND ci.conceptID=c.entityID(+) AND i.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND i.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadInstancePropertiesFromQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 11 AS rowType,ri.modelID AS modelID,si.entityURI AS objectURI,si.instanceVersion AS objectVersion,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,ti.entityURI AS targetInstanceURI,ti.instanceVersion AS targetInstanceVersion,'' "+
                	  "FROM OIModelEntity si,(SELECT /*+ INDEX(RelationInstance RelInstance_2) */ * FROM RelationInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ri,OIModelEntity p,OIModelEntity ti "+
                      "WHERE si.entityID=ri.sourceInstanceID(+) AND ri.propertyID=p.entityID(+) AND ri.targetInstanceID=ti.entityID(+) AND si.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND si.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(") UNION ALL SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 11 AS rowType,ai.modelID AS modelID,si.entityURI AS objectURI,si.instanceVersion AS objectVersion,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,'',0,ai.textValue AS textValue "+
                	  "FROM OIModelEntity si,(SELECT /*+ INDEX(AttributeInstance AttrInstance_2) */ * FROM AttributeInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ai,OIModelEntity p "+
                      "WHERE si.entityID=ai.sourceInstanceID(+) AND ai.propertyID=p.entityID(+) AND si.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND si.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadInstancePropertiesToQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ 12 AS rowType,ri.modelID AS modelID,ti.entityURI AS objectURI,ti.instanceVersion AS objectVersion,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,si.entityURI AS sourceInstanceURI,si.instanceVersion AS sourceInstanceVersion,'' "+
                	  "FROM OIModelEntity ti,(SELECT /*+ INDEX(RelationInstance RelInstance_3) */ * FROM RelationInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ri,OIModelEntity p,OIModelEntity si "+
                      "WHERE ti.entityID=ri.targetInstanceID(+) AND ri.propertyID=p.entityID(+) AND ri.sourceInstanceID=si.entityID(+) AND ti.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND ti.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertyInstancesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        buffer.append("(SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ ri.modelID AS modelID,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,si.entityURI AS sourceInstanceURI,si.instanceVersion AS sourceInstanceVersion,ti.entityURI AS targetInstanceURI,ti.instanceVersion AS targetInstanceVersion,'' "+
                	  "FROM OIModelEntity p,(SELECT /*+ INDEX(RelationInstance RelInstance_1) */ * FROM RelationInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ri,OIModelEntity si,OIModelEntity ti "+
                      "WHERE p.entityID=ri.propertyID(+) AND ri.sourceInstanceID=si.entityID(+) AND ri.targetInstanceID=ti.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(") UNION ALL SELECT /*+ ORDERED INDEX(OIModelEntity OIModelEntity_1) */ ai.modelID AS modelID,p.entityURI AS propertyURI,p.propertyVersion AS propertyVersion,si.entityURI AS sourceInstanceURI,si.instanceVersion AS sourceInstanceVersion,'',0,textValue AS textValue "+
                	  "FROM OIModelEntity p,(SELECT /*+ INDEX(AttributeInstance AttrInstance_1) */ * FROM AttributeInstance WHERE modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ai,OIModelEntity si "+
                      "WHERE p.entityID=ai.propertyID(+) AND ai.sourceInstanceID=si.entityID(+) AND p.entityURI IN (");
        buffer.append(entityURIs);
        buffer.append(") AND p.modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")) ORDER BY propertyURI");
        return buffer.toString();
    }
    protected String getLoadRelatedEntitiesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        boolean first=true;
        buffer.append("(");
        if ((loadFlag & OIModel.LOAD_SUPER_CONCEPTS)!=0)
            first=appendQueryPart(first,getLoadSuperConceptsQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_SUB_CONCEPTS)!=0)
            first=appendQueryPart(first,getLoadSubConceptsQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_PROPERTIES_FROM)!=0)
            first=appendQueryPart(first,getLoadPropertiesFromQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_PROPERTIES_TO)!=0)
            first=appendQueryPart(first,getLoadPropertiesToQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_CONCEPT_INSTANCES)!=0)
            first=appendQueryPart(first,getLoadConceptInstancesQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_SUPER_PROPERTIES)!=0)
            first=appendQueryPart(first,getLoadSuperPropertiesQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_SUB_PROPERTIES)!=0)
            first=appendQueryPart(first,getLoadSubPropertiesQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_PROPERTY_DOMAINS)!=0)
            first=appendQueryPart(first,getLoadPropertyDomainsQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_PROPERTY_RANGES)!=0)
            first=appendQueryPart(first,getLoadPropertyRangesQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_INSTANCE_PARENT_CONCEPTS)!=0)
            first=appendQueryPart(first,getLoadParentConceptsQuery(entityURIs,loadFlag),buffer);
        buffer.append(") ORDER BY rowType,objectURI");
        return buffer.toString();
    }
    protected String getLoadInstancePropertyValuesQuery(String entityURIs,int loadFlag) {
        StringBuffer buffer=new StringBuffer();
        boolean first=true;
        buffer.append("(");
        if ((loadFlag & OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES)!=0)
            first=appendQueryPart(first,getLoadInstancePropertiesFromQuery(entityURIs,loadFlag),buffer);
        if ((loadFlag & OIModel.LOAD_INSTANCE_TO_PROPERTY_VALUES)!=0)
            first=appendQueryPart(first,getLoadInstancePropertiesToQuery(entityURIs,loadFlag),buffer);
        buffer.append(") ORDER BY rowType,objectURI");
        return buffer.toString();
    }
    protected boolean appendQueryPart(boolean first,String query,StringBuffer buffer) {
        if (!first)
            buffer.append(" UNION ALL ");
        buffer.append(" ( ");
        buffer.append(query);
        buffer.append(" ) ");
        return false;
    }
    protected String getLoadConceptsQuery() {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT entityURI,conceptVersion FROM OIModelEntity WHERE conceptVersion<>"+Integer.MAX_VALUE+" AND modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadPropertiesQuery() {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT entityURI,propertyVersion FROM OIModelEntity WHERE propertyVersion<>"+Integer.MAX_VALUE+" AND modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    protected String getLoadInstancesQuery() {
        StringBuffer buffer=new StringBuffer();
        buffer.append("SELECT entityURI,instanceVersion FROM OIModelEntity WHERE instanceVersion<>"+Integer.MAX_VALUE+" AND modelID IN (");
        buffer.append(m_modelIDRestriction);
        buffer.append(")");
        return buffer.toString();
    }
    /**
     * Returns <code>true</code> if the connection is to an Oracle database.
     *
     * @param connection                        the connection
     * @return                                  <code>true</code> if this is Oracle
     */
    public static boolean isOracle8i(Connection connection) {
        try {
            DatabaseMetaData databaseMetaData=connection.getMetaData();
            return "Oracle".equalsIgnoreCase(databaseMetaData.getDatabaseProductName()) && databaseMetaData.getDatabaseProductVersion().startsWith("Oracle8");
        }
        catch (Throwable ignored) {
            // Oracle shouldn't throw an error here
            return false;
        }
    }
}
