package edu.unika.aifb.kaon.engineeringserver.loader;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Abstract base class for all loaders. Subclasses are implementations for different databases.
 */
public abstract class AbstractSQLObjectLoader extends AbstractObjectLoader {
    /** All flags that will trigger the execution of the query for loading basic information. */
    protected static final int LOAD_BASICS_FLAGS=OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS;
    /** All flags that will trigger the execution of the query for loading related entities. */
    protected static final int LOAD_RELATED_ENTITIES_FLAGS=OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_SUB_CONCEPTS | OIModel.LOAD_PROPERTIES_FROM | OIModel.LOAD_PROPERTIES_TO | OIModel.LOAD_CONCEPT_INSTANCES | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_SUB_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS;
    /** All flags that will trigger the execution of the query for loading property values. */
    protected static final int LOAD_PROPERTY_VALUES_FLAGS=OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES | OIModel.LOAD_INSTANCE_TO_PROPERTY_VALUES;

    /** Connection from which objects are loaded. */
    protected Connection m_connection;

    /**
     * Creates an instance of this class and attaches it to given connection.
     *
     * @param connection                    connection from which loading is performed
     * @param modelID                       the ID of the model for which loading is performed
     */
    public AbstractSQLObjectLoader(Connection connection,int modelID) {
        super(modelID);
        m_connection=connection;
    }
    /**
     * Returns the list of IDs of concepts that are in the model.
     *
     * @return                              the array of IDs of concepts that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] loadConcepts() throws KAONException {
        String query=getLoadConceptsQuery();
        return loadIDs(query);
    }
    /**
     * Returns the list of IDs of properties that are in the model.
     *
     * @return                              the array of IDs of properties that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] loadProperties() throws KAONException {
        String query=getLoadPropertiesQuery();
        return loadIDs(query);
    }
    /**
     * Returns the list of IDs of instances that are in the model.
     *
     * @return                              the array of IDs of instances that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] loadInstances() throws KAONException {
        String query=getLoadInstancesQuery();
        return loadIDs(query);
    }
    /**
     * Executes a query that returns a list of IDs as the result.
     *
     * @param query                         the query to execute
     * @return                              the list of IDs returned by the query
     * @throws KAONException                thrown if there is an error
     */
    protected EntityID[] loadIDs(String query) throws KAONException {
        try {
            List list=new ArrayList();
            Statement statement=m_connection.createStatement();
            try {
                ResultSet resultSet=statement.executeQuery(query);
                try {
                    while (resultSet.next()) {
                        String entityURI=resultSet.getString(1);
                        int version=resultSet.getInt(2);
                        list.add(new EntityID(entityURI,version));
                    }
                }
                finally {
                    resultSet.close();
                }
            }
            finally {
                statement.close();
            }
            EntityID[] vector=new EntityID[list.size()];
            list.toArray(vector);
            return vector;
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Returns the URIs from the supplied array formatted as string and suitable for embedding into IN clause.
     *
     * @param entityURIs                    the array of URIs
     * @param start                         the start index (inclusive)
     * @param end                           the end index (exclusive)
     * @return                              the URIs formatted for embedding into IN clause
     */
    protected String getEntityURIsAsString(String[] entityURIs,int start,int end) {
        StringBuffer buffer=new StringBuffer();
        for (int i=start;i<end;i++) {
            if (i!=start)
                buffer.append(",");
            buffer.append("'");
            buffer.append(entityURIs[i]);
            buffer.append("'");
        }
        return buffer.toString();
    }
    /**
     * Called to load information about specified entities.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @return                              information about loaded entities
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityInfo[] loadEntities(String[] entityURIs,int loadFlag) throws KAONException {
        try {
            if (entityURIs.length==0)
                return new EntityInfo[0];
            try {
                if (entityURIs.length<getMaximumURIsToLoadUsingINLiterals())
                    loadEntitiesInClause(entityURIs,loadFlag);
                else
                    loadEntitiesURIJoinTable(entityURIs,loadFlag);
                return getLoadedObjectsArray();
            }
            finally {
                m_conceptInfos.clear();
                m_propertyInfos.clear();
                m_instanceInfos.clear();
            }
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Called to load information about specified entities by using the IN clause.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @throws SQLException                 thrown if there is a problem with fetching information
     */
    protected void loadEntitiesInClause(String[] entityURIs,int loadFlag) throws SQLException {
        int queryMaximumInLiterals=getQueryMaximumInLiterals();
        for (int start=0;start<entityURIs.length;start+=queryMaximumInLiterals) {
            int end=Math.min(start+queryMaximumInLiterals,entityURIs.length);
            String entityURIsAsString=getEntityURIsAsString(entityURIs,start,end);
            if ((loadFlag & LOAD_BASICS_FLAGS)!=0)
                loadBasics(entityURIsAsString,loadFlag);
            if ((loadFlag & LOAD_RELATED_ENTITIES_FLAGS)!=0)
                loadRelatedEntities(entityURIsAsString,loadFlag);
            if ((loadFlag & LOAD_PROPERTY_VALUES_FLAGS)!=0)
                loadInstancePropertyValues(entityURIsAsString,loadFlag);
            if ((loadFlag & OIModel.LOAD_PROPERTY_INSTANCES)!=0)
                loadPropertyInstances(entityURIsAsString,loadFlag);
        }
    }
    /**
     * Called to load information about specified entities by using the URI join table.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @throws SQLException                 thrown if there is a problem with fetching information
     */
    protected void loadEntitiesURIJoinTable(String[] entityURIs,int loadFlag) throws SQLException {
        long joinID=insertEntityURIs(entityURIs);
        try {
            String entityURIsCondition="SELECT entityURI FROM URIJoinTable WHERE joinID="+joinID;
            if ((loadFlag & LOAD_BASICS_FLAGS)!=0)
                loadBasics(entityURIsCondition,loadFlag);
            if ((loadFlag & LOAD_RELATED_ENTITIES_FLAGS)!=0)
                loadRelatedEntities(entityURIsCondition,loadFlag);
            if ((loadFlag & LOAD_PROPERTY_VALUES_FLAGS)!=0)
                loadInstancePropertyValues(entityURIsCondition,loadFlag);
            if ((loadFlag & OIModel.LOAD_PROPERTY_INSTANCES)!=0)
                loadPropertyInstances(entityURIsCondition,loadFlag);
        }
        finally {
            deleteEntityURIs(joinID);
        }
    }
    /**
     * Inserts elements into the URI join table.
     *
     * @param entityURIs                    the entity URIs
     * @return                              the join ID for the URIs
     * @throws SQLException                 thrown if there is an error
     */
    protected long insertEntityURIs(String[] entityURIs) throws SQLException {
        long joinID=System.currentTimeMillis();
        PreparedStatement insertURI=m_connection.prepareStatement("INSERT INTO URIJoinTable (joinID,entityURI) VALUES (?,?)");
        try {
            for (int i=0;i<entityURIs.length;i++) {
                insertURI.setLong(1,joinID);
                insertURI.setString(2,entityURIs[i]);
                insertURI.addBatch();
            }
            insertURI.executeBatch();
        }
        finally {
            insertURI.close();
        }
        return joinID;
    }
    /**
     * Deletes the elements from the URI join table.
     *
     * @param joinID                        the join ID
     * @throws SQLException                 thrown if there is an error
     */
    protected void deleteEntityURIs(long joinID) throws SQLException {
        Statement deleteURIs=m_connection.createStatement();
        try {
            deleteURIs.executeUpdate("DELETE FROM URIJoinTable WHERE joinID="+joinID);
        }
        finally {
            deleteURIs.close();
        }
    }
    /**
     * Loads basics of objects.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @throws SQLException                 thrown if there is an error
     */
    protected void loadBasics(String entityURIs,int loadFlag) throws SQLException {
        String query=getLoadBasicsQuery(entityURIs,loadFlag);
        Statement statement=m_connection.createStatement();
        try {
            ResultSet resultSet=statement.executeQuery(query);
            try {
                while (resultSet.next()) {
                    String entityURI=resultSet.getString(1);
                    int modelID=resultSet.getInt(2);
                    if ((loadFlag & OIModel.LOAD_CONCEPT_BASICS)!=0) {
                        ConceptInfo conceptInfo=getConceptInfo(entityURI,resultSet.getInt(3));
                        conceptInfo.m_modelID=modelID;
                    }
                    if ((loadFlag & OIModel.LOAD_INSTANCE_BASICS)!=0) {
                        InstanceInfo instanceInfo=getInstanceInfo(entityURI,resultSet.getInt(4));
                        instanceInfo.m_modelID=modelID;
                    }
                    if ((loadFlag & OIModel.LOAD_PROPERTY_BASICS)!=0) {
                        PropertyInfo propertyInfo=getPropertyInfo(entityURI,resultSet.getInt(5));
                        propertyInfo.m_isAttribute=resultSet.getInt(6)!=0;
                        propertyInfo.m_symmetric=resultSet.getInt(7)!=0;
                        propertyInfo.m_transitive=resultSet.getInt(8)!=0;
                        propertyInfo.m_modelID=modelID;
                        String inversePropertyURI=resultSet.getString(9);
                        if (!resultSet.wasNull())
                            propertyInfo.m_inversePropertyID=new RelationEntityID(inversePropertyURI,resultSet.getInt(10),resultSet.getInt(11));
                    }
                }
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }
    /**
     * Executes the query for loading entities related to entities.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @throws SQLException                 thrown if there is an error
     */
    protected void loadRelatedEntities(String entityURIs,int loadFlag) throws SQLException {
        String query=getLoadRelatedEntitiesQuery(entityURIs,loadFlag);
        Statement statement=m_connection.createStatement();
        try {
            ResultSet resultSet=statement.executeQuery(query);
            try {
                List objects=new ArrayList();
                int lastRowType=-1;
                String lastEntityURI=null;
                int lastVersion=0;
                while (resultSet.next()) {
                    int rowType=resultSet.getInt(1);
                    int modelID=resultSet.getInt(2);
                    String entityURI=resultSet.getString(3);
                    int version=resultSet.getInt(4);
                    if (lastRowType!=rowType || !lastEntityURI.equals(entityURI)) {
                        if (lastEntityURI!=null)
                            consumeRelatedEntities(lastEntityURI,lastVersion,lastRowType,objects);
                        objects.clear();
                        lastRowType=rowType;
                        lastEntityURI=entityURI;
                        lastVersion=version;
                    }
                    String relatedEntityURI=resultSet.getString(5);
                    if (!resultSet.wasNull()) {
                        int relatedVersion=resultSet.getInt(6);
                        if (rowType==8) {
                            int minimumCardinality=resultSet.getInt(7);
                            int maximumCardinality=resultSet.getInt(8);
                            objects.add(new DomainInfo(relatedEntityURI,relatedVersion,modelID,minimumCardinality,maximumCardinality));
                        }
                        else if (rowType==1 || rowType==6 || rowType==9 || rowType==10)
                            objects.add(new RelationEntityID(relatedEntityURI,relatedVersion,modelID));
                        else
                            objects.add(new EntityID(relatedEntityURI,relatedVersion));
                    }
                }
                if (lastEntityURI!=null)
                    consumeRelatedEntities(lastEntityURI,lastVersion,lastRowType,objects);
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }
    /**
     * Loads the property values for instances.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @throws SQLException                 thrown if there is an error
     */
    protected void loadInstancePropertyValues(String entityURIs,int loadFlag) throws SQLException {
        String query=getLoadInstancePropertyValuesQuery(entityURIs,loadFlag);
        Statement statement=m_connection.createStatement();
        try {
            ResultSet resultSet=statement.executeQuery(query);
            try {
                Map values=new HashMap();
                int lastRowType=-1;
                String lastEntityURI=null;
                int lastVersion=0;
                EntityID propertyID=null;
                while (resultSet.next()) {
                    int rowType=resultSet.getInt(1);
                    int modelID=resultSet.getInt(2);
                    String entityURI=resultSet.getString(3);
                    int version=resultSet.getInt(4);
                    if (lastRowType!=rowType || !lastEntityURI.equals(entityURI)) {
                        if (lastEntityURI!=null)
                            consumeInstancePropertyValues(lastEntityURI,lastVersion,lastRowType,values);
                        values.clear();
                        lastRowType=rowType;
                        lastEntityURI=entityURI;
                        lastVersion=version;
                        propertyID=null;
                    }
                    String propertyURI=resultSet.getString(5);
                    if (!resultSet.wasNull()) {
                        int propertyVersion=resultSet.getInt(6);
                        if (propertyID==null || !propertyID.m_uri.equals(propertyURI) || propertyID.m_version!=propertyVersion)
                            propertyID=new EntityID(propertyURI,propertyVersion);
                        Object value;
                        String relatedEntityURI=resultSet.getString(7);
                        int relatedEntityVersion=resultSet.getInt(8);
                        if (relatedEntityVersion!=0)
                            value=new RelationEntityID(relatedEntityURI,relatedEntityVersion,modelID);
                        else {
                            String textValue=resultSet.getString(9);
                            value=new PropertyValue(textValue,modelID);
                        }
                        addPropertyInstanceValue(propertyID,value,values);
                    }
                }
                if (lastEntityURI!=null)
                    consumeInstancePropertyValues(lastEntityURI,lastVersion,lastRowType,values);
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }
    /**
     * Loads property instances.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @throws SQLException                 thrown if there is an error
     */
    protected void loadPropertyInstances(String entityURIs,int loadFlag) throws SQLException {
        String query=getLoadPropertyInstancesQuery(entityURIs,loadFlag);
        Statement statement=m_connection.createStatement();
        try {
            ResultSet resultSet=statement.executeQuery(query);
            try {
                List objects=new ArrayList();
                String lastEntityURI=null;
                int lastVersion=0;
                while (resultSet.next()) {
                    int modelID=resultSet.getInt(1);
                    String entityURI=resultSet.getString(2);
                    int version=resultSet.getInt(3);
                    if (lastEntityURI==null || !lastEntityURI.equals(entityURI)) {
                        if (lastEntityURI!=null)
                            consumePropertyInstances(lastEntityURI,lastVersion,objects);
                        objects.clear();
                        lastEntityURI=entityURI;
                        lastVersion=version;
                    }
                    String sourceInstanceURI=resultSet.getString(4);
                    if (!resultSet.wasNull()) {
                        int sourceInstanceVersion=resultSet.getInt(5);
                        Object targetValue;
                        String targetInstanceURI=resultSet.getString(6);
                        int targetInstanceVersion=resultSet.getInt(7);
                        if (targetInstanceVersion!=0)
                            targetValue=new EntityID(targetInstanceURI,targetInstanceVersion);
                        else
                            targetValue=resultSet.getString(8);
                        objects.add(new Object[] { new RelationEntityID(sourceInstanceURI,sourceInstanceVersion,modelID),targetValue });
                    }
                }
                if (lastEntityURI!=null)
                    consumePropertyInstances(lastEntityURI,lastVersion,objects);
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }
    /**
     * Returns the maximum number of in literals in a query.
     *
     * @return                              the maximum number of in literals in a query
     */
    protected int getQueryMaximumInLiterals() {
        return 200;
    }
    /**
     * Returns the maximum number of URIs that are loaded using IN literals.
     *
     * @return                              the maximum number of URIs that are loaded using IN literals
     */
    protected int getMaximumURIsToLoadUsingINLiterals() {
        return 400;
    }
    /**
     * Returns the query for loading object basics.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @return                              the query for loading object basics
     */
    protected abstract String getLoadBasicsQuery(String entityURIs,int loadFlag);
    /**
     * Returns the query for loading related entities.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @return                              the query for loading related items
     */
    protected abstract String getLoadRelatedEntitiesQuery(String entityURIs,int loadFlag);
    /**
     * Returns the query for loading instance property values.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @return                              the query for loading instance property values
     */
    protected abstract String getLoadInstancePropertyValuesQuery(String entityURIs,int loadFlag);
    /**
     * Returns the query for loading property instances.
     *
     * @param entityURIs                    the list of entity URIs
     * @param loadFlag                      the flag specifying the information to load
     * @return                              the query for loading property instances
     */
    protected abstract String getLoadPropertyInstancesQuery(String entityURIs,int loadFlag);
    /**
     * Returns the query for loading all concepts.
     *
     * @return                              the query for loading all concepts
     */
    protected abstract String getLoadConceptsQuery();
    /**
     * Returns the query for loading all properties.
     *
     * @return                              the query for loading all properties
     */
    protected abstract String getLoadPropertiesQuery();
    /**
     * Returns the query for loading all instances.
     *
     * @return                              the query for loading all instances
     */
    protected abstract String getLoadInstancesQuery();
    /**
     * Examines the connection and returns the type of the loader.
     *
     * @param connection                    the connection
     * @return                              the type of the loader
     */
    public static int getObjectLoaderType(Connection connection) {
        if (ObjectLoaderOracle8i.isOracle8i(connection))
            return 1;
        else if (ObjectLoaderPostgreSQL.isPostgreSQL(connection))
            return 2;
        else
            return 0;
    }
    /**
     * Creates a loader for supplied connection of given type.
     *
     * @param connection                    the connection
     * @param modelID                       the ID of the model
     * @param objectLoaderType              the type of the loader
     * @return                              the loader
     * @throws KAONException                thrown if there is an error
     */
    public static AbstractSQLObjectLoader createSQLObjectLoader(Connection connection,int modelID,int objectLoaderType) throws KAONException {
        switch (objectLoaderType) {
        case 1:
            return new ObjectLoaderOracle8i(connection,modelID);
        case 2:
            return new ObjectLoaderPostgreSQL(connection,modelID);
        default:
            return new ObjectLoaderSQL2(connection,modelID);
        }
    }
}
