package edu.unika.aifb.kaon.engineeringserver.loader;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Abstract base class for all loaders.
 */
public abstract class AbstractObjectLoader {
    /** The ID of the model. */
    protected int m_modelID;
    /** Map of ConceptInfo objects. */
    protected Map m_conceptInfos;
    /** Map of PropertyInfo objects. */
    protected Map m_propertyInfos;
    /** Map of InstanceInfo objects. */
    protected Map m_instanceInfos;

    /**
     * Creates an instance of this class.
     *
     * @param modelID                       the ID of the model
     */
    public AbstractObjectLoader(int modelID) {
        m_modelID=modelID;
        m_conceptInfos=new HashMap();
        m_propertyInfos=new HashMap();
        m_instanceInfos=new HashMap();
    }
    /**
     * Returns the ConceptInfo object with given URI.
     *
     * @param entityURI                     URI for which ConceptInfo is required
     * @param conceptVersion                the current version
     * @return                              the concept info object with given URI
     */
    protected ConceptInfo getConceptInfo(String entityURI,int conceptVersion) {
        ConceptInfo result=(ConceptInfo)m_conceptInfos.get(entityURI);
        if (result==null) {
            result=new ConceptInfo(entityURI,conceptVersion);
            m_conceptInfos.put(entityURI,result);
        }
        return result;
    }
    /**
     * Returns the PropertyInfo object with given URI.
     *
     * @param entityURI                     URI for which PropertyInfo is required
     * @param propertyVersion               the current version
     * @return                              the property info object with given URI
     */
    protected PropertyInfo getPropertyInfo(String entityURI,int propertyVersion) {
        PropertyInfo result=(PropertyInfo)m_propertyInfos.get(entityURI);
        if (result==null) {
            result=new PropertyInfo(entityURI,propertyVersion);
            m_propertyInfos.put(entityURI,result);
        }
        return result;
    }
    /**
     * Returns the InstanceInfo object with given URI.
     *
     * @param entityURI                     URI for which InstanceInfo is required
     * @param instanceVersion               the current version
     * @return                              the instance info object with given URI
     */
    protected InstanceInfo getInstanceInfo(String entityURI,int instanceVersion) {
        InstanceInfo result=(InstanceInfo)m_instanceInfos.get(entityURI);
        if (result==null) {
            result=new InstanceInfo(entityURI,instanceVersion);
            m_instanceInfos.put(entityURI,result);
        }
        return result;
    }
    /**
     * Creates a combined array of load entities objects.
     *
     * @return                              the array combining the entities of all loadded objects
     */
    protected EntityInfo[] getLoadedObjectsArray() {
        EntityInfo[] result=new EntityInfo[m_conceptInfos.size()+m_propertyInfos.size()+m_instanceInfos.size()];
        int index=0;
        Iterator iterator=m_conceptInfos.values().iterator();
        while (iterator.hasNext())
            result[index++]=(EntityInfo)iterator.next();
        iterator=m_propertyInfos.values().iterator();
        while (iterator.hasNext())
            result[index++]=(EntityInfo)iterator.next();
        iterator=m_instanceInfos.values().iterator();
        while (iterator.hasNext())
            result[index++]=(EntityInfo)iterator.next();
        return result;
    }
    /**
     * Called when related entities for some object have been loaded by the loadRelatedItems() method.
     *
     * @param entityURI                     the URI of the entity loaded
     * @param version                       the version of the entity loaded
     * @param rowType                       the type of the entity loaded
     * @param objects                       a vector of loaded entity IDs
     */
    protected void consumeRelatedEntities(String entityURI,int version,int rowType,List objects) {
        if (rowType==8) {
            DomainInfo[] vector=new DomainInfo[objects.size()];
            objects.toArray(vector);
            getPropertyInfo(entityURI,version).m_domainInfos=vector;
        }
        else if (rowType==1 || rowType==6 || rowType==9 || rowType==10) {
            RelationEntityID[] vector=new RelationEntityID[objects.size()];
            objects.toArray(vector);
            switch (rowType) {
            case 1:
                getConceptInfo(entityURI,version).m_superConceptIDs=vector;
                break;
            case 6:
                getPropertyInfo(entityURI,version).m_superPropertyIDs=vector;
                break;
            case 9:
                getPropertyInfo(entityURI,version).m_rangeConceptIDs=vector;
                break;
            case 10:
                getInstanceInfo(entityURI,version).m_instanceParentConceptIDs=vector;
                break;
            }
        }
        else {
            EntityID[] vector=new EntityID[objects.size()];
            objects.toArray(vector);
            switch (rowType) {
            case 2:
                getConceptInfo(entityURI,version).m_subConceptIDs=vector;
                break;
            case 3:
                getConceptInfo(entityURI,version).m_propertiesFromIDs=vector;
                break;
            case 4:
                getConceptInfo(entityURI,version).m_propertiesToIDs=vector;
                break;
            case 5:
                getConceptInfo(entityURI,version).m_instanceIDs=vector;
                break;
            case 7:
                getPropertyInfo(entityURI,version).m_subPropertyIDs=vector;
                break;
            }
        }
    }
    /**
     * Called when instance property values for some object have been loaded by the loadInstancePropertyValues() method.
     *
     * @param entityURI                     the URI of the entity loaded
     * @param version                       the version of the entity loaded
     * @param rowType                       the type of the entity loaded
     * @param values                        map of property values
     */
    protected void consumeInstancePropertyValues(String entityURI,int version,int rowType,Map values) {
        Object[][] vector=getPropertyValues(values);
        InstanceInfo instanceInfo=getInstanceInfo(entityURI,version);
        switch (rowType) {
        case 11:
            instanceInfo.m_propertyValuesFrom=vector;
            break;
        case 12:
            instanceInfo.m_propertyValuesTo=vector;
            break;
        }
    }
    /**
     * Called to set the value of the property into the map.
     *
     * @param propertyID                    the ID of the property
     * @param value                         the value to add
     * @param values                        the map of values
     */
    protected void addPropertyInstanceValue(EntityID propertyID,Object value,Map values) {
        Object currentValue=values.get(propertyID);
        if (currentValue==null)
            values.put(propertyID,value);
        else if (currentValue instanceof Set)
            ((Set)currentValue).add(value);
        else {
            Set set=new HashSet();
            set.add(currentValue);
            set.add(value);
            values.put(propertyID,set);
        }
    }
    /**
     * Converts a map of property values to an double vector.
     *
     * @param map                           map of property values
     * @return                              array suitable for transport in InstanceInfo class
     */
    protected Object[][] getPropertyValues(Map map) {
        Object[][] result=new Object[map.size()][];
        int index=0;
        Iterator properties=map.keySet().iterator();
        while (properties.hasNext()) {
            EntityID propertyID=(EntityID)properties.next();
            Object value=map.get(propertyID);
            if (value instanceof Set) {
                Set set=(Set)value;
                result[index]=new Object[set.size()+1];
                result[index][0]=propertyID;
                int valueIndex=1;
                Iterator values=((Set)value).iterator();
                while (values.hasNext())
                    result[index][valueIndex++]=values.next();
            }
            else {
                result[index]=new Object[2];
                result[index][0]=propertyID;
                result[index][1]=value;
            }
            index++;
        }
        return result;
    }
    /**
     * Called when property instances for some object have been loaded by the loadPropertyInstances() method.
     *
     * @param entityURI                     the URI of the entity loaded
     * @param version                       the version of the entity loaded
     * @param objects                       a vector of loaded entity IDs
     */
    protected void consumePropertyInstances(String entityURI,int version,List objects) {
        Object[][] vector=new Object[objects.size()][];
        objects.toArray(vector);
        getPropertyInfo(entityURI,version).m_propertyInstances=vector;
    }
    /**
     * Returns the list of IDs of concepts that are in the model.
     *
     * @return                              the array of IDs of concepts that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public abstract EntityID[] loadConcepts() throws KAONException;
    /**
     * Returns the list of IDs of properties that are in the model.
     *
     * @return                              the array of IDs of properties that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public abstract EntityID[] loadProperties() throws KAONException;
    /**
     * Returns the list of IDs of instances that are in the model.
     *
     * @return                              the array of IDs of instances that are in the model
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public abstract EntityID[] loadInstances() throws KAONException;
    /**
     * Called to load information about specified entities.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @return                              information about loaded entities
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public abstract EntityInfo[] loadEntities(String[] entityURIs,int loadFlag) throws KAONException;
}
