package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * DAO for PropertyInstance objects.
 */
public class PropertyInstanceDAO extends AbstractDAO {
    /** The set of added relation instances. */
    protected Set m_addedRelationInstances;
    /** The set of removed relation instances. */
    protected Set m_removedRelationInstances;
    /** The set of added attribute instances. */
    protected Set m_addedAttributeInstances;
    /** The set of removed attribute instances. */
    protected Set m_removedAttributeInstances;
    /** The statement for creating RelationInstance objects. */
    protected PreparedStatement m_createRelationInstance;
    /** The statement for deleting RelationInstance objects. */
    protected PreparedStatement m_deleteRelationInstance;
    /** The statement for creating AttributeInstance objects. */
    protected PreparedStatement m_createAttributeInstance;
    /** The statement for deleting AttributeInstance objects. */
    protected PreparedStatement m_deleteAttributeInstance;
    /** The statement for selecting PropertyInstances of given property and source instance. */
    protected PreparedStatement m_selectPropertyInstancesForPropertyAndSourceInstance;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public PropertyInstanceDAO(EngineeringServerDAO engineeringServerDAO) {
        super(engineeringServerDAO);
        m_addedRelationInstances=new HashSet();
        m_removedRelationInstances=new HashSet();
        m_addedAttributeInstances=new HashSet();
        m_removedAttributeInstances=new HashSet();
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        closeStatement(m_createRelationInstance);
        closeStatement(m_deleteRelationInstance);
        closeStatement(m_createAttributeInstance);
        closeStatement(m_deleteAttributeInstance);
        closeStatement(m_selectPropertyInstancesForPropertyAndSourceInstance);
    }
    /**
     * Flushes this DAO.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flush() throws SQLException {
        if (!m_removedRelationInstances.isEmpty()) {
            if (m_deleteRelationInstance==null)
                m_deleteRelationInstance=getConnection().prepareStatement("DELETE FROM RelationInstance WHERE modelID=? AND propertyID=? AND sourceInstanceID=? AND targetInstanceID=?");
            m_deleteRelationInstance.clearBatch();
            Iterator iterator=m_removedRelationInstances.iterator();
            while (iterator.hasNext()) {
                RelationInstance relationInstance=(RelationInstance)iterator.next();
                m_deleteRelationInstance.setInt(1,relationInstance.m_modelID);
                m_deleteRelationInstance.setInt(2,relationInstance.m_property.getEntityID());
                m_deleteRelationInstance.setInt(3,relationInstance.m_sourceInstance.getEntityID());
                m_deleteRelationInstance.setInt(4,relationInstance.m_targetInstance.getEntityID());
                m_deleteRelationInstance.addBatch();
            }
            assertOneRecordUpdated(m_deleteRelationInstance.executeBatch());
            m_removedRelationInstances.clear();
        }
        if (!m_addedRelationInstances.isEmpty()) {
            if (m_createRelationInstance==null)
                m_createRelationInstance=getConnection().prepareStatement("INSERT INTO RelationInstance (modelID,propertyID,sourceInstanceID,targetInstanceID) VALUES (?,?,?,?)");
            m_createRelationInstance.clearBatch();
            Iterator iterator=m_addedRelationInstances.iterator();
            while (iterator.hasNext()) {
                RelationInstance relationInstance=(RelationInstance)iterator.next();
                m_createRelationInstance.setInt(1,relationInstance.m_modelID);
                m_createRelationInstance.setInt(2,relationInstance.m_property.getEntityID());
                m_createRelationInstance.setInt(3,relationInstance.m_sourceInstance.getEntityID());
                m_createRelationInstance.setInt(4,relationInstance.m_targetInstance.getEntityID());
                m_createRelationInstance.addBatch();
            }
            assertOneRecordUpdated(m_createRelationInstance.executeBatch());
            m_addedRelationInstances.clear();
        }
        if (!m_removedAttributeInstances.isEmpty()) {
            if (m_deleteAttributeInstance==null)
                m_deleteAttributeInstance=getConnection().prepareStatement("DELETE FROM AttributeInstance WHERE modelID=? AND propertyID=? AND sourceInstanceID=? AND textValue=?");
            m_deleteAttributeInstance.clearBatch();
            Iterator iterator=m_removedAttributeInstances.iterator();
            while (iterator.hasNext()) {
                AttributeInstance attributeInstance=(AttributeInstance)iterator.next();
                m_deleteAttributeInstance.setInt(1,attributeInstance.m_modelID);
                m_deleteAttributeInstance.setInt(2,attributeInstance.m_property.getEntityID());
                m_deleteAttributeInstance.setInt(3,attributeInstance.m_sourceInstance.getEntityID());
                m_deleteAttributeInstance.setString(4,attributeInstance.m_textValue);
                m_deleteAttributeInstance.addBatch();
            }
            assertOneRecordUpdated(m_deleteAttributeInstance.executeBatch());
            m_removedAttributeInstances.clear();
        }
        if (!m_addedAttributeInstances.isEmpty()) {
            if (m_createAttributeInstance==null)
                m_createAttributeInstance=getConnection().prepareStatement("INSERT INTO AttributeInstance (modelID,propertyID,sourceInstanceID,textValue) VALUES (?,?,?,?)");
            m_createAttributeInstance.clearBatch();
            Iterator iterator=m_addedAttributeInstances.iterator();
            while (iterator.hasNext()) {
                AttributeInstance attributeInstance=(AttributeInstance)iterator.next();
                m_createAttributeInstance.setInt(1,attributeInstance.m_modelID);
                m_createAttributeInstance.setInt(2,attributeInstance.m_property.getEntityID());
                m_createAttributeInstance.setInt(3,attributeInstance.m_sourceInstance.getEntityID());
                m_createAttributeInstance.setString(4,attributeInstance.m_textValue);
                m_createAttributeInstance.addBatch();
            }
            assertOneRecordUpdated(m_createAttributeInstance.executeBatch());
            m_addedAttributeInstances.clear();
        }
    }
    /**
     * Creates a property instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param targetInstance                the target instance (may be <code>null</code>)
     * @param textValue                     the text value (may be <code>null</code>)
     * @throws SQLException                 thrown if there is an error
     */
    public void createPropertyInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,OIModelEntity targetInstance,String textValue) throws SQLException {
        if (targetInstance!=null)
            createRelationInstance(modelID,property,sourceInstance,targetInstance);
        if (textValue!=null)
            createAttributeInstance(modelID,property,sourceInstance,textValue);
    }
    /**
     * Deletes a property instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param targetInstance                the target instance (may be <code>null</code>)
     * @param textValue                     the text value (may be <code>null</code>)
     * @throws SQLException                 thrown if there is an error
     */
    public void deletePropertyInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,OIModelEntity targetInstance,String textValue) throws SQLException {
        if (targetInstance!=null)
            deleteRelationInstance(modelID,property,sourceInstance,targetInstance);
        if (textValue!=null)
            deleteAttributeInstance(modelID,property,sourceInstance,textValue);
    }
    /**
     * Creates a relation instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param targetInstance                the target instance
     * @throws SQLException                 thrown if there is an error
     */
    public void createRelationInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,OIModelEntity targetInstance) throws SQLException {
        RelationInstance relationInstance=new RelationInstance(modelID,property,sourceInstance,targetInstance);
        if (m_addedRelationInstances.contains(relationInstance))
            throw new SQLException("Relation instance "+modelID+", "+property.getEntityID()+", "+sourceInstance.getEntityID()+", "+targetInstance.getEntityID()+" already added.");
        if (m_removedRelationInstances.contains(relationInstance))
            m_engineeringServerDAO.flush();
        m_addedRelationInstances.add(relationInstance);
    }
    /**
     * Deletes a relation instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param targetInstance                the target instance
     * @throws SQLException                 thrown if there is an error
     */
    public void deleteRelationInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,OIModelEntity targetInstance) throws SQLException {
        RelationInstance relationInstance=new RelationInstance(modelID,property,sourceInstance,targetInstance);
        if (m_removedRelationInstances.contains(relationInstance))
            throw new SQLException("Relation instance "+modelID+", "+property.getEntityID()+", "+sourceInstance.getEntityID()+", "+targetInstance.getEntityID()+" already added.");
        if (m_addedRelationInstances.contains(relationInstance))
            m_engineeringServerDAO.flush();
        m_removedRelationInstances.add(relationInstance);
    }
    /**
     * Creates an attribute instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param textValue                     the attribute value
     * @throws SQLException                 thrown if there is an error
     */
    public void createAttributeInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,String textValue) throws SQLException {
        AttributeInstance attributeInstance=new AttributeInstance(modelID,property,sourceInstance,textValue);
        if (m_addedAttributeInstances.contains(attributeInstance))
            throw new SQLException("Attribute instance "+modelID+", "+property.getEntityID()+", "+sourceInstance.getEntityID()+", "+textValue+" already added.");
        if (m_removedAttributeInstances.contains(attributeInstance))
            m_engineeringServerDAO.flush();
        m_addedAttributeInstances.add(attributeInstance);
    }
    /**
     * Deletes an attribute instance object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @param textValue                     the attribute value
     * @throws SQLException                 thrown if there is an error
     */
    public void deleteAttributeInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,String textValue) throws SQLException {
        AttributeInstance attributeInstance=new AttributeInstance(modelID,property,sourceInstance,textValue);
        if (m_removedAttributeInstances.contains(attributeInstance))
            throw new SQLException("Attribute instance "+modelID+", "+property.getEntityID()+", "+sourceInstance.getEntityID()+", "+textValue+" already added.");
        if (m_addedAttributeInstances.contains(attributeInstance))
            m_engineeringServerDAO.flush();
        m_removedAttributeInstances.add(attributeInstance);
    }
    /**
     * Selects all instances of given property and given source instance.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param sourceInstance                the source instance
     * @return                              an array of either <code>EntityID</code> or <code>String</code> objects
     * @throws SQLException                 thrown if there is an error
     */
    public Object[] getPropertyInstances(int modelID,OIModelEntity property,OIModelEntity sourceInstance) throws SQLException {
        m_engineeringServerDAO.flush();
        if (m_selectPropertyInstancesForPropertyAndSourceInstance==null)
            m_selectPropertyInstancesForPropertyAndSourceInstance=getConnection().prepareStatement(
                "    SELECT textValue,0 FROM AttributeInstance ai WHERE ai.modelID=? AND ai.propertyID=? AND ai.sourceInstanceID=? "+
                "UNION ALL "+
                "    SELECT ti.entityURI,ti.instanceVersion FROM RelationInstance ri,OIModelEntity ti WHERE ri.targetInstanceID=ti.entityID AND ri.modelID=? AND ri.propertyID=? AND ri.sourceInstanceID=?"
            );
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(1,modelID);
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(2,property.getEntityID());
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(3,sourceInstance.getEntityID());
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(4,modelID);
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(5,property.getEntityID());
        m_selectPropertyInstancesForPropertyAndSourceInstance.setInt(6,sourceInstance.getEntityID());
        Set result=new HashSet();
        ResultSet resultSet=m_selectPropertyInstancesForPropertyAndSourceInstance.executeQuery();
        try {
            while (resultSet.next()) {
                String firstColumn=resultSet.getString(1);
                int instanceVersion=resultSet.getInt(2);
                if (instanceVersion==0)
                    result.add(firstColumn);
                else
                    result.add(new EntityID(firstColumn,instanceVersion));
            }
        }
        finally {
            resultSet.close();
        }
        return result.toArray();
    }

    /**
     * The key class for the relation instances
     */
    protected static class RelationInstance {
        public int m_modelID;
        public OIModelEntity m_property;
        public OIModelEntity m_sourceInstance;
        public OIModelEntity m_targetInstance;
        protected int m_hashCode;

        public RelationInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,OIModelEntity targetInstance) {
            m_modelID=modelID;
            m_property=property;
            m_sourceInstance=sourceInstance;
            m_targetInstance=targetInstance;
            m_hashCode=m_modelID+7*(m_property.hashCode()+7*(m_sourceInstance.hashCode()+7*m_targetInstance.hashCode()));
        }
        public int hashCode() {
            return m_hashCode;
        }
        public boolean equals(Object that) {
            if (this==that)
                return true;
            if (!(that instanceof RelationInstance))
                return false;
            RelationInstance thatRelationInstance=(RelationInstance)that;
            return m_modelID==thatRelationInstance.m_modelID && m_property==thatRelationInstance.m_property && m_sourceInstance==thatRelationInstance.m_sourceInstance && m_targetInstance==thatRelationInstance.m_targetInstance;
        }
    }

    /**
     * The key class for the attribute instances
     */
    protected static class AttributeInstance {
        public int m_modelID;
        public OIModelEntity m_property;
        public OIModelEntity m_sourceInstance;
        public String m_textValue;
        protected int m_hashCode;

        public AttributeInstance(int modelID,OIModelEntity property,OIModelEntity sourceInstance,String textValue) {
            m_modelID=modelID;
            m_property=property;
            m_sourceInstance=sourceInstance;
            m_textValue=textValue;
            m_hashCode=m_modelID+7*(m_property.hashCode()+7*(m_sourceInstance.hashCode()+7*textValue.hashCode()));
        }
        public int hashCode() {
            return m_hashCode;
        }
        public boolean equals(Object that) {
            if (this==that)
                return true;
            if (!(that instanceof AttributeInstance))
                return false;
            AttributeInstance thatAttributeInstance=(AttributeInstance)that;
            return m_modelID==thatAttributeInstance.m_modelID && m_property==thatAttributeInstance.m_property && m_sourceInstance==thatAttributeInstance.m_sourceInstance && m_textValue.equals(thatAttributeInstance.m_textValue);
        }
    }
}
