package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

/**
 * DAO for PropertyDomain objects.
 */
public class PropertyDomainDAO extends AbstractRelationshipDAO {
    /** The set of updated domains. */
    protected Set m_updatedRelationships;
    /** The map of relationships to their cardinalities. */
    protected Map m_cardinalities;
    /** Selectes the cardinalities of a PropertyDomain object. */
    protected PreparedStatement m_selectPropertyDomainCardinalities;
    /** Updates the cardinalities of a PropertyDomain object. */
    protected PreparedStatement m_updatePropertyDomainCardinalities;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public PropertyDomainDAO(EngineeringServerDAO engineeringServerDAO) {
        super(engineeringServerDAO);
        m_updatedRelationships=new HashSet();
        m_cardinalities=new HashMap();
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        super.close();
        closeStatement(m_selectPropertyDomainCardinalities);
        closeStatement(m_updatePropertyDomainCardinalities);
    }
    /**
     * Sets the cardinality of the relationship.
     *
     * @param relationship                  the relationship
     * @param minimumCardinality            the minimum cardinality
     * @param maximumCardinality            the maximum cardinality
     */
    protected void setCardinality(Relationship relationship,int minimumCardinality,int maximumCardinality) {
        Cardinality cardinality=(Cardinality)m_cardinalities.get(relationship);
        if (cardinality==null) {
            cardinality=new Cardinality();
            m_cardinalities.put(relationship,cardinality);
        }
        cardinality.m_minimumCardinality=minimumCardinality;
        cardinality.m_maximumCardinality=maximumCardinality;
    }
    /**
     * Creates a concept hierarchy object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param concept                       the concept
     * @param minimumCardinality            the minimum cardinality
     * @param maximumCardinality            the maximum cardinality
     * @throws SQLException                 thrown if there is an error
     */
    public void createPropertyDomain(int modelID,OIModelEntity property,OIModelEntity concept,int minimumCardinality,int maximumCardinality) throws SQLException {
        Relationship relationship=new Relationship(modelID,property,concept);
        if (m_addedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+property.getEntityID()+", "+concept.getEntityID()+" already added.");
        if (m_updatedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+property.getEntityID()+", "+concept.getEntityID()+" already updated - can't create it now.");
        if (m_removedRelationships.contains(relationship))
            m_engineeringServerDAO.flush();
        m_addedRelationships.add(relationship);
        setCardinality(relationship,minimumCardinality,maximumCardinality);
    }
    /**
     * Deletes a concept hierarchy object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param concept                       the concept
     * @throws SQLException                 thrown if there is an error
     */
    public void deletePropertyDomain(int modelID,OIModelEntity property,OIModelEntity concept) throws SQLException {
        Relationship relationship=new Relationship(modelID,property,concept);
        if (m_removedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+property.getEntityID()+", "+concept.getEntityID()+" already removed.");
        if (m_addedRelationships.contains(relationship) || m_updatedRelationships.contains(relationship))
            m_engineeringServerDAO.flush();
        m_removedRelationships.add(relationship);
        m_cardinalities.remove(relationship);
    }
    /**
     * Returns the current cardinalities of given PropertyDomain object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param concept                       the concept
     * @return                              an array containing the minimum and maximum cardinality
     * @throws SQLException                 thrown if there is an error
     */
    public int[] selectPropertyDomainCardinalities(int modelID,OIModelEntity property,OIModelEntity concept) throws SQLException {
        Relationship relationship=new Relationship(modelID,property,concept);
        if (m_removedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+property.getEntityID()+", "+concept.getEntityID()+" already removed.");
        Cardinality cardinality=(Cardinality)m_cardinalities.get(relationship);
        if (cardinality==null) {
            cardinality=new Cardinality();
            if (m_selectPropertyDomainCardinalities==null)
                m_selectPropertyDomainCardinalities=getConnection().prepareStatement("SELECT minimumCardinality,maximumCardinality FROM PropertyDomain WHERE modelID=? AND propertyID=? AND conceptID=?");
            m_selectPropertyDomainCardinalities.setInt(1,modelID);
            m_selectPropertyDomainCardinalities.setInt(2,property.getEntityID());
            m_selectPropertyDomainCardinalities.setInt(3,concept.getEntityID());
            ResultSet resultSet=m_selectPropertyDomainCardinalities.executeQuery();
            try {
                if (!resultSet.next())
                    throw new SQLException("Property domain object not found.");
                else {
                    cardinality.m_minimumCardinality=resultSet.getInt(1);
                    cardinality.m_maximumCardinality=resultSet.getInt(2);
                }
            }
            finally {
                resultSet.close();
            }
            m_cardinalities.put(relationship,cardinality);
        }
        return new int[] { cardinality.m_minimumCardinality,cardinality.m_maximumCardinality };
    }
    /**
     * Updates the cardinalities of given PropertyDomain object.
     *
     * @param modelID                       the ID of the model
     * @param property                      the property
     * @param concept                       the concept
     * @param minimumCardinality            the minimum cardinality
     * @param maximumCardinality            the maximum cardinality
     * @throws SQLException                 thrown if there is an error
     */
    public void updatePropertyDomainCarinalities(int modelID,OIModelEntity property,OIModelEntity concept,int minimumCardinality,int maximumCardinality) throws SQLException {
        Relationship relationship=new Relationship(modelID,property,concept);
        if (m_removedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+property.getEntityID()+", "+concept.getEntityID()+" was removed.");
        if (!m_addedRelationships.contains(relationship))
            m_updatedRelationships.add(relationship);
        setCardinality(relationship,minimumCardinality,maximumCardinality);
    }
    /**
     * Flushes the additions from this DAO.
     *
     * @throws SQLException                 thrown if there is an error
     */
    protected void flushAdditions() throws SQLException {
        if (!m_addedRelationships.isEmpty()) {
            if (m_addRelationships==null)
                m_addRelationships=getConnection().prepareStatement(getAddRelationshipsSQL());
            m_addRelationships.clearBatch();
            Iterator iterator=m_addedRelationships.iterator();
            while (iterator.hasNext()) {
                Relationship relationship=(Relationship)iterator.next();
                Cardinality cardinality=(Cardinality)m_cardinalities.get(relationship);
                m_addRelationships.setInt(1,relationship.m_modelID);
                m_addRelationships.setInt(2,relationship.m_entity1.getEntityID());
                m_addRelationships.setInt(3,relationship.m_entity2.getEntityID());
                m_addRelationships.setInt(4,cardinality.m_minimumCardinality);
                m_addRelationships.setInt(5,cardinality.m_maximumCardinality);
                m_addRelationships.addBatch();
            }
            assertOneRecordUpdated(m_addRelationships.executeBatch());
            m_addedRelationships.clear();
        }
    }
    /**
     * Flushes the changes in this DAO.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flush() throws SQLException {
        flushAdditions();
        if (!m_updatedRelationships.isEmpty()) {
            if (m_updatePropertyDomainCardinalities==null)
                m_updatePropertyDomainCardinalities=getConnection().prepareStatement("UPDATE PropertyDomain SET minimumCardinality=?,maximumCardinality=? WHERE modelID=? AND propertyID=? AND conceptID=?");
            m_updatePropertyDomainCardinalities.clearBatch();
            Iterator iterator=m_updatedRelationships.iterator();
            while (iterator.hasNext()) {
                Relationship relationship=(Relationship)iterator.next();
                Cardinality cardinality=(Cardinality)m_cardinalities.get(relationship);
                m_updatePropertyDomainCardinalities.setInt(1,cardinality.m_minimumCardinality);
                m_updatePropertyDomainCardinalities.setInt(2,cardinality.m_maximumCardinality);
                m_updatePropertyDomainCardinalities.setInt(3,relationship.m_modelID);
                m_updatePropertyDomainCardinalities.setInt(4,relationship.m_entity1.getEntityID());
                m_updatePropertyDomainCardinalities.setInt(5,relationship.m_entity2.getEntityID());
                m_updatePropertyDomainCardinalities.addBatch();
            }
            assertOneRecordUpdated(m_updatePropertyDomainCardinalities.executeBatch());
            m_updatedRelationships.clear();
        }
        flushRemovals();
    }
    /**
     * The SQL statement for creating relationships.
     *
     * @return                              the SQL statement for creating relationships
     */
    protected String getAddRelationshipsSQL() {
        return "INSERT INTO PropertyDomain (modelID,propertyID,conceptID,minimumCardinality,maximumCardinality) VALUES (?,?,?,?,?)";
    }
    /**
     * The SQL statement for removing relationships.
     *
     * @return                              the SQL statement for removing relationships
     */
    protected String getRemoveRelationshipsSQL() {
        return "DELETE FROM PropertyDomain WHERE modelID=? AND propertyID=? AND conceptID=?";
    }

    protected static class Cardinality {
        public int m_minimumCardinality;
        public int m_maximumCardinality;
    }
}
