package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.sql.SQLException;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * The manager for server-side objects.
 */
public class ObjectManager {
    /** The engineering server DAO. */
    protected EngineeringServerDAO m_engineeringServerDAO;
    /** The map of OIModelObjects objects indexed by the model ID. */
    protected Map m_oimodelObjectsByModelID;
    /** The map of OIModelObjects objects indexed by the model logical URI. */
    protected Map m_oimodelObjectsByLogicalURI;
    /** The set of URIs that have been loaded. */
    protected Set m_loadedURIs;
    /** Set of updated concepts. */
    protected Set m_updatedConcepts;
    /** Set of updated properties. */
    protected Set m_updatedProperties;
    /** Set of updated instances. */
    protected Set m_updatedInstances;
    /** Map of additional entity info objects. */
    protected Map m_additionalEntityInfos;
    /** Set of EntityIDs of updated concepts. */
    protected Set m_conceptEntityIDs;
    /** Set of EntityIDs of updated properties. */
    protected Set m_propertyEntityIDs;
    /** Set of EntityIDs of updated instances. */
    protected Set m_instanceEntityIDs;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     * @throws SQLException                 thrown if there is an error
     */
    public ObjectManager(EngineeringServerDAO engineeringServerDAO) throws SQLException {
        m_engineeringServerDAO=engineeringServerDAO;
        m_oimodelObjectsByModelID=new HashMap();
        m_oimodelObjectsByLogicalURI=new HashMap();
        m_loadedURIs=new HashSet();
        loadAllModels();
        m_updatedConcepts=new HashSet();
        m_updatedProperties=new HashSet();
        m_updatedInstances=new HashSet();
        m_additionalEntityInfos=new HashMap();
        m_conceptEntityIDs=new HashSet();
        m_propertyEntityIDs=new HashSet();
        m_instanceEntityIDs=new HashSet();
    }
    /**
     * Loads all models.
     *
     * @throws SQLException                 thrown if there is an error
     */
    protected void loadAllModels() throws SQLException {
        Object[][] oimodelInformation=m_engineeringServerDAO.getOIModelDAO().getAllOIModelInformation();
        for (int i=oimodelInformation.length-1;i>=0;i--) {
            Integer modelID=(Integer)oimodelInformation[i][0];
            String logicalURI=(String)oimodelInformation[i][1];
            OIModelObjects oimodelObjects=new OIModelObjects(modelID.intValue());
            m_oimodelObjectsByModelID.put(modelID,oimodelObjects);
            m_oimodelObjectsByLogicalURI.put(logicalURI,oimodelObjects);
        }
        updateIncludedOIModels(oimodelInformation);
    }
    /**
     * Updates the information about the included OI-model objects.
     *
     * @param oimodelInformation            the information about OI-models
     * @throws SQLException                 thrown if there is an error
     */
    protected void updateIncludedOIModels(Object[][] oimodelInformation) throws SQLException {
        for (int i=oimodelInformation.length-1;i>=0;i--) {
            Object[] row=oimodelInformation[i];
            Integer modelID=(Integer)row[0];
            OIModelObjects oimodelObjects=(OIModelObjects)m_oimodelObjectsByModelID.get(modelID);
            for (int j=2;j<row.length;j++) {
                Integer includedModelID=(Integer)row[j];
                if (!includedModelID.equals(modelID)) {
                    OIModelObjects includedOIModelObjects=(OIModelObjects)m_oimodelObjectsByModelID.get(includedModelID);
                    if (includedOIModelObjects==null)
                        throw new SQLException("Model with ID '"+includedModelID+"' doesn't exist.");
                    oimodelObjects.addIncludedOIModelObjects(includedOIModelObjects);
                }
            }
        }
    }
    /**
     * Updates the information about the included OI-model objects.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void updateIncludedOIModels() throws SQLException {
        Object[][] oimodelInformation=m_engineeringServerDAO.getOIModelDAO().getAllOIModelInformation();
        updateIncludedOIModels(oimodelInformation);
    }
    /**
     * Returns the model ID for given logical URI.
     *
     * @param logicalURI                    the logical URI
     * @return                              the model ID for given logical URI (or -1 if model doesn't exist)
     */
    public int getOIModelID(String logicalURI) {
        OIModelObjects oimodelObjects=(OIModelObjects)m_oimodelObjectsByLogicalURI.get(logicalURI);
        if (oimodelObjects==null)
            return -1;
        else
            return oimodelObjects.getModelID();
    }
    /**
     * Flushes the created objects.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flushCreateChange() throws SQLException {
        List entitiesToCreate=new ArrayList();
        List entitiesToSave=new ArrayList();
        Iterator oimodels=m_oimodelObjectsByModelID.values().iterator();
        while (oimodels.hasNext()) {
            OIModelObjects oimodelObjects=(OIModelObjects)oimodels.next();
            Iterator objects=oimodelObjects.entities();
            while (objects.hasNext()) {
                OIModelEntity entity=(OIModelEntity)objects.next();
                if (entity.isCreatedButNotSaved())
                    entitiesToCreate.add(entity);
                else if (entity.isChanged())
                    entitiesToSave.add(entity);
            }
        }
        if (!entitiesToCreate.isEmpty()) {
            OIModelEntity[] entities=new OIModelEntity[entitiesToCreate.size()];
            entitiesToCreate.toArray(entities);
            m_engineeringServerDAO.getOIModelEntityDAO().createOIModelEntities(entities);
        }
        if (!entitiesToSave.isEmpty()) {
            OIModelEntity[] entities=new OIModelEntity[entitiesToSave.size()];
            entitiesToSave.toArray(entities);
            m_engineeringServerDAO.getOIModelEntityDAO().saveOIModelEntities(entities);
        }
    }
    /**
     * Flushes the deleted objects.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flushDelete() throws SQLException {
        List entitiesToDelete=new ArrayList();
        Iterator oimodels=m_oimodelObjectsByModelID.values().iterator();
        while (oimodels.hasNext()) {
            OIModelObjects oimodelObjects=(OIModelObjects)oimodels.next();
            Iterator objects=oimodelObjects.entities();
            while (objects.hasNext()) {
                OIModelEntity oimodelEntity=(OIModelEntity)objects.next();
                if (oimodelEntity.canBeDeleted()) {
                    AdditionalEntityInfo result=(AdditionalEntityInfo)m_additionalEntityInfos.get(oimodelEntity);
                    if (result==null || (!result.m_isConcept && !result.m_isProperty && !result.m_isInstance))
                        entitiesToDelete.add(oimodelEntity);
                }
            }
        }
        if (!entitiesToDelete.isEmpty()) {
            OIModelEntity[] entities=new OIModelEntity[entitiesToDelete.size()];
            entitiesToDelete.toArray(entities);
            m_engineeringServerDAO.getOIModelEntityDAO().deleteOIModelEntities(entities);
        }
    }
    /**
     * Returns the OIModelObjects for given model ID.
     *
     * @param modelID                       the ID of the model which is requested
     * @return                              the OIModelObjects object
     */
    protected OIModelObjects getOIModelObjects(int modelID) {
        return (OIModelObjects)m_oimodelObjectsByModelID.get(new Integer(modelID));
    }
    /**
     * Caches given OIModelEntity objects.
     *
     * @param oimodelEntity                 the entity
     */
    protected void cacheOIModelEntity(OIModelEntity oimodelEntity) {
        OIModelObjects oimodelObjects=getOIModelObjects(oimodelEntity.getModelID());
        oimodelObjects.cacheOIModelEntity(oimodelEntity);
    }
    /**
     * Loads the objects with given URIs.
     * @param entityURIs                    the URIs of the entities
     * @throws SQLException                 thrown if entities cannot be loaded
     */
    public void loadEntities(String[] entityURIs) throws SQLException {
        List remainingURIs=new ArrayList();
        for (int i=0;i<entityURIs.length;i++) {
            String uri=entityURIs[i];
            if (!m_loadedURIs.contains(uri))
                remainingURIs.add(uri);
        }
        if (!remainingURIs.isEmpty()) {
            String[] remainingURIsArray=new String[remainingURIs.size()];
            remainingURIs.toArray(remainingURIsArray);
            Set loadedEntities=m_engineeringServerDAO.getOIModelEntityDAO().loadEntitiesByURIs(remainingURIsArray);
            Iterator entities=loadedEntities.iterator();
            while (entities.hasNext()) {
                OIModelEntity oimodelEntity=(OIModelEntity)entities.next();
                cacheOIModelEntity(oimodelEntity);
            }
            for (int i=0;i<remainingURIsArray.length;i++)
                m_loadedURIs.add(remainingURIsArray[i]);
        }
    }
    /**
     * Returns the OIModelEntity with given URI. This method loads the entity it it hasn't been loaded yet
     *
     * @param modelID                       the ID of the model
     * @param entityURI                     the URI of the entity
     * @return                              the entity with given URI
     * @throws SQLException                 thrown if entity cannot be loaded
     */
    public OIModelEntity getOIModelEntity(int modelID,String entityURI) throws SQLException {
        loadEntities(new String[] { entityURI });
        OIModelObjects oimodelObjects=getOIModelObjects(modelID);
        OIModelEntity result=oimodelObjects.getCachedOIModelEntity(entityURI);
        if (result!=null)
            return result;
        else
            throw new SQLException("OIModelEntity '"+entityURI+"' doesn't exist.");
    }
    /**
     * Returns the OIModelEntity with entity ID.
     *
     * @param entityID                      the ID of the entity
     * @return                              the entity with given ID
     * @throws SQLException                 thrown if entity cannot be loaded
     */
    public OIModelEntity getOIModelEntity(int entityID) throws SQLException {
        Iterator iterator=m_oimodelObjectsByModelID.values().iterator();
        while (iterator.hasNext()) {
            OIModelObjects oimodelObjects=(OIModelObjects)iterator.next();
            OIModelEntity result=oimodelObjects.getCachedOIModelEntity(entityID);
            if (result!=null)
                return result;
        }
        OIModelEntity result=m_engineeringServerDAO.getOIModelEntityDAO().loadEntityByID(entityID);
        cacheOIModelEntity(result);
        return result;
    }
    /**
     * Creates a new OIModelEntity with given URI.
     *
     * @param modelID                       the ID of the model
     * @param entityURI                     the URI of the entity
     * @return                              the new entity with given URI
     * @throws SQLException                 thrown if entity cannot be created
     */
    public OIModelEntity createOIModelEntity(int modelID,String entityURI) throws SQLException {
        OIModelObjects oimodelObjects=getOIModelObjects(modelID);
        if (oimodelObjects.getCachedOIModelEntity(entityURI)!=null)
            throw new SQLException("OIModelEntity with URI '"+entityURI+"' already exists in model "+modelID+".");
        OIModelEntity result=new OIModelEntity(modelID,entityURI);
        cacheOIModelEntity(result);
        return result;
    }
    /**
     * Returns an additional entity info for given entity.
     *
     * @param oimodelEntity                 the entity
     * @return                              additional entity
     */
    public AdditionalEntityInfo getAdditionalEntityInfo(OIModelEntity oimodelEntity) {
        AdditionalEntityInfo result=(AdditionalEntityInfo)m_additionalEntityInfos.get(oimodelEntity);
        if (result==null) {
            result=new AdditionalEntityInfo(oimodelEntity);
            m_additionalEntityInfos.put(oimodelEntity,result);
        }
        return result;
    }
    /**
     * Adjusts the versions of all entities whose existence changed.
     */
    public void adjustVersionsOfAddedOrRemovedEntities() {
        Iterator entities=m_additionalEntityInfos.keySet().iterator();
        while (entities.hasNext()) {
            OIModelEntity oimodelEntity=(OIModelEntity)entities.next();
            AdditionalEntityInfo info=(AdditionalEntityInfo)m_additionalEntityInfos.get(oimodelEntity);
            if (info.m_isConcept && oimodelEntity.getConceptVersion()==Integer.MAX_VALUE)
                oimodelEntity.setConceptVersion(0);
            if (!info.m_isConcept && oimodelEntity.getConceptVersion()!=Integer.MAX_VALUE)
                oimodelEntity.setConceptVersion(Integer.MAX_VALUE);
            if (info.m_isProperty && oimodelEntity.getPropertyVersion()==Integer.MAX_VALUE)
                oimodelEntity.setPropertyVersion(0);
            if (!info.m_isProperty && oimodelEntity.getPropertyVersion()!=Integer.MAX_VALUE)
                oimodelEntity.setPropertyVersion(Integer.MAX_VALUE);
            if (info.m_isInstance && oimodelEntity.getInstanceVersion()==Integer.MAX_VALUE)
                oimodelEntity.setInstanceVersion(0);
            if (!info.m_isInstance && oimodelEntity.getInstanceVersion()!=Integer.MAX_VALUE)
                oimodelEntity.setInstanceVersion(Integer.MAX_VALUE);
        }
        m_additionalEntityInfos.clear();
    }
    /**
     * Adjusts the versions of all touched entities.
     */
    public void adjustVersionsOfChangedEntities() {
        Iterator iterator=m_updatedConcepts.iterator();
        while (iterator.hasNext()) {
            OIModelEntity oimodelEntity=(OIModelEntity)iterator.next();
            oimodelEntity.setConceptVersion(getNewVersion(oimodelEntity.getConceptVersion()));
            m_conceptEntityIDs.add(new EntityID(oimodelEntity.getEntityURI(),oimodelEntity.getConceptVersion()));
        }
        m_updatedConcepts.clear();
        iterator=m_updatedProperties.iterator();
        while (iterator.hasNext()) {
            OIModelEntity oimodelEntity=(OIModelEntity)iterator.next();
            oimodelEntity.setPropertyVersion(getNewVersion(oimodelEntity.getPropertyVersion()));
            m_propertyEntityIDs.add(new EntityID(oimodelEntity.getEntityURI(),oimodelEntity.getPropertyVersion()));
        }
        m_updatedProperties.clear();
        iterator=m_updatedInstances.iterator();
        while (iterator.hasNext()) {
            OIModelEntity oimodelEntity=(OIModelEntity)iterator.next();
            oimodelEntity.setInstanceVersion(getNewVersion(oimodelEntity.getInstanceVersion()));
            m_instanceEntityIDs.add(new EntityID(oimodelEntity.getEntityURI(),oimodelEntity.getInstanceVersion()));
        }
        m_updatedInstances.clear();
    }
    /**
     * Returns the new version of the updated entity.
     *
     * @param existingVersion               current entity version
     * @return                              the new version
     */
    protected int getNewVersion(int existingVersion) {
        if (existingVersion==Integer.MAX_VALUE)
            return Integer.MAX_VALUE;
        else
            return existingVersion+1;
    }
    /**
     * Called when a concept is updated.
     *
     * @param oimodelEntity                 concept entity that has been updated
     */
    public void conceptUpdated(OIModelEntity oimodelEntity) {
        m_updatedConcepts.add(oimodelEntity);
    }
    /**
     * Called when a property is updated.
     *
     * @param oimodelEntity                 property entity that has been updated
     */
    public void propertyUpdated(OIModelEntity oimodelEntity) {
        m_updatedProperties.add(oimodelEntity);
    }
    /**
     * Called when an instance is updated.
     *
     * @param oimodelEntity                 instance entity that has been updated
     */
    public void instanceUpdated(OIModelEntity oimodelEntity) {
        m_updatedInstances.add(oimodelEntity);
    }
    /**
     * Returns the IDs of updated concepts.
     *
     * @return                              IDs of updated concepts
     */
    public EntityID[] getUpdatedConceptIDs() {
        EntityID[] result=new EntityID[m_conceptEntityIDs.size()];
        m_conceptEntityIDs.toArray(result);
        return result;
    }
    /**
     * Returns the IDs of updated properties.
     *
     * @return                              IDs of updated properties
     */
    public EntityID[] getUpdatedPropertyIDs() {
        EntityID[] result=new EntityID[m_propertyEntityIDs.size()];
        m_propertyEntityIDs.toArray(result);
        return result;
    }
    /**
     * Returns the IDs of updated instances.
     *
     * @return                              IDs of updated instances
     */
    public EntityID[] getUpdatedInstanceIDs() {
        EntityID[] result=new EntityID[m_instanceEntityIDs.size()];
        m_instanceEntityIDs.toArray(result);
        return result;
    }

    /**
     * Represents a set of objects in an OI-model.
     */
    protected static class OIModelObjects {
        /** The ID of the OI-model. */
        protected int m_modelID;
        /** The array of all included OI-models. */
        protected List m_allIncludedOIModels;
        /** The map of OIModelEntity objects indexed by their URIs. */
        protected Map m_oimodelEntities;

        /**
         * Creates an instance of this class.
         *
         * @param modelID                       the ID of the model that this manager manages
         */
        public OIModelObjects(int modelID) {
            m_modelID=modelID;
            m_allIncludedOIModels=new ArrayList();
            m_oimodelEntities=new HashMap();
        }
        /**
         * Adds an included OI-model.
         *
         * @param oimodelObjects                the included object
         */
        public void addIncludedOIModelObjects(OIModelObjects oimodelObjects) {
            m_allIncludedOIModels.add(oimodelObjects);
        }
        /**
         * Returns the model ID.
         *
         * @return                              the ID of the model
         */
        public int getModelID() {
            return m_modelID;
        }
        /**
         * Cache the given entity object.
         *
         * @param oimodelEntity                 the entity to be cached
         */
        public void cacheOIModelEntity(OIModelEntity oimodelEntity) {
            String uri=oimodelEntity.getEntityURI();
            if (!m_oimodelEntities.containsKey(uri))
                m_oimodelEntities.put(uri,oimodelEntity);
        }
        /**
         * Removes the given entity object from the cache.
         *
         * @param oimodelEntity                 the entity to be removed from the cache
         */
        public void uncacheOIModelEntity(OIModelEntity oimodelEntity) {
            String uri=oimodelEntity.getEntityURI();
            m_oimodelEntities.remove(uri);
        }
        /**
         * Returns the entity with given URI it is exists in this model or any of the included models.
         *
         * @param entityURI                     the URI of the entity
         * @return                              the entity with given URI, or <code>null</code> if entity cannot be located
         */
        public OIModelEntity getCachedOIModelEntity(String entityURI) {
            OIModelEntity result=(OIModelEntity)m_oimodelEntities.get(entityURI);
            if (result!=null)
                return result;
            for (int i=m_allIncludedOIModels.size()-1;i>=0;i--) {
                OIModelObjects oimodelObjects=(OIModelObjects)m_allIncludedOIModels.get(i);
                result=(OIModelEntity)oimodelObjects.m_oimodelEntities.get(entityURI);
                if (result!=null)
                    return result;
            }
            return null;
        }
        /**
         * Returns the OIModelEntity with entity ID that has been cached in this object.
         *
         * @param entityID                      the ID of the entity
         * @return                              the entity with given ID or <code>null</code> if the object hasn't been cached
         */
        public OIModelEntity getCachedOIModelEntity(int entityID) {
            Iterator values=m_oimodelEntities.values().iterator();
            while (values.hasNext()) {
                OIModelEntity entity=(OIModelEntity)values.next();
                if (entity.getEntityID()==entityID)
                    return entity;
            }
            return null;
        }
        /**
         * Returns the iterator of all entities.
         *
         * @return                              the iterator of all entities
         */
        public Iterator entities() {
            return m_oimodelEntities.values().iterator();
        }
    }

    /**
     * Additional information about an entity.
     */
    public static class AdditionalEntityInfo {
        public boolean m_isConcept;
        public boolean m_isProperty;
        public boolean m_isInstance;

        public AdditionalEntityInfo(OIModelEntity oimodelEntity) {
            m_isConcept=oimodelEntity.getConceptVersion()!=Integer.MAX_VALUE;
            m_isProperty=oimodelEntity.getPropertyVersion()!=Integer.MAX_VALUE;
            m_isInstance=oimodelEntity.getInstanceVersion()!=Integer.MAX_VALUE;
        }
    }
}
