package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Set;
import java.util.HashSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

/**
 * DAO for OIModelEntity objects.
 */
public class OIModelEntityDAO extends AbstractDAO {
    /** The statement for selecting etities by ID. */
    protected PreparedStatement m_selectEntityByID;
    /** The statement for creating etities. */
    protected PreparedStatement m_createEntity;
    /** The statement for saving etities. */
    protected PreparedStatement m_saveEntity;
    /** The statement for delete etities. */
    protected PreparedStatement m_deleteEntity;
    /** The statement for inserting URIs. */
    protected PreparedStatement m_insertURIs;
    /** The statement for deleting URIs. */
    protected PreparedStatement m_deleteURIs;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public OIModelEntityDAO(EngineeringServerDAO engineeringServerDAO) {
        super(engineeringServerDAO);
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        closeStatement(m_selectEntityByID);
        closeStatement(m_createEntity);
        closeStatement(m_deleteEntity);
        closeStatement(m_saveEntity);
        closeStatement(m_insertURIs);
        closeStatement(m_deleteURIs);
    }
    /**
     * Loads the entity with given ID.
     *
     * @param entityID                      the ID of the entity
     * @return                              the loaded entity
     * @throws SQLException                 thrown if there is an error
     */
    public OIModelEntity loadEntityByID(int entityID) throws SQLException {
        if (m_selectEntityByID==null)
            m_selectEntityByID=getConnection().prepareStatement("SELECT entityID,modelID,entityURI,conceptVersion,propertyVersion,isAttribute,isSymmetric,isTransitive,instanceVersion FROM OIModelEntity WHERE entityID=?");
        m_selectEntityByID.setInt(1,entityID);
        ResultSet resultSet=m_selectEntityByID.executeQuery();
        try {
            if (!resultSet.next())
                throw new SQLException("Entity not found");
            else
                return createEntity(resultSet);
        }
        finally {
            resultSet.close();
        }
    }
    /**
     * Loads entities with given URIs.
     *
     * @param entityURIs                    the URIs of the entities
     * @return                              the set of loaded entities
     * @throws SQLException                 thrown if there is an error
     */
    public Set loadEntitiesByURIs(String[] entityURIs) throws SQLException {
        Set result=new HashSet();
        if (entityURIs.length<getMaximumURIsToLoadUsingINLiterals()) {
            int queryMaximumInLiterals=getQueryMaximumInLiterals();
            for (int start=0;start<entityURIs.length;start+=queryMaximumInLiterals) {
                int end=Math.min(start+queryMaximumInLiterals,entityURIs.length);
                StringBuffer buffer=new StringBuffer();
                for (int i=start;i<end;i++) {
                    if (i!=start)
                        buffer.append(",");
                    buffer.append("'");
                    buffer.append(entityURIs[i]);
                    buffer.append("'");
                }
                loadEntitiesFromCondition(buffer.toString(),result);
            }
        }
        else {
            long joinID=insertEntityURIs(entityURIs);
            try {
                loadEntitiesFromCondition("SELECT entityURI FROM URIJoinTable WHERE joinID="+joinID,result);
            }
            finally {
                deleteEntityURIs(joinID);
            }
        }
        return result;
    }
    /**
     * Loads entities matching given condition.
     *
     * @param condition                     the condition for the query for loading entities
     * @param result                        the set to fill with loaded entities
     * @throws SQLException                 thrown if there is an error
     */
    public void loadEntitiesFromCondition(String condition,Set result) throws SQLException {
        String query="SELECT entityID,modelID,entityURI,conceptVersion,propertyVersion,isAttribute,isSymmetric,isTransitive,instanceVersion FROM OIModelEntity WHERE entityURI IN ("+condition+")";
        Statement statement=getConnection().createStatement();
        try {
            ResultSet resultSet=statement.executeQuery(query);
            try {
                while (resultSet.next()) {
                    OIModelEntity entity=createEntity(resultSet);
                    result.add(entity);
                }
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
    }
    /**
     * Inserts elements into the URI join table.
     *
     * @param entityURIs                    the entity URIs
     * @return                              the join ID for the URIs
     * @throws SQLException                 thrown if there is an error
     */
    protected long insertEntityURIs(String[] entityURIs) throws SQLException {
        long joinID=System.currentTimeMillis();
        if (m_insertURIs==null)
            m_insertURIs=getConnection().prepareStatement("INSERT INTO URIJoinTable (joinID,entityURI) VALUES (?,?)");
        m_insertURIs.clearBatch();
        for (int i=0;i<entityURIs.length;i++) {
            m_insertURIs.setLong(1,joinID);
            m_insertURIs.setString(2,entityURIs[i]);
            m_insertURIs.addBatch();
        }
        m_insertURIs.executeBatch();
        return joinID;
    }
    /**
     * Deletes the elements from the URI join table.
     *
     * @param joinID                        the join ID
     * @throws SQLException                 thrown if there is an error
     */
    protected void deleteEntityURIs(long joinID) throws SQLException {
        if (m_deleteURIs==null)
            m_deleteURIs=getConnection().prepareStatement("DELETE FROM URIJoinTable WHERE joinID=?");
        m_deleteURIs.setLong(1,joinID);
        m_deleteURIs.executeUpdate();
    }
    /**
     * Creates an OIModelEntity object from the result set.
     *
     * @param resultSet                     the result set
     * @return                              the OIModelEntity object
     * @throws SQLException                 thrown if there is an error
     */
    protected OIModelEntity createEntity(ResultSet resultSet) throws SQLException {
        int entityID=resultSet.getInt(1);
        int modelID=resultSet.getInt(2);
        String entityURI=resultSet.getString(3);
        OIModelEntity entity=new OIModelEntity(modelID,entityID,entityURI);
        entity.setConceptVersion(resultSet.getInt(4));
        entity.setPropertyVersion(resultSet.getInt(5));
        entity.setIsAttribute(resultSet.getInt(6)!=0);
        entity.setIsSymmetric(resultSet.getInt(7)!=0);
        entity.setIsTransitive(resultSet.getInt(8)!=0);
        entity.setInstanceVersion(resultSet.getInt(9));
        return entity;
    }
    /**
     * Creates an entity with given URI.
     *
     * @param entities                      the OIModelEntity objects being saved
     * @throws SQLException                 thrown if there is an error
     */
    public void createOIModelEntities(OIModelEntity[] entities) throws SQLException {
        int startingEntityID=getNextCounter("oimodelentity",entities.length);
        // first assign entity IDs to all entities
        try {
            for (int i=0;i<entities.length;i++)
                entities[i].setEntityID(startingEntityID+i);
            if (m_createEntity==null)
                m_createEntity=getConnection().prepareStatement("INSERT INTO OIModelEntity (entityID,modelID,entityURI,conceptVersion,propertyVersion,isAttribute,isSymmetric,isTransitive,instanceVersion) VALUES (?,?,?,?,?,?,?,?,?)");
            m_createEntity.clearBatch();
            for (int i=0;i<entities.length;i++) {
                OIModelEntity entity=entities[i];
                m_createEntity.setInt(1,entity.getEntityID());
                m_createEntity.setInt(2,entity.getModelID());
                m_createEntity.setString(3,entity.getEntityURI());
                m_createEntity.setInt(4,entity.getConceptVersion());
                m_createEntity.setInt(5,entity.getPropertyVersion());
                m_createEntity.setInt(6,entity.getIsAttribute() ? 1 : 0);
                m_createEntity.setInt(7,entity.getIsSymmetric() ? 1 : 0);
                m_createEntity.setInt(8,entity.getIsTransitive() ? 1 : 0);
                m_createEntity.setInt(9,entity.getInstanceVersion());
                m_createEntity.addBatch();
            }
            assertOneRecordUpdated(m_createEntity.executeBatch());
        }
        catch (SQLException e) {
            for (int i=0;i<entities.length;i++)
                entities[i].setEntityID(0);
            throw e;
        }
    }
    /**
     * Saves given entities.
     *
     * @param entities                      the OIModelEntity objects being saved
     * @throws SQLException                 thrown if there is an error
     */
    public void saveOIModelEntities(OIModelEntity[] entities) throws SQLException {
        if (m_saveEntity==null)
            m_saveEntity=getConnection().prepareStatement("UPDATE OIModelEntity SET conceptVersion=?,propertyVersion=?,isAttribute=?,isSymmetric=?,isTransitive=?,instanceVersion=? WHERE entityID=?");
        m_saveEntity.clearBatch();
        for (int i=0;i<entities.length;i++) {
            m_saveEntity.setInt(1,entities[i].getConceptVersion());
            m_saveEntity.setInt(2,entities[i].getPropertyVersion());
            m_saveEntity.setInt(3,entities[i].getIsAttribute() ? 1 : 0);
            m_saveEntity.setInt(4,entities[i].getIsSymmetric() ? 1 : 0);
            m_saveEntity.setInt(5,entities[i].getIsTransitive() ? 1 : 0);
            m_saveEntity.setInt(6,entities[i].getInstanceVersion());
            m_saveEntity.setInt(7,entities[i].getEntityID());
            m_saveEntity.addBatch();
        }
        assertOneRecordUpdated(m_saveEntity.executeBatch());
        for (int i=0;i<entities.length;i++)
            entities[i].resetObjectChanged();
    }
    /**
     * Deletes given entities.
     *
     * @param entities                      the OIModelEntity objects being deleted
     * @throws SQLException                 thrown if there is an error
     */
    public void deleteOIModelEntities(OIModelEntity[] entities) throws SQLException {
        if (m_deleteEntity==null)
            m_deleteEntity=getConnection().prepareStatement("DELETE FROM OIModelEntity WHERE entityID=?");
        m_deleteEntity.clearBatch();
        for (int i=0;i<entities.length;i++) {
            m_deleteEntity.setInt(1,entities[i].getEntityID());
            m_deleteEntity.addBatch();
        }
        assertOneRecordUpdated(m_deleteEntity.executeBatch());
        for (int i=0;i<entities.length;i++)
            entities[i].setEntityID(0);
    }
}
