package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import edu.unika.aifb.kaon.api.oimodel.KAONConnection;

/**
 * DAO for OIModel objects.
 */
public class OIModelDAO extends AbstractDAO {
    /** The statement for selecting OIModel IDs. */
    protected PreparedStatement m_selectOIModelID;
    /** The statement for selecting logical URI of the model. */
    protected PreparedStatement m_selectOIModelLogicalURI;
    /** The statement for creating OIModel objects. */
    protected PreparedStatement m_createOIModel;
    /** The statement for adding included models. */
    protected PreparedStatement m_addIncludedOIModel;
    /** The statement for removing included models. */
    protected PreparedStatement m_removeIncludedOIModel;
    /** The statement for deleting all included models. */
    protected PreparedStatement m_deleteAllIncludedOIModels;
    /** The statement for inserting a record into the list of all included models. */
    protected PreparedStatement m_insertIntoAllIncludedOIModels;
    /** Selects included OIModels of some model. */
    protected PreparedStatement m_selectIncludedOIModels;
    /** Selects all included OIModels of some model. */
    protected PreparedStatement m_selectAllIncludedOIModels;
    /** Selects OIModels that include of some model. */
    protected PreparedStatement m_selectIncludingOIModels;
    /** Selects all OIModels. */
    protected PreparedStatement m_selectAllOIModels;
    /** Selects all information about an OI-model. */
    protected PreparedStatement m_selectAllOIModelInformation;
    /** Selects OIModel attributes. */
    protected PreparedStatement m_selectOIModelAttributes;
    /** Adds the OI-model attribute. */
    protected PreparedStatement m_addOIModelAttribute;
    /** Deletes the OI-model attribute. */
    protected PreparedStatement m_deleteOIModelAttribute;
    /** Changes the OI-model attribute. */
    protected PreparedStatement m_changeOIModelAttribute;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public OIModelDAO(EngineeringServerDAO engineeringServerDAO) {
        super(engineeringServerDAO);
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        closeStatement(m_selectOIModelID);
        closeStatement(m_selectOIModelLogicalURI);
        closeStatement(m_createOIModel);
        closeStatement(m_addIncludedOIModel);
        closeStatement(m_removeIncludedOIModel);
        closeStatement(m_deleteAllIncludedOIModels);
        closeStatement(m_insertIntoAllIncludedOIModels);
        closeStatement(m_selectIncludedOIModels);
        closeStatement(m_selectAllIncludedOIModels);
        closeStatement(m_selectIncludingOIModels);
        closeStatement(m_selectAllOIModels);
        closeStatement(m_selectAllOIModelInformation);
        closeStatement(m_selectOIModelAttributes);
        closeStatement(m_addOIModelAttribute);
        closeStatement(m_deleteOIModelAttribute);
        closeStatement(m_changeOIModelAttribute);
    }
    /**
     * Returns the ID of the OIModel with given logical URI.
     *
     * @param logicalURI                    the logical URI of the OIModel
     * @return                              the ID of the loaded OIModel
     * @throws SQLException                 thrown if there is an error
     */
    public int getOIModelID(String logicalURI) throws SQLException {
        if (m_selectOIModelID==null)
            m_selectOIModelID=getConnection().prepareStatement("SELECT modelID FROM OIModel WHERE logicalURI=?");
        m_selectOIModelID.setString(1,logicalURI);
        ResultSet resultSet=m_selectOIModelID.executeQuery();
        try {
            if (resultSet.next())
                return resultSet.getInt(1);
            else
                throw new SQLException("Cannot find an OIModel with logical URI '"+logicalURI+"'.");
        }
        finally {
            resultSet.close();
        }
    }
    /**
     * Returns the logical URI for given model ID.
     *
     * @param modelID                       the ID of the OIModel
     * @return                              the logical URI of the OIModel
     * @throws SQLException                 thrown if there is an error
     */
    public String getOIModelLogicalURI(int modelID) throws SQLException {
        if (m_selectOIModelLogicalURI==null)
            m_selectOIModelLogicalURI=getConnection().prepareStatement("SELECT logicalURI FROM OIModel WHERE modelID=?");
        m_selectOIModelLogicalURI.setInt(1,modelID);
        ResultSet resultSet=m_selectOIModelLogicalURI.executeQuery();
        try {
            if (resultSet.next())
                return resultSet.getString(1);
            else
                throw new SQLException("Cannot find an OIModel with model ID '"+modelID+"'.");
        }
        finally {
            resultSet.close();
        }
    }
    /**
     * Creates an OI-model with given URI and makes sure that the root OI-model is included as well.
     *
     * @param logicalURI                    the logical URI of the OIModel
     * @return                              the ID of the created OIModel
     * @throws SQLException                 thrown if there is an error
     */
    public int createOIModel(String logicalURI) throws SQLException {
        int modelID=createOIModelInternal(logicalURI);
        if (!KAONConnection.ROOT_OIMODEL_URI.equals(logicalURI)) {
            int rootModelID=getOIModelID(KAONConnection.ROOT_OIMODEL_URI);
            addIncludedOIModel(modelID,rootModelID);
        }
        else
            updateAllIncludedModels(modelID);
        return modelID;
    }
    /**
     * Creates an OI-model with given logical URI.
     *
     * @param logicalURI                    the logical URI of the OIModel
     * @return                              the ID of the created OIModel
     * @throws SQLException                 thrown if there is an error
     */
    protected int createOIModelInternal(String logicalURI) throws SQLException {
        int modelID=getNextCounter("oimodel");
        if (m_createOIModel==null)
            m_createOIModel=getConnection().prepareStatement("INSERT INTO OIModel (modelID,logicalURI) VALUES (?,?)");
        m_createOIModel.setInt(1,modelID);
        m_createOIModel.setString(2,logicalURI);
        assertOneRecordUpdated(m_createOIModel.executeUpdate());
        return modelID;
    }
    /**
     * Deletes given OIModel. Cascade delete constraints in the schema will make sure that all
     * entities within the model are deleted as well.
     *
     * @param modelID                       the ID of the model to be deleted
     * @throws SQLException                 thrown if there is an error
     */
    public void deleteOIModel(int modelID) throws SQLException {
        m_engineeringServerDAO.flush();
        int[] includingModelIDs=getIncludingModelIDs(modelID);
        executeUpdate("DELETE FROM IncludedOIModel WHERE includingModelID="+modelID);
        executeUpdate("DELETE FROM AllIncludedOIModels WHERE includingModelID="+modelID);
        executeUpdate("DELETE FROM OIModelAttribute WHERE modelID="+modelID);
        executeUpdate("DELETE FROM InverseProperty WHERE modelID="+modelID);
        executeUpdate("DELETE FROM ConceptHierarchy WHERE modelID="+modelID);
        executeUpdate("DELETE FROM ConceptInstance WHERE modelID="+modelID);
        executeUpdate("DELETE FROM PropertyHierarchy WHERE modelID="+modelID);
        executeUpdate("DELETE FROM PropertyDomain WHERE modelID="+modelID);
        executeUpdate("DELETE FROM PropertyRange WHERE modelID="+modelID);
        executeUpdate("DELETE FROM RelationInstance WHERE modelID="+modelID);
        executeUpdate("DELETE FROM AttributeInstance WHERE modelID="+modelID);
        executeUpdate("DELETE FROM OIModelEntity WHERE modelID="+modelID);
        assertOneRecordUpdated(executeUpdate("DELETE FROM OIModel WHERE modelID="+modelID));
        for (int i=0;i<includingModelIDs.length;i++)
            updateAllIncludedModels(includingModelIDs[i]);
    }
    /**
     * Adds an OIModel to the list of included models.
     *
     * @param modelID                       the ID of the model to which an included model is added
     * @param includedModelID               the ID of the model that is included
     * @throws SQLException                 thrown if there is an error
     */
    public void addIncludedOIModel(int modelID,int includedModelID) throws SQLException {
        m_engineeringServerDAO.flush();
        if (m_addIncludedOIModel==null)
            m_addIncludedOIModel=getConnection().prepareStatement("INSERT INTO IncludedOIModel (includingModelID,includedModelID) VALUES (?,?)");
        m_addIncludedOIModel.setInt(1,modelID);
        m_addIncludedOIModel.setInt(2,includedModelID);
        assertOneRecordUpdated(m_addIncludedOIModel.executeUpdate());
        updateAllIncludedModels(modelID);
    }
    /**
     * Adds an OIModel from the list of included models.
     *
     * @param modelID                       the ID of the model from which an included model is removed
     * @param includedModelID               the ID of the model that is removed
     * @throws SQLException                 thrown if there is an error
     */
    public void removeIncludedOIModel(int modelID,int includedModelID) throws SQLException {
        m_engineeringServerDAO.flush();
        if (m_removeIncludedOIModel==null)
            m_removeIncludedOIModel=getConnection().prepareStatement("DELETE FROM IncludedOIModel WHERE includingModelID=? AND includedModelID=?");
        m_removeIncludedOIModel.setInt(1,modelID);
        m_removeIncludedOIModel.setInt(2,includedModelID);
        assertOneRecordUpdated(m_removeIncludedOIModel.executeUpdate());
        updateAllIncludedModels(modelID);
    }
    /**
     * Returns the list of models directly included by some model.
     *
     * @param modelID                       the ID of the model requested
     * @return                              the array of included model IDs
     * @throws SQLException                 thrown if there is an error
     */
    public int[] getIncludedModelIDs(int modelID) throws SQLException {
        if (m_selectIncludedOIModels==null)
            m_selectIncludedOIModels=getConnection().prepareStatement("SELECT includedModelID FROM IncludedOIModel WHERE includingModelID=?");
        m_selectIncludedOIModels.setInt(1,modelID);
        ResultSet resultSet=m_selectIncludedOIModels.executeQuery();
        List modelIDs=new ArrayList();
        try {
            while (resultSet.next()) {
                int includedModelID=resultSet.getInt(1);
                modelIDs.add(new Integer(includedModelID));
            }
        }
        finally {
            resultSet.close();
        }
        int[] result=new int[modelIDs.size()];
        for (int i=0;i<modelIDs.size();i++)
            result[i]=((Integer)modelIDs.get(i)).intValue();
        return result;
    }
    /**
     * Returns the list of all models included by some model.
     *
     * @param modelID                       the ID of the model requested
     * @return                              the array of all included model IDs
     * @throws SQLException                 thrown if there is an error
     */
    public int[] getAllIncludedModelIDs(int modelID) throws SQLException {
        if (m_selectAllIncludedOIModels==null)
            m_selectAllIncludedOIModels=getConnection().prepareStatement("SELECT includedModelID FROM AllIncludedOIModels WHERE includingModelID=?");
        m_selectAllIncludedOIModels.setInt(1,modelID);
        ResultSet resultSet=m_selectAllIncludedOIModels.executeQuery();
        List modelIDs=new ArrayList();
        try {
            while (resultSet.next()) {
                int includedModelID=resultSet.getInt(1);
                modelIDs.add(new Integer(includedModelID));
            }
        }
        finally {
            resultSet.close();
        }
        int[] result=new int[modelIDs.size()];
        for (int i=0;i<modelIDs.size();i++)
            result[i]=((Integer)modelIDs.get(i)).intValue();
        return result;
    }
    /**
     * Returns the list of models that include some model.
     *
     * @param modelID                       the ID of the model requested
     * @return                              the array of model IDs that include given model
     * @throws SQLException                 thrown if there is an error
     */
    public int[] getIncludingModelIDs(int modelID) throws SQLException {
        if (m_selectIncludingOIModels==null)
            m_selectIncludingOIModels=getConnection().prepareStatement("SELECT includingModelID FROM IncludedOIModel WHERE includedModelID=?");
        m_selectIncludingOIModels.setInt(1,modelID);
        ResultSet resultSet=m_selectIncludingOIModels.executeQuery();
        List modelIDs=new ArrayList();
        try {
            while (resultSet.next()) {
                int includingModelID=resultSet.getInt(1);
                modelIDs.add(new Integer(includingModelID));
            }
        }
        finally {
            resultSet.close();
        }
        int[] result=new int[modelIDs.size()];
        for (int i=0;i<modelIDs.size();i++)
            result[i]=((Integer)modelIDs.get(i)).intValue();
        return result;
    }
    /**
     * Returns the list of all models.
     *
     * @return                              the array of all included model IDs
     * @throws SQLException                 thrown if there is an error
     */
    public int[] getAllOIModelIDs() throws SQLException {
        if (m_selectAllOIModels==null)
            m_selectAllOIModels=getConnection().prepareStatement("SELECT modelID FROM OIModel");
        ResultSet resultSet=m_selectAllOIModels.executeQuery();
        List modelIDs=new ArrayList();
        try {
            while (resultSet.next()) {
                int modelID=resultSet.getInt(1);
                modelIDs.add(new Integer(modelID));
            }
        }
        finally {
            resultSet.close();
        }
        int[] result=new int[modelIDs.size()];
        for (int i=0;i<modelIDs.size();i++)
            result[i]=((Integer)modelIDs.get(i)).intValue();
        return result;
    }
    /**
     * Returns all information about the OI-model.
     *
     * @return                              each row contains information about the model ID, logical URI following by the IDs of the included models
     * @throws SQLException                 thrown if there is an error
     */
    public Object[][] getAllOIModelInformation() throws SQLException {
        // The inner join is OK since each model has at least itself in AllIncludedOIModels.
        if (m_selectAllOIModelInformation==null)
            m_selectAllOIModelInformation=getConnection().prepareStatement("SELECT modelID,logicalURI,includedModelID FROM OIModel,AllIncludedOIModels WHERE modelID=includingModelID ORDER BY modelID");
        List rows=new ArrayList();
        List row=new ArrayList();
        // now artificially add the information about the Root OI-model
        int lastModelID=getOIModelID(KAONConnection.ROOT_OIMODEL_URI);
        row.add(new Integer(lastModelID));
        row.add(KAONConnection.ROOT_OIMODEL_URI);
        // now execute the query and process it
        ResultSet resultSet=m_selectAllOIModelInformation.executeQuery();
        try {
            while (resultSet.next()) {
                int modelID=resultSet.getInt(1);
                if (modelID!=lastModelID) {
                    rows.add(row.toArray());
                    row.clear();
                    row.add(new Integer(modelID));
                    String logicalURI=resultSet.getString(2);
                    row.add(logicalURI);
                    lastModelID=modelID;
                }
                int includedModelID=resultSet.getInt(3);
                row.add(new Integer(includedModelID));
            }
            rows.add(row.toArray());
        }
        finally {
            resultSet.close();
        }
        Object[][] result=new Object[rows.size()][];
        rows.toArray(result);
        return result;
    }
    /**
     * Creates a duplicate of supplied model.
     *
     * @param sourceLogicalURI              the logical URI of the source model
     * @param targetLogicalURI              the logical URI of the target model
     * @return                              the ID of the target model
     * @throws SQLException                 thrown if there is an error
     */
    public int createDuplicate(String sourceLogicalURI,String targetLogicalURI) throws SQLException {
        m_engineeringServerDAO.flush();
        int sourceModelID=getOIModelID(sourceLogicalURI);
        int targetModelID=createOIModelInternal(targetLogicalURI);
        executeUpdate("INSERT INTO IncludedOIModel (includingModelID,includedModelID) SELECT "+targetModelID+",includedModelID FROM IncludedOIModel WHERE includingModelID="+sourceModelID);
        updateAllIncludedModels(targetModelID);
        executeUpdate("INSERT INTO OIModelAttribute (modelID,attributeKey,attributeValue) SELECT "+targetModelID+",attributeKey,attributeValue FROM OIModelAttribute WHERE modelID="+sourceModelID);
        int baseNewEntityID=getNextCounter("oimodelentity");
        int entityIDOffset=baseNewEntityID;
        Statement statement=getConnection().createStatement();
        try {
            ResultSet resultSet=statement.executeQuery("SELECT min(entityID) FROM OIModelEntity WHERE modelID="+sourceModelID);
            try {
                if (resultSet.next())
                    entityIDOffset-=resultSet.getInt(1);
            }
            finally {
                resultSet.close();
            }
        }
        finally {
            statement.close();
        }
        int addedEntities=0;
        addedEntities+=executeUpdate("INSERT INTO OIModelEntity (entityID,modelID,entityURI,conceptVersion,propertyVersion,isAttribute,isSymmetric,isTransitive,instanceVersion) SELECT t.entityID+"+entityIDOffset+","+targetModelID+",t.entityURI,t.conceptVersion,t.propertyVersion,t.isAttribute,t.isSymmetric,t.isTransitive,t.instanceVersion FROM OIModelEntity t WHERE t.modelID="+sourceModelID);
        updateCounter("oimodelentity",baseNewEntityID+addedEntities-1);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"InverseProperty",new String[] { "propertyID","inversePropertyID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"ConceptHierarchy",new String[] { "superConceptID","subConceptID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"PropertyHierarchy",new String[] { "superPropertyID","subPropertyID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"PropertyDomain",new String[] { "propertyID","conceptID" },new String[] { "minimumCardinality","maximumCardinality" });
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"PropertyRange",new String[] { "propertyID","conceptID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"ConceptInstance",new String[] { "conceptID","instanceID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"RelationInstance",new String[] { "propertyID","sourceInstanceID","targetInstanceID" },null);
        copyEntities(sourceModelID,targetModelID,entityIDOffset,"AttributeInstance",new String[] { "propertyID","sourceInstanceID" },new String[] { "textValue" });
        return targetModelID;
    }
    /**
     * Executes all queries that copy data from the source model to the target model with taking into account the shifting of entity IDs.
     *
     * @param sourceModelID                 the ID of the source OI-model
     * @param targetModelID                 the ID of the target OI-model
     * @param entityIDOffset                the offset to add to transform the entity IDs
     * @param tableName                     the name of the table
     * @param fieldNames                    the array of field names
     * @param additionalFields              the array of additional fields
     * @throws SQLException                 thrown in case of error
     */
    protected void copyEntities(int sourceModelID,int targetModelID,int entityIDOffset,String tableName,String[] fieldNames,String[] additionalFields) throws SQLException {
        boolean[] translateID=new boolean[fieldNames.length];
        int steps=1;
        for (int i=1;i<=fieldNames.length;i++)
            steps*=2;
        for (int step=0;step<steps;step++) {
            int number=step;
            for (int i=0;i<fieldNames.length;i++) {
                translateID[i]=(number % 2)==0;
                number=number/2;
            }
            StringBuffer query=new StringBuffer();
            query.append("INSERT INTO ");
            query.append(tableName);
            query.append(" (modelID");
            for (int i=0;i<fieldNames.length;i++) {
                query.append(",");
                query.append(fieldNames[i]);
            }
            if (additionalFields!=null)
                for (int i=0;i<additionalFields.length;i++) {
                    query.append(",");
                    query.append(additionalFields[i]);
                }
            query.append(") SELECT ");
            query.append(targetModelID);
            for (int i=0;i<fieldNames.length;i++) {
                query.append(",");
                query.append(fieldNames[i]);
                if (translateID[i]) {
                    query.append("+");
                    query.append(entityIDOffset);
                }
            }
            if (additionalFields!=null)
                for (int i=0;i<additionalFields.length;i++) {
                    query.append(",");
                    query.append(additionalFields[i]);
                }
            query.append(" FROM ");
            query.append(tableName);
            query.append(" t");
            for (int i=0;i<fieldNames.length;i++) {
                query.append(",OIModelEntity e");
                query.append(i);
            }
            query.append(" WHERE t.modelID=");
            query.append(sourceModelID);
            for (int i=0;i<fieldNames.length;i++) {
                query.append(" AND ");
                query.append(fieldNames[i]);
                query.append("=e");
                query.append(i);
                query.append(".entityID AND e");
                query.append(i);
                query.append(".modelID");
                if (translateID[i])
                    query.append("=");
                else
                    query.append("<>");
                query.append(sourceModelID);
            }
            String queryText=query.toString();
            executeUpdate(queryText);
        }
    }
    /**
     * Updates the list of all included models for given model and recursively calls the same metod for
     * all models that include givne model.
     *
     * @param modelID                       the ID of the model being updated
     * @throws SQLException                 thrown if there is an error
     */
    protected void updateAllIncludedModels(int modelID) throws SQLException {
        if (m_deleteAllIncludedOIModels==null)
            m_deleteAllIncludedOIModels=getConnection().prepareStatement("DELETE FROM AllIncludedOIModels WHERE includingModelID=?");
        m_deleteAllIncludedOIModels.setInt(1,modelID);
        m_deleteAllIncludedOIModels.executeUpdate();
        if (m_insertIntoAllIncludedOIModels==null)
            m_insertIntoAllIncludedOIModels=getConnection().prepareStatement("INSERT INTO AllIncludedOIModels (includingModelID,includedModelID) VALUES (?,?)");
        Set allIncludedModels=new HashSet();
        allIncludedModels.add(new Integer(modelID));
        int[] includedModelIDs=getIncludedModelIDs(modelID);
        for (int i=0;i<includedModelIDs.length;i++) {
            allIncludedModels.add(new Integer(includedModelIDs[i]));
            int[] allIncludedModelIDs=getAllIncludedModelIDs(includedModelIDs[i]);
            for (int j=0;j<allIncludedModelIDs.length;j++)
                allIncludedModels.add(new Integer(allIncludedModelIDs[j]));
        }
        m_insertIntoAllIncludedOIModels.clearBatch();
        Iterator iterator=allIncludedModels.iterator();
        while (iterator.hasNext()) {
            Integer includedModelID=(Integer)iterator.next();
            m_insertIntoAllIncludedOIModels.setInt(1,modelID);
            m_insertIntoAllIncludedOIModels.setInt(2,includedModelID.intValue());
            m_insertIntoAllIncludedOIModels.addBatch();
        }
        m_insertIntoAllIncludedOIModels.executeBatch();
        int[] includingModelIDs=getIncludingModelIDs(modelID);
        for (int i=0;i<includingModelIDs.length;i++)
            updateAllIncludedModels(includingModelIDs[i]);
    }
    /**
     * Returns all OI-models of an OI-model.
     *
     * @param modelID                       the ID of the model whose attributes are required
     * @return                              the attributes of the model
     * @throws SQLException                 thrown if there is an error
     */
    public Map getOIModelAttributes(int modelID) throws SQLException {
        if (m_selectOIModelAttributes==null)
            m_selectOIModelAttributes=getConnection().prepareStatement("SELECT attributeKey,attributeValue FROM OIModelAttribute WHERE modelID=?");
        m_selectOIModelAttributes.setInt(1,modelID);
        Map result=new HashMap();
        ResultSet resultSet=m_selectOIModelAttributes.executeQuery();
        try {
            while (resultSet.next()) {
                String key=resultSet.getString(1);
                String value=resultSet.getString(2);
                result.put(key,value);
            }
            return result;
        }
        finally {
            resultSet.close();
        }
    }
    /**
     * Sets the attribute of the OI-model.
     *
     * @param modelID                       the ID of the model whose attribute is changed
     * @param key                           the key
     * @param value                         the value
     * @throws SQLException                 thrown if there is an error
     */
    public void setOIModelAttribute(int modelID,String key,String value) throws SQLException {
        if (value==null) {
            if (m_deleteOIModelAttribute==null)
                m_deleteOIModelAttribute=getConnection().prepareStatement("DELETE FROM OIModelAttribute WHERE modelID=? AND attributeKey=?");
            m_deleteOIModelAttribute.setInt(1,modelID);
            m_deleteOIModelAttribute.setString(2,key);
            m_deleteOIModelAttribute.executeUpdate();      // it is allowed to remove an attribute which doesn't exist
        }
        else {
            // first try to change the attribute
            if (m_changeOIModelAttribute==null)
                m_changeOIModelAttribute=getConnection().prepareStatement("UPDATE OIModelAttribute SET attributeValue=? WHERE modelID=? AND attributeKey=?");
            m_changeOIModelAttribute.setString(1,value);
            m_changeOIModelAttribute.setInt(2,modelID);
            m_changeOIModelAttribute.setString(3,key);
            if (m_changeOIModelAttribute.executeUpdate()!=1) {
                // now try to add the attribute
                if (m_addOIModelAttribute==null)
                    m_addOIModelAttribute=getConnection().prepareStatement("INSERT INTO OIModelAttribute (modelID,attributeKey,attributeValue) VALUES (?,?,?)");
                m_addOIModelAttribute.setInt(1,modelID);
                m_addOIModelAttribute.setString(2,key);
                m_addOIModelAttribute.setString(3,value);
                assertOneRecordUpdated(m_addOIModelAttribute.executeUpdate());
            }
        }
    }
}
