package edu.unika.aifb.kaon.engineeringserver.dao;

import java.sql.SQLException;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.DatabaseMetaData;

import edu.unika.aifb.kaon.api.oimodel.KAONConnection;
import edu.unika.aifb.kaon.api.vocabulary.KAONVocabularyAdaptor;

/**
 * Manager of all DAO objects for the engineering server.
 */
public class EngineeringServerDAO {
    /** The connection to the database. */
    protected Connection m_connection;
    /** The statement for selecting the counter of given type. */
    protected PreparedStatement m_selectCounter;
    /** The statement for updating the counter of given type. */
    protected PreparedStatement m_updateCounter;
    /** The statement for creating the counter of given type. */
    protected PreparedStatement m_createCounter;
    /** The object manager. */
    protected ObjectManager m_objectManager;
    /** The OIModel DAO. */
    protected OIModelDAO m_oimodelDAO;
    /** The OIModelEntity DAO. */
    protected OIModelEntityDAO m_oimodelEntityDAO;
    /** The InverseProperty DAO. */
    protected InversePropertyDAO m_inversePropertyDAO;
    /** The ConceptHierarchy DAO. */
    protected ConceptHierarchyDAO m_conceptHierarchyDAO;
    /** The ConceptInstance DAO. */
    protected ConceptInstanceDAO m_conceptInstanceDAO;
    /** The PropertyHierarchy DAO. */
    protected PropertyHierarchyDAO m_propertyHierarchyDAO;
    /** The PropertyDomain DAO. */
    protected PropertyDomainDAO m_propertyDomainDAO;
    /** The PropertyRange DAO. */
    protected PropertyRangeDAO m_propertyRangeDAO;
    /** The PropertyInstance DAO. */
    protected PropertyInstanceDAO m_propertyInstanceDAO;

    /**
     * Creates an instance of this class.
     *
     * @param connection                    the connection to the database
     */
    public EngineeringServerDAO(Connection connection) {
        m_connection=connection;
    }
    /**
     * Returns the new ID of given type.
     *
     * @param type                          the type of the counter
     * @param span                          the span
     * @return                              next counter of given type
     * @throws SQLException                 thrown in case of failure
     */
    public int getNextCounter(String type,int span) throws SQLException {
        if (m_selectCounter==null)
            m_selectCounter=m_connection.prepareStatement("SELECT counter FROM PKCounter WHERE type=?");
        m_selectCounter.setString(1,type);
        ResultSet resultSet=m_selectCounter.executeQuery();
        try {
            int currentCounter;
            if (resultSet.next()) {
                currentCounter=resultSet.getInt(1);
                updateCounter(type,currentCounter+span);
            }
            else {
                if (m_createCounter==null)
                    m_createCounter=m_connection.prepareStatement("INSERT INTO PKCounter (type,counter) VALUES (?,"+span+")");
                currentCounter=0;
                m_createCounter.setString(1,type);
                m_createCounter.executeUpdate();
            }
            return currentCounter+1;
        }
        finally {
            resultSet.close();
        }
    }
    /**
     * Sets the new value of the counter.
     *
     * @param type                          the type of the counter
     * @param newValue                      the new value of the counter
     * @throws SQLException                 thrown in case of failure
     */
    public void updateCounter(String type,int newValue) throws SQLException {
        if (m_updateCounter==null)
            m_updateCounter=m_connection.prepareStatement("UPDATE PKCounter SET counter=? WHERE type=?");
        m_updateCounter.setInt(1,newValue);
        m_updateCounter.setString(2,type);
        m_updateCounter.executeUpdate();
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        closeStatement(m_selectCounter);
        closeStatement(m_updateCounter);
        closeStatement(m_createCounter);
        closeDAO(m_oimodelDAO);
        closeDAO(m_oimodelEntityDAO);
        closeDAO(m_inversePropertyDAO);
        closeDAO(m_conceptHierarchyDAO);
        closeDAO(m_conceptInstanceDAO);
        closeDAO(m_propertyHierarchyDAO);
        closeDAO(m_propertyDomainDAO);
        closeDAO(m_propertyRangeDAO);
        closeDAO(m_propertyInstanceDAO);
        try {
            m_connection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
        }
        catch (SQLException ignored) {
        }
    }
    /**
     * Flushes the changes in all DAOs.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flush() throws SQLException {
        if (m_objectManager!=null)
            m_objectManager.flushCreateChange();
        if (m_inversePropertyDAO!=null)
            m_inversePropertyDAO.flush();
        if (m_conceptHierarchyDAO!=null)
            m_conceptHierarchyDAO.flush();
        if (m_conceptInstanceDAO!=null)
            m_conceptInstanceDAO.flush();
        if (m_propertyHierarchyDAO!=null)
            m_propertyHierarchyDAO.flush();
        if (m_propertyDomainDAO!=null)
            m_propertyDomainDAO.flush();
        if (m_propertyRangeDAO!=null)
            m_propertyRangeDAO.flush();
        if (m_propertyInstanceDAO!=null)
            m_propertyInstanceDAO.flush();
        if (m_objectManager!=null)
            m_objectManager.flushDelete();
    }
    /**
     * Closes given DAO if it is not <code>null</code>.
     *
     * @param dao                           the DAO being closed
     */
    protected void closeDAO(AbstractDAO dao) {
        if (dao!=null)
            dao.close();
    }
    /**
     * Closes given statement (if it is not equal to <code>null</code>) and ignores the exception.
     *
     * @param statement                     statement to close
     */
    protected void closeStatement(Statement statement) {
        if (statement!=null)
            try {
                statement.close();
            }
            catch (SQLException ignored) {
            }
    }
    /**
     * Returns the connection.
     *
     * @return                              the connection
     */
    public Connection getConnection() {
        return m_connection;
    }
    /**
     * Returns the object manager.
     *
     * @return                              the object manager
     * @throws SQLException                 throws if there is an error
     */
    public ObjectManager getObjectManager() throws SQLException {
        if (m_objectManager==null)
            m_objectManager=new ObjectManager(this);
        return m_objectManager;
    }
    /**
     * Returns the DAO for OIModel.
     *
     * @return                              the DAO for OIModel
     */
    public OIModelDAO getOIModelDAO() {
        if (m_oimodelDAO==null)
            m_oimodelDAO=new OIModelDAO(this);
        return m_oimodelDAO;
    }
    /**
     * Returns the DAO for OIModelEntity.
     *
     * @return                              the DAO for OIModelEntity
     */
    public OIModelEntityDAO getOIModelEntityDAO() {
        if (m_oimodelEntityDAO==null)
            m_oimodelEntityDAO=new OIModelEntityDAO(this);
        return m_oimodelEntityDAO;
    }
    /**
     * Returns the DAO for ConceptHierarchy.
     *
     * @return                              the DAO for ConceptHierarchy
     */
    public ConceptHierarchyDAO getConceptHierarchyDAO() {
        if (m_conceptHierarchyDAO==null)
            m_conceptHierarchyDAO=new ConceptHierarchyDAO(this);
        return m_conceptHierarchyDAO;
    }
    /**
     * Returns the DAO for InverseProperty.
     *
     * @return                              the DAO for InverseProperty
     */
    public InversePropertyDAO getInversePropertyDAO() {
        if (m_inversePropertyDAO==null)
            m_inversePropertyDAO=new InversePropertyDAO(this);
        return m_inversePropertyDAO;
    }
    /**
     * Returns the DAO for ConceptInstance.
     *
     * @return                              the DAO for ConceptInstance
     */
    public ConceptInstanceDAO getConceptInstanceDAO() {
        if (m_conceptInstanceDAO==null)
            m_conceptInstanceDAO=new ConceptInstanceDAO(this);
        return m_conceptInstanceDAO;
    }
    /**
     * Returns the DAO for PropertyHierarchy.
     *
     * @return                              the DAO for PropertyHierarchy
     */
    public PropertyHierarchyDAO getPropertyHierarchyDAO() {
        if (m_propertyHierarchyDAO==null)
            m_propertyHierarchyDAO=new PropertyHierarchyDAO(this);
        return m_propertyHierarchyDAO;
    }
    /**
     * Returns the DAO for PropertyDomain.
     *
     * @return                              the DAO for PropertyDomain
     */
    public PropertyDomainDAO getPropertyDomainDAO() {
        if (m_propertyDomainDAO==null)
            m_propertyDomainDAO=new PropertyDomainDAO(this);
        return m_propertyDomainDAO;
    }
    /**
     * Returns the DAO for PropertyRange.
     *
     * @return                              the DAO for PropertyRange
     */
    public PropertyRangeDAO getPropertyRangeDAO() {
        if (m_propertyRangeDAO==null)
            m_propertyRangeDAO=new PropertyRangeDAO(this);
        return m_propertyRangeDAO;
    }
    /**
     * Returns the DAO for PropertyInstance.
     *
     * @return                              the DAO for PropertyInstance
     */
    public PropertyInstanceDAO getPropertyInstanceDAO() {
        if (m_propertyInstanceDAO==null)
            m_propertyInstanceDAO=new PropertyInstanceDAO(this);
        return m_propertyInstanceDAO;
    }
    /**
     * Makes sure that default models are created.
     */
    public void createDefaultModels() {
        try {
            int rootModel=getOIModelDAO().createOIModel(KAONConnection.ROOT_OIMODEL_URI);
            OIModelEntity entity=new OIModelEntity(rootModel,KAONVocabularyAdaptor.INSTANCE.getRoot());
            entity.setConceptVersion(1);
            getOIModelEntityDAO().createOIModelEntities(new OIModelEntity[] { entity } );
        }
        catch (SQLException modelExists) {
        }
    }
    /**
     * Returns the type of the DAO that must be created for given connection.
     *
     * @param connection                        the connection
     * @return                                  the type of the DAO
     */
    public static int getDAOType(Connection connection) {
        if (isOracle(connection))
            return 1;
        else
            return 0;
    }
    /**
     * Creates a DAO of given type.
     *
     * @param connection                        the connection
     * @param daoType                           the type of the DAO
     * @return                                  the DAO of given type
     * @throws SQLException                     thrown in case of error
     */
    public static EngineeringServerDAO createDAO(Connection connection,int daoType) throws SQLException {
        if (daoType!=1)
            connection.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
        return new EngineeringServerDAO(connection);
    }
    /**
     * Returns <code>true</code> if the connection is to an Oracle database.
     *
     * @param connection                        the connection
     * @return                                  <code>true</code> if this is Oracle
     */
    public static boolean isOracle(Connection connection) {
        try {
            DatabaseMetaData databaseMetaData=connection.getMetaData();
            return "Oracle".equalsIgnoreCase(databaseMetaData.getDatabaseProductName());
        }
        catch (Throwable ignored) {
            // Oracle shouldn't throw an error here
            return false;
        }
    }
}
