package edu.unika.aifb.kaon.engineeringserver.dao;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * Abstract base class for all DAO objects that manage relationships in the model.
 */
public abstract class AbstractRelationshipDAO extends AbstractDAO {
    /** The statement for adding relationships. */
    protected PreparedStatement m_addRelationships;
    /** The statement for removing relationships. */
    protected PreparedStatement m_removeRelationships;
    /** The set of all added relationships. */
    protected Set m_addedRelationships;
    /** The set of all removed relationships. */
    protected Set m_removedRelationships;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public AbstractRelationshipDAO(EngineeringServerDAO engineeringServerDAO) {
        super(engineeringServerDAO);
        m_addedRelationships=new HashSet();
        m_removedRelationships=new HashSet();
    }
    /**
     * Closes this DAO.
     */
    public void close() {
        closeStatement(m_addRelationships);
        closeStatement(m_removeRelationships);
    }
    /**
     * Flushes the additions from this DAO.
     *
     * @throws SQLException                 thrown if there is an error
     */
    protected void flushAdditions() throws SQLException {
        if (!m_addedRelationships.isEmpty()) {
            if (m_addRelationships==null)
                m_addRelationships=getConnection().prepareStatement(getAddRelationshipsSQL());
            m_addRelationships.clearBatch();
            Iterator iterator=m_addedRelationships.iterator();
            while (iterator.hasNext()) {
                Relationship relationship=(Relationship)iterator.next();
                m_addRelationships.setInt(1,relationship.m_modelID);
                m_addRelationships.setInt(2,relationship.m_entity1.getEntityID());
                m_addRelationships.setInt(3,relationship.m_entity2.getEntityID());
                m_addRelationships.addBatch();
            }
            assertOneRecordUpdated(m_addRelationships.executeBatch());
            m_addedRelationships.clear();
        }
    }
    /**
     * Flushes the removals from this DAO.
     *
     * @throws SQLException                 throws if there is an error
     */
    protected void flushRemovals() throws SQLException {
        if (!m_removedRelationships.isEmpty()) {
            if (m_removeRelationships==null)
                m_removeRelationships=getConnection().prepareStatement(getRemoveRelationshipsSQL());
            m_removeRelationships.clearBatch();
            Iterator iterator=m_removedRelationships.iterator();
            while (iterator.hasNext()) {
                Relationship relationship=(Relationship)iterator.next();
                m_removeRelationships.setInt(1,relationship.m_modelID);
                m_removeRelationships.setInt(2,relationship.m_entity1.getEntityID());
                m_removeRelationships.setInt(3,relationship.m_entity2.getEntityID());
                m_removeRelationships.addBatch();
            }
            assertOneRecordUpdated(m_removeRelationships.executeBatch());
            m_removedRelationships.clear();
        }
    }
    /**
     * Flushes this DAO.
     *
     * @throws SQLException                 thrown if there is an error
     */
    public void flush() throws SQLException {
        flushRemovals();
        flushAdditions();
    }
    /**
     * Adds a relationsip.
     *
     * @param modelID                       the ID of the model
     * @param entity1                       the first entity
     * @param entity2                       the second entity
     * @throws SQLException
     */
    protected void addRelationship(int modelID,OIModelEntity entity1,OIModelEntity entity2) throws SQLException {
        Relationship relationship=new Relationship(modelID,entity1,entity2);
        if (m_addedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+entity1.getEntityURI()+", "+entity2.getEntityURI()+" already added.");
        if (m_removedRelationships.contains(relationship))
            m_engineeringServerDAO.flush();
        m_addedRelationships.add(relationship);
    }
    /**
     * Removes a relationsip.
     *
     * @param modelID                       the ID of the model
     * @param entity1                       the first entity
     * @param entity2                       the second entity
     * @throws SQLException
     */
    protected void removeRelationship(int modelID,OIModelEntity entity1,OIModelEntity entity2) throws SQLException {
        Relationship relationship=new Relationship(modelID,entity1,entity2);
        if (m_removedRelationships.contains(relationship))
            throw new SQLException("Relationship "+modelID+", "+entity1.getEntityURI()+", "+entity2.getEntityURI()+" already removed.");
        if (m_addedRelationships.contains(relationship))
            m_engineeringServerDAO.flush();
        m_removedRelationships.add(relationship);
    }
    /**
     * The SQL statement for creating relationships.
     *
     * @return                              the SQL statement for creating relationships
     */
    protected abstract String getAddRelationshipsSQL();
    /**
     * The SQL statement for removing relationships.
     *
     * @return                              the SQL statement for removing relationships
     */
    protected abstract String getRemoveRelationshipsSQL();

    /**
     * Represents the key of a ternary relationship between a model, source entity and a target entity.
     */
    protected static class Relationship {
        public int m_modelID;
        public OIModelEntity m_entity1;
        public OIModelEntity m_entity2;
        protected int m_hashCode;

        public Relationship(int modelID,OIModelEntity entity1,OIModelEntity entity2) {
            m_modelID=modelID;
            m_entity1=entity1;
            m_entity2=entity2;
            m_hashCode=m_modelID+7*(m_entity1.hashCode()+7*m_entity2.hashCode());
        }
        public int hashCode() {
            return m_hashCode;
        }
        public boolean equals(Object that) {
            if (this==that)
                return true;
            if (!(that instanceof Relationship))
                return false;
            Relationship thatRelationship=(Relationship)that;
            return m_modelID==thatRelationship.m_modelID && m_entity1==thatRelationship.m_entity1 && m_entity2==thatRelationship.m_entity2;
        }
    }
}
