package edu.unika.aifb.kaon.engineeringserver.dao;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.SQLException;

/**
 * Abstract base class for all DAO objects.
 */
public abstract class AbstractDAO {
    /** The engineering server for this DAO. */
    protected EngineeringServerDAO m_engineeringServerDAO;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO          the engineering server DAO
     */
    public AbstractDAO(EngineeringServerDAO engineeringServerDAO) {
        m_engineeringServerDAO=engineeringServerDAO;
    }
    /**
     * Returns the new ID of given type.
     *
     * @param type                          the type of the counter
     * @return                              next counter of given type
     * @throws SQLException                 thrown in case there is an error
     */
    public int getNextCounter(String type) throws SQLException {
        return getNextCounter(type,1);
    }
    /**
     * Returns the new ID of given type.
     *
     * @param type                          the type of the counter
     * @param span                          the span how much to reserve
     * @return                              next counter of given type
     * @throws SQLException                 thrown in case there is an error
     */
    public int getNextCounter(String type,int span) throws SQLException {
        return m_engineeringServerDAO.getNextCounter(type,span);
    }
    /**
     * Updates the counter.
     *
     * @param type                          the type of the counter
     * @param newValue                      the new value of the counter
     * @throws SQLException                 thrown in case of failure
     */
    public void updateCounter(String type,int newValue) throws SQLException {
        m_engineeringServerDAO.updateCounter(type,newValue);
    }
    /**
     * Returns the database connection.
     *
     * @return                              the database connection
     */
    public Connection getConnection() {
        return m_engineeringServerDAO.getConnection();
    }
    /**
     * Closes given statement (if it is not equal to <code>null</code>) and ignores the exception.
     *
     * @param statement                     statement to close
     */
    protected void closeStatement(Statement statement) {
        if (statement!=null)
            try {
                statement.close();
            }
            catch (SQLException ignored) {
            }
    }
    /**
     * Checks whether an update has update exactly one record and throws an exception if this is not the case.
     *
     * @param rowCount                      the row count of the update command
     * @throws SQLException                 thrown if row count is not equal to 1
     */
    protected void assertOneRecordUpdated(int rowCount) throws SQLException {
        if (rowCount!=1 && rowCount!=Statement.SUCCESS_NO_INFO)
            throw new SQLException("Query should update exactly one row.");
    }
    /**
     * Checks whether an update has update exactly one record and throws an exception if this is not the case.
     *
     * @param rowCounts                     the row counts of the update command
     * @throws SQLException                 thrown if row count is not equal to 1
     */
    protected void assertOneRecordUpdated(int[] rowCounts) throws SQLException {
        for (int i=0;i<rowCounts.length;i++)
            if (rowCounts[i]!=1 && rowCounts[i]!=Statement.SUCCESS_NO_INFO)
                throw new SQLException("Query should update exactly one row.");
    }
    /**
     * Executes an update statement.
     *
     * @param sql                           the statement to execute
     * @return                              the number of updated rows
     * @throws SQLException                 thrown if there is an error with executing the update
     */
    protected int executeUpdate(String sql) throws SQLException {
        Statement statement=getConnection().createStatement();
        try {
            return statement.executeUpdate(sql);
        }
        finally {
            statement.close();
        }
    }
    /**
     * Returns the maximum number of in literals in a query.
     *
     * @return                              the maximum number of in literals in a query
     */
    protected int getQueryMaximumInLiterals() {
        return 200;
    }
    /**
     * Returns the maximum number of URIs that are loaded using IN literals.
     *
     * @return                              the maximum number of URIs that are loaded using IN literals
     */
    protected int getMaximumURIsToLoadUsingINLiterals() {
        return 400;
    }
    /**
     * Closes all statements opened by this DAO.
     */
    public abstract void close();
}
