package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.Hashtable;
import java.util.Collections;
import java.security.Principal;
import javax.security.auth.Subject;
import java.net.URI;
import java.net.URISyntaxException;
import java.rmi.RemoteException;
import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.jboss.security.SecurityAssociation;
import org.jboss.security.SimplePrincipal;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

import edu.unika.aifb.kaon.engineeringserver.interfaces.*;

/**
 * Implements a KAON connection for remote engineering server.
 */
public class RemoteKAONConnection extends AbstractKAONConnection {
    /** The URI of the server that this connection is serving. */
    public static final String SERVER_URI="SERVER_URI";
    /** The user name. */
    public static final String USER_NAME="USER_NAME";
    /** The password. */
    public static final String PASSWORD="PASSWORD";
    /** Specifies whether to listen for changes. */
    public static final String CHANGE_LISTENER="CHANGE_LISTENER";

    static {
        KAONManager.registerPhysicalURIToDefaultParametersResolver(new PhysicalURIResolver());
        SecurityAssociation.setServer();
    }

    /** The engineering server bean. */
    protected EngineeringServer m_engineeringServer;
    /** The change listener of this connection. */
    protected ChangeListener m_changeListener;
    /** The principal of the connection. */
    protected Principal m_principal;
    /** The credentials of the connection. */
    protected Object m_credential;
    /** The subject of the connection. */
    protected Subject m_subject;

    /**
     * Creates an instance of this class and connects it to given JBoss server.
     *
     * @param host                              the host name of the JBoss server
     * @param port                              the port number of the JBoss server
     * @throws KAONException                    thrown if there is an error
     */
    public RemoteKAONConnection(String host,int port) throws KAONException {
        Map parameters=new HashMap();
        parameters.put(SERVER_URI,"jboss://"+host+":"+port);
        initialize(parameters);
    }
    /**
     * Creates an instance of this class and connects it to given JBoss server.
     *
     * @param host                              the host name of the JBoss server
     * @param port                              the port number of the JBoss server
     * @param userName                          the name of the user
     * @param password                          the password
     * @throws KAONException                    thrown if there is an error
     */
    public RemoteKAONConnection(String host,int port,String userName,String password) throws KAONException {
        Map parameters=new HashMap();
        parameters.put(SERVER_URI,"jboss://"+userName+"@"+host+":"+port);
        parameters.put(PASSWORD,password);
        initialize(parameters);
    }
    /**
     * Creates an instance of this class from a map of parameters.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if there is an error
     */
    public RemoteKAONConnection(Map parameters) throws KAONException {
        initialize(parameters);
    }
    /**
     * Initializes this connection.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if there is an error
     */
    protected void initialize(Map parameters) throws KAONException {
        super.initialize(parameters);
        String serverURI=(String)m_parameters.get(SERVER_URI);
        if (serverURI==null)
            throw new KAONException("SERVER_URI parameter must be specified.");
        try {
            m_serverURI=new URI(serverURI);
        }
        catch (URISyntaxException e) {
            throw new KAONException("SERVER_URI has invalid syntax.",e);
        }
        String scheme=m_serverURI.getScheme();
        if (!"jboss".equals(scheme) && !"jndi".equals(scheme))
            throw new KAONException("Scheme "+scheme+" is not supported by RemoteKAONConnection");
        String userName=(String)m_parameters.get(USER_NAME);
        if (userName==null)
            userName=m_serverURI.getUserInfo();
        if (userName!=null) {
            String password=(String)m_parameters.get(PASSWORD);
            if (password==null)
                throw new KAONException("If user name is specified, then the password must be specified as well.");
            m_principal=new SimplePrincipal(userName);
            m_credential=password.toCharArray();
            m_subject=new Subject(false,Collections.singleton(m_principal),Collections.singleton(m_credential),Collections.EMPTY_SET);
        }
        m_engineeringServer=createEngineeringServer();
        String changeListener=(String)m_parameters.get(CHANGE_LISTENER);
        boolean swingThreadChangeListener="swingthread".equals(changeListener);
        if ("true".equalsIgnoreCase(changeListener) || swingThreadChangeListener) {
            Hashtable jndiProperties=null;
            if ("jboss".equals(m_serverURI.getScheme()))
                jndiProperties=getJBossJNDIParameters(m_serverURI.getHost(),m_serverURI.getPort());
            m_changeListener=new ChangeListener(this,swingThreadChangeListener,jndiProperties);
        }
    }
    /**
     * Closes this connection.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public void close() throws KAONException {
        // The locking order between change listener and KAONConnection is important. One must acquire a lock on the listener before
        // one acquires a lock on the connection. Otherwise, it can happen that the listener is processing a message (and thus a lock inside
        // JBoss is acquired), but we want to close a connection (thus requiring a lock on the connection).
        if (m_changeListener!=null) {
            synchronized (m_changeListener) {
                synchronized (this) {
                    super.close();
                    m_oimodels=null;
                    m_engineeringServer=null;
                    m_changeListener.close();
                    m_changeListener=null;
                }
            }
        }
        else {
            synchronized (this) {
                super.close();
                m_oimodels=null;
                m_engineeringServer=null;
            }
        }
    }
    /**
     * Returns a child connection. This can be used to implement advanced features, such as connection pooling.
     *
     * @return                                  a connection that is the child of this connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized KAONConnection getConnection() throws KAONException {
        return new RemoteKAONConnection(m_parameters);
    }
    /**
     * Creates an OI-model with given physical and logical URIs.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized OIModel createOIModel(String physicalURI,String logicalURI) throws KAONException {
        try {
            physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
            checkCreatingPhysicalURI(physicalURI,logicalURI);
            int modelID=getEngineeringServer().createOIModel(logicalURI);
            return getOIModel(modelID);
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Creates a new OI-model with supplied logical and physical URI and makes it an exact duplicate of supplied OI-model.
     *
     * @param sourceOIModel             the source OI-model
     * @param physicalURI               the physical URI of the new model
     * @param logicalURI                the logical URI of the new model
     * @return                          newly created OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized OIModel createDuplicate(OIModel sourceOIModel,String physicalURI,String logicalURI) throws KAONException {
        try {
            physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
            checkCreatingPhysicalURI(physicalURI,logicalURI);
            int modelID=getEngineeringServer().createDuplicate(sourceOIModel.getLogicalURI(),logicalURI);
            return getOIModel(modelID);
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Returns the set of logical URIs of all OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized Set getAllOIModelLogicalURIs() throws KAONException {
        try {
            Object[][] information=getEngineeringServer().getAllOIModelInformation();
            Set result=new HashSet();
            for (int i=0;i<information.length;i++)
                result.add(information[i][1]);
            return result;
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Creates the object of the engineering server bean object.
     *
     * @return                                  the engineering server bean object
     * @throws KAONException                    thrown if there is an error
     */
    protected EngineeringServer createEngineeringServer() throws KAONException {
        try {
            setSecurityAssociation();
            if ("jndi".equals(m_serverURI.getScheme()))
                return EngineeringServerUtil.getHome().create();
            else {
                Hashtable jndiProperties=getJBossJNDIParameters(m_serverURI.getHost(),m_serverURI.getPort());
                return EngineeringServerUtil.getHome(jndiProperties).create();
            }
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
        catch (NamingException e) {
            throw new KAONException("Cannot locate remote objects",e);
        }
        catch (CreateException e) {
            throw new KAONException("Error creating engineering server",e);
        }
    }
    /**
     * Returns the model ID for logical URI.
     *
     * @param logicalURI                        the logical URI
     * @return                                  the model ID of model with given URI
     * @throws KAONException                    thrown if there is an error
     */
    protected int getModelIDForLogicalURI(String logicalURI) throws KAONException {
        try {
            return getEngineeringServer().getOIModelIDForLogicalURI(logicalURI);
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Create the OI-model source for this connection.
     *
     * @param serverURI                         the URI of the server
     * @param modelID                           the ID of the model
     * @return                                  the OI-model source for this connection
     * @throws KAONException                    thrown if OI-model source cannot be created
     */
    protected OIModelSource createOIModelSource(String serverURI,int modelID) throws KAONException {
        return new RemoteEngineeringSource(serverURI,modelID,this);
    }
    /**
     * Returns the IDs of included models.
     *
     * @param modelID                           the ID of the model
     * @return                                  the array of included models
     * @throws KAONException                    thrown if there is an error
     */
    protected int[] getIncludedModelIDs(int modelID) throws KAONException {
        try {
            return getEngineeringServer().getDirectlyIncludedOIModelIDs(modelID);
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Returns the IDs of all OI-models in the database.
     *
     * @return                                  the array of IDs of all OI-models in the database
     * @throws KAONException                    thrown of there is an error
     */
    protected int[] getAllOIModelIDs() throws KAONException {
        try {
            return getEngineeringServer().getAllOIModelIDs();
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Executes an update of the model.
     *
     * @param changeList                        list of changes to apply
     * @return                                  object containing information about the update
     * @throws KAONException                    thrown if there is a problem with fetching or updating information
     */
    protected UpdateResultHolder updateOIModel(List changeList) throws KAONException {
        try {
            return getEngineeringServer().updateOIModel(m_connectionIdentifier,changeList);
        }
        catch (RemoteException e) {
            throw new KAONException("Error in remote call",e);
        }
    }
    /**
     * Prepares the execution context and returns the engineering server.
     *
     * @return                                  the engineering server
     */
    public synchronized EngineeringServer getEngineeringServer() {
        setSecurityAssociation();
        return m_engineeringServer;
    }
    /**
     * Sets the correct security association.
     */
    protected void setSecurityAssociation() {
        if (m_subject!=null) {
            SecurityAssociation.setPrincipal(m_principal);
            SecurityAssociation.setCredential(m_credential);
            SecurityAssociation.setSubject(m_subject);
        }
    }
    /**
     * Creates a hashtable with JNDI properties for connecting to JBoss server.
     *
     * @param server                        the name of the JBoss server
     * @param port                          the port of the JBoss server
     * @return                              the hashtable with parameters
     */
    public static Hashtable getJBossJNDIParameters(String server,int port) {
        if (port==-1)
            port=1099;
        Hashtable jndiProperties=new Hashtable();
        jndiProperties.put("java.naming.factory.initial","org.jnp.interfaces.NamingContextFactory");
        jndiProperties.put("java.naming.provider.url","jnp://"+server+":"+port);
        jndiProperties.put("java.naming.factory.url.pkgs","org.jboss.naming:org.jnp.interfaces");
        return jndiProperties;
    }

    /**
     * The resolver for RDF physical URIs.
     */
    protected static class PhysicalURIResolver implements PhysicalURIToDefaultParametersResolver {
        public Map getDefaultParametersForPhysicalURI(String physicalURI,Map contextParameters) {
            try {
                URI uri=new URI(physicalURI);
                String scheme=uri.getScheme();
                if ("jboss".equals(scheme) || "jndi".equals(scheme)) {
                    URI serverURI=new URI(uri.getScheme(),uri.getUserInfo(),uri.getHost(),uri.getPort(),null,null,null);
                    Map parameters=new HashMap();
                    parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.engineeringserver.client.RemoteKAONConnection");
                    parameters.put(SERVER_URI,serverURI.toString());
                    if (contextParameters!=null) {
                        URI contextServerURI=new URI((String)contextParameters.get(SERVER_URI));
                        if (contextServerURI.equals(serverURI)) {
                            if (contextParameters.containsKey(CHANGE_LISTENER))
                                parameters.put(CHANGE_LISTENER,contextParameters.get(CHANGE_LISTENER));
                            if (contextParameters.containsKey(USER_NAME))
                                parameters.put(USER_NAME,contextParameters.get(USER_NAME));
                            if (contextParameters.containsKey(PASSWORD))
                                parameters.put(PASSWORD,contextParameters.get(PASSWORD));
                        }
                    }
                    if ((parameters.containsKey(USER_NAME) || serverURI.getUserInfo()!=null) && !parameters.containsKey(PASSWORD))
                        parameters.put(PASSWORD,"???");
                    return parameters;
                }
            }
            catch (URISyntaxException e) {
            }
            return null;
        }
    }
}
