package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.Collections;
import java.security.Principal;
import javax.security.auth.Subject;
import java.net.URI;
import java.net.URISyntaxException;
import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.jboss.security.SecurityAssociation;
import org.jboss.security.SimplePrincipal;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

import edu.unika.aifb.kaon.engineeringserver.interfaces.*;

/**
 * Implements a KAON connection for local engineering server.
 */
public class LocalKAONConnection extends AbstractKAONConnection {
    /** The user name. */
    public static final String USER_NAME="USER_NAME";
    /** The password. */
    public static final String PASSWORD="PASSWORD";
    /** Specifies whether to listen for changes. */
    public static final String CHANGE_LISTENER="CHANGE_LISTENER";

    static {
        KAONManager.registerPhysicalURIToDefaultParametersResolver(new PhysicalURIResolver());
        SecurityAssociation.setServer();
    }

    /** The engineering server bean. */
    protected EngineeringServerLocal m_engineeringServer;
    /** The change listener of this connection. */
    protected ChangeListener m_changeListener;
    /** The principal of the connection. */
    protected Principal m_principal;
    /** The credentials of the connection. */
    protected Object m_credential;
    /** The subject of the connection. */
    protected Subject m_subject;

    /**
     * Creates an instance of this class from a map of parameters.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if there is an error
     */
    public LocalKAONConnection(Map parameters) throws KAONException {
        initialize(parameters);
    }
    /**
     * Initializes this connection.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if there is an error
     */
    protected void initialize(Map parameters) throws KAONException {
        super.initialize(parameters);
        try {
            m_serverURI=new URI("local://localhost");
        }
        catch (URISyntaxException cantHappen) {
        }
        String userName=(String)m_parameters.get(USER_NAME);
        if (userName!=null) {
            String password=(String)m_parameters.get(PASSWORD);
            if (password==null)
                throw new KAONException("If user name is specified, then the password must be specified as well.");
            m_principal=new SimplePrincipal(userName);
            m_credential=password.toCharArray();
            m_subject=new Subject(false,Collections.singleton(m_principal),Collections.singleton(m_credential),Collections.EMPTY_SET);
        }
        setSecurityAssociation();
        m_engineeringServer=createEngineeringServer();
        String changeListener=(String)m_parameters.get(CHANGE_LISTENER);
        boolean swingThreadChangeListener="swingthread".equals(changeListener);
        if ("true".equalsIgnoreCase(changeListener) || swingThreadChangeListener)
            m_changeListener=new ChangeListener(this,swingThreadChangeListener,null);
    }
    /**
     * Closes this connection.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        super.close();
        m_oimodels=null;
        m_engineeringServer=null;
        if (m_changeListener!=null)
            m_changeListener.close();
        m_changeListener=null;
    }
    /**
     * Returns a child connection. This can be used to implement advanced features, such as connection pooling.
     *
     * @return                                  a connection that is the child of this connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized KAONConnection getConnection() throws KAONException {
        return new LocalKAONConnection(m_parameters);
    }
    /**
     * Creates an OI-model with given physical and logical URIs.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the created OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized OIModel createOIModel(String physicalURI,String logicalURI) throws KAONException {
        physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
        checkCreatingPhysicalURI(physicalURI,logicalURI);
        int modelID=getEngineeringServer().createOIModel(logicalURI);
        return getOIModel(modelID);
    }
    /**
     * Creates a new OI-model with supplied logical and physical URI and makes it an exact duplicate of supplied OI-model.
     *
     * @param sourceOIModel                     the source OI-model
     * @param physicalURI                       the physical URI of the new model
     * @param logicalURI                        the logical URI of the new model
     * @return                                  newly created OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized OIModel createDuplicate(OIModel sourceOIModel,String physicalURI,String logicalURI) throws KAONException {
        physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
        checkCreatingPhysicalURI(physicalURI,logicalURI);
        int modelID=getEngineeringServer().createDuplicate(sourceOIModel.getLogicalURI(),logicalURI);
        return getOIModel(modelID);
    }
    /**
     * Returns the set of logical URIs of all OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized Set getAllOIModelLogicalURIs() throws KAONException {
        Object[][] information=getEngineeringServer().getAllOIModelInformation();
        Set result=new HashSet();
        for (int i=0;i<information.length;i++)
            result.add(information[i][1]);
        return result;
    }
    /**
     * Creates the object of the engineering server bean object.
     *
     * @return                              the engineering server bean object
     * @throws KAONException                    thrown if there is an error
     */
    protected EngineeringServerLocal createEngineeringServer() throws KAONException {
        try {
            return EngineeringServerUtil.getLocalHome().create();
        }
        catch (NamingException e) {
            throw new KAONException("Cannot locate local objects",e);
        }
        catch (CreateException e) {
            throw new KAONException("Error creating engineering server",e);
        }
    }
    /**
     * Returns the model ID for logical URI.
     *
     * @param logicalURI                        the logical URI
     * @return                                  the model ID of model with given URI
     * @throws KAONException                    thrown if there is an error
     */
    protected int getModelIDForLogicalURI(String logicalURI) throws KAONException {
        return getEngineeringServer().getOIModelIDForLogicalURI(logicalURI);
    }
    /**
     * Create the OI-model source for this connection.
     *
     * @param serverURI                         the URI of the server
     * @param modelID                           the ID of the model
     * @return                                  the OI-model source for this connection
     * @throws KAONException                    thrown if OI-model source cannot be created
     */
    protected OIModelSource createOIModelSource(String serverURI,int modelID) throws KAONException {
        return new LocalEngineeringSource(serverURI,modelID,this);
    }
    /**
     * Returns the IDs of included models.
     *
     * @param modelID                           the ID of the model
     * @return                                  the array of included models
     * @throws KAONException                    thrown if there is an error
     */
    protected int[] getIncludedModelIDs(int modelID) throws KAONException {
        return getEngineeringServer().getDirectlyIncludedOIModelIDs(modelID);
    }
    /**
     * Returns the IDs of all OI-models in the database.
     *
     * @return                                  the array of IDs of all OI-models in the database
     * @throws KAONException                    thrown of there is an error
     */
    protected int[] getAllOIModelIDs() throws KAONException {
        return getEngineeringServer().getAllOIModelIDs();
    }
    /**
     * Executes an update of the model.
     *
     * @param changeList                        list of changes to apply
     * @return                                  object containing information about the update
     * @throws KAONException                    thrown if there is a problem with fetching or updating information
     */
    protected UpdateResultHolder updateOIModel(List changeList) throws KAONException {
        return getEngineeringServer().updateOIModel(m_connectionIdentifier,changeList);
    }
    /**
     * Returns the engineering server.
     *
     * @return                                  the engineering server
     */
    public synchronized EngineeringServerLocal getEngineeringServer() {
        setSecurityAssociation();
        return m_engineeringServer;
    }
    /**
     * Sets the correct security association.
     */
    protected void setSecurityAssociation() {
        if (m_subject!=null) {
            SecurityAssociation.setPrincipal(m_principal);
            SecurityAssociation.setCredential(m_credential);
            SecurityAssociation.setSubject(m_subject);
        }
    }

    /**
     * The resolver for RDF physical URIs.
     */
    protected static class PhysicalURIResolver implements PhysicalURIToDefaultParametersResolver {
        public Map getDefaultParametersForPhysicalURI(String physicalURI,Map contextParameters) {
            try {
                URI uri=new URI(physicalURI);
                String scheme=uri.getScheme();
                if ("local".equals(scheme) && "localhost".equals(uri.getHost())) {
                    Map parameters=new HashMap();
                    parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.engineeringserver.client.LocalKAONConnection");
                    if (contextParameters!=null) {
                        if (contextParameters.containsKey(CHANGE_LISTENER))
                            parameters.put(CHANGE_LISTENER,contextParameters.get(CHANGE_LISTENER));
                        if (contextParameters.containsKey(USER_NAME))
                            parameters.put(USER_NAME,contextParameters.get(USER_NAME));
                        if (contextParameters.containsKey(PASSWORD))
                            parameters.put(PASSWORD,contextParameters.get(PASSWORD));
                    }
                    if (parameters.containsKey(USER_NAME) && !parameters.containsKey(PASSWORD))
                        parameters.put(PASSWORD,"???");
                    return parameters;
                }
            }
            catch (URISyntaxException e) {
            }
            return null;
        }
    }
}
