package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.SQLException;
import java.sql.Connection;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

import edu.unika.aifb.kaon.engineeringserver.dao.*;
import edu.unika.aifb.kaon.engineeringserver.loader.*;
import edu.unika.aifb.kaon.engineeringserver.change.*;
import edu.unika.aifb.kaon.engineeringserver.query.*;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.jdbc.*;

/**
 * Implements a KAON connection for direct connection to the engineering server.
 */
public class DirectKAONConnection extends AbstractKAONConnection {
    /** The server URI. */
    public static final String SERVER_URI="SERVER_URI";
    /** The class name of the database driver. */
    public static final String JDBC_DRIVER="JDBC_DRIVER";
    /** The database connection string that this connection is serving. */
    public static final String DATABASE_CONNECTION_STRING="DATABASE_CONNECTION_STRING";
    /** The user name. */
    public static final String USER_NAME="USER_NAME";
    /** The password. */
    public static final String PASSWORD="PASSWORD";

    static {
        KAONManager.registerPhysicalURIToDefaultParametersResolver(new PhysicalURIResolver());
    }

    /** The connection pool. */
    protected ConnectionPool m_connectionPool;
    /** The type of the object loader. */
    protected int m_objectLoaderType;
    /** The type of the DAO. */
    protected int m_daoType;
    /** The database manager for the connection. */
    protected DatabaseManager m_databaseManager;

    /**
     * Creates an instance of this class and connects it to given database server.
     *
     * @param connectionString                  the connection string
     * @param userName                          the user name
     * @param password                          the password
     * @param serverURI                         the URI of the server
     * @throws KAONException                    thrown if connection cannot be created
     */
    public DirectKAONConnection(String connectionString,String userName,String password,String serverURI) throws KAONException {
        Map parameters=new HashMap();
        parameters.put(DATABASE_CONNECTION_STRING,connectionString);
        parameters.put(USER_NAME,userName);
        parameters.put(PASSWORD,password);
        parameters.put(SERVER_URI,serverURI);
        initialize(parameters);
    }
    /**
     * Creates an instance of this class from a map of parameters.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if connection cannot be created
     */
    public DirectKAONConnection(Map parameters) throws KAONException {
        initialize(parameters);
    }
    /**
     * Creates an instance of this class and initializes it with a connection pool and a server URI.
     *
     * @param parameters                        the parameters
     * @param connectionPool                    the connection pool
     * @param serverURI                         the server URI
     */
    public DirectKAONConnection(Map parameters,ConnectionPool connectionPool,URI serverURI) {
        m_parameters=new HashMap(parameters);
        m_connectionPool=connectionPool;
        m_connectionPool.addUser();
        m_serverURI=serverURI;
        m_oimodels=new HashMap();
    }
    /**
     * Initializes this connection.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if connection cannot be created
     */
    protected void initialize(Map parameters) throws KAONException {
        super.initialize(parameters);
        String serverURI=(String)parameters.get(SERVER_URI);
        if (serverURI==null)
            throw new KAONException("SERVER_URI parameter must be specified.");
        try {
            m_serverURI=new URI(serverURI);
        }
        catch (URISyntaxException e) {
            throw new KAONException("SERVER_URI has invalid syntax.",e);
        }
        String scheme=m_serverURI.getScheme();
        if (!"direct".equals(scheme))
            throw new KAONException("Scheme "+scheme+" is not supported by DirectKAONConnection.");
        String path=m_serverURI.getPath();
        if (path==null)
            throw new KAONException("Path must be specified.");
        if (path.startsWith("/"))
            path=path.substring(1);
        String databaseType;
        String rest;
        int slashPosition=path.indexOf('/');
        if (slashPosition==-1) {
            databaseType=path;
            rest="";
        }
        else {
            databaseType=path.substring(0,slashPosition);
            rest=path.substring(slashPosition+1);
        }
        String jdbcDriver=(String)m_parameters.get(JDBC_DRIVER);
        String connectionString=(String)m_parameters.get(DATABASE_CONNECTION_STRING);
        if ("MSSQL".equals(databaseType)) {
            if (jdbcDriver==null)
                jdbcDriver="com.microsoft.jdbc.sqlserver.SQLServerDriver";
            if (connectionString==null)
                connectionString="jdbc:microsoft:sqlserver://"+m_serverURI.getHost()+":"+getServerPort(m_serverURI,1433)+";DatabaseName="+rest.trim()+";SelectMethod=cursor";
        }
        else if ("IBMDB2".equals(databaseType)) {
            if (jdbcDriver==null)
                jdbcDriver="COM.ibm.db2.jdbc.net.DB2Driver";
            if (connectionString==null) {
                int port=getServerPort(m_serverURI,-1);
                if (port==-1)
                    connectionString="jdbc:db2://"+m_serverURI.getHost()+"/"+rest.trim();
                else
                    connectionString="jdbc:db2://"+m_serverURI.getHost()+":"+port+"/"+rest.trim();
            }
        }
        else if ("Oracle".equals(databaseType)) {
            if (jdbcDriver==null)
                jdbcDriver="oracle.jdbc.OracleDriver";
            if (connectionString==null)
                connectionString="jdbc:oracle:thin:@"+m_serverURI.getHost()+":"+getServerPort(m_serverURI,1521)+":"+rest.trim();
        }
        else if ("PostgreSQL".equals(databaseType)) {
            if (jdbcDriver==null)
                jdbcDriver="org.postgresql.Driver";
            if (connectionString==null)
                connectionString="jdbc:postgresql://"+m_serverURI.getHost()+":"+getServerPort(m_serverURI,5432)+"/"+rest.trim();
        }
        if (jdbcDriver!=null)
            try {
                Class.forName(jdbcDriver);
            }
            catch (ClassNotFoundException e) {
                throw new KAONException("Cannot load JDBC driver class",e);
            }
        if (connectionString==null)
            throw new KAONException("Connection string must be specified");
        String userName=(String)m_parameters.get(USER_NAME);
        if (userName==null)
            userName=m_serverURI.getUserInfo();
        String password=(String)m_parameters.get(PASSWORD);
        m_connectionPool=new ConnectionPool(connectionString,userName,password,10);
        m_connectionPool.addUser();
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            m_objectLoaderType=AbstractSQLObjectLoader.getObjectLoaderType(connection);
            m_daoType=EngineeringServerDAO.getDAOType(connection);
            m_databaseManager=EngineeringServerExtensionalDatabase.createDatabaseManager(connection);
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            dao.createDefaultModels();
            connection.commit();
        }
        catch (SQLException e) {
            throw new KAONException("Error connecting to database",e);
        }
        catch (DatalogException e) {
            throw new KAONException("Error connecting to database",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Closes this connection.
     *
     * @throws KAONException                    thrown if connection cannot be closed
     */
    public synchronized void close() throws KAONException {
        super.close();
        if (m_connectionPool!=null) {
            m_connectionPool.removeUser();
            m_connectionPool=null;
        }
        m_oimodels=null;
    }
    /**
     * Returns a child connection. This can be used to implement advanced features, such as connection pooling.
     *
     * @return                                  a connection that is the child of this connection
     */
    public synchronized KAONConnection getConnection() {
        DirectKAONConnection childConnection=new DirectKAONConnection(m_parameters,m_connectionPool,m_serverURI);
        childConnection.m_objectLoaderType=m_objectLoaderType;
        childConnection.m_daoType=m_daoType;
        childConnection.m_databaseManager=m_databaseManager;
        return childConnection;
    }
    /**
     * Creates an OI-model with given physical and logical URIs.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  new model
     * @throws KAONException                    thrown if model cannot be created
     */
    public synchronized OIModel createOIModel(String physicalURI,String logicalURI) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        int modelID;
        try {
            physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
            checkCreatingPhysicalURI(physicalURI,logicalURI);
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            modelID=dao.getOIModelDAO().createOIModel(logicalURI);
            connection.commit();
        }
        catch (SQLException e) {
            throw new KAONException("OI-model with physical URI '"+physicalURI+"' and logical URI '"+logicalURI+"' cannot be created.",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
        return getOIModel(modelID);
    }
    /**
     * Creates a new OI-model with supplied logical and physical URI and makes it an exact duplicate of supplied OI-model.
     *
     * @param sourceOIModel             the source OI-model
     * @param physicalURI               the physical URI of the new model
     * @param logicalURI                the logical URI of the new model
     * @return                          newly created OI-model
     * @throws KAONException            thrown if there is an error
     */
    public synchronized OIModel createDuplicate(OIModel sourceOIModel,String physicalURI,String logicalURI) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            physicalURI=getDefaultPhysicalURI(physicalURI,logicalURI);
            checkCreatingPhysicalURI(physicalURI,logicalURI);
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            int modelID=dao.getOIModelDAO().createDuplicate(sourceOIModel.getLogicalURI(),logicalURI);
            connection.commit();
            return getOIModel(modelID);
        }
        catch (SQLException e) {
            throw new KAONException("OI-model with physical URI '"+physicalURI+"' and logical URI '"+logicalURI+"' cannot be created.",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Returns the set of logical URIs of all OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized Set getAllOIModelLogicalURIs() throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            Object[][] information=dao.getOIModelDAO().getAllOIModelInformation();
            connection.commit();
            Set result=new HashSet();
            for (int i=0;i<information.length;i++)
                result.add(information[i][1]);
            return result;
        }
        catch (SQLException e) {
            throw new KAONException("Cannot obtain the information about all OI-models.",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Returns the model ID for logical URI.
     *
     * @param logicalURI                        the logical URI
     * @return                                  the model ID of model with given URI
     * @throws KAONException                    thrown if there is an error
     */
    protected int getModelIDForLogicalURI(String logicalURI) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            int modelID=dao.getOIModelDAO().getOIModelID(logicalURI);
            connection.commit();
            return modelID;
        }
        catch (SQLException e) {
            throw new KAONException("Cannot locate model ID for model with logical URI '"+logicalURI+"'");
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Returns the IDs of included models.
     *
     * @param modelID                           the ID of the model
     * @return                                  the array of included models
     * @throws KAONException                    thrown if there is an error
     */
    protected int[] getIncludedModelIDs(int modelID) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            int[] modelIDs=dao.getOIModelDAO().getIncludedModelIDs(modelID);
            connection.commit();
            return modelIDs;
        }
        catch (SQLException e) {
            throw new KAONException("Cannot locate models included into model with ID "+modelID);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Returns the IDs of all OI-models in the database.
     *
     * @return                                  the array of IDs of all OI-models in the database
     * @throws KAONException                    thrown of there is an error
     */
    protected int[] getAllOIModelIDs() throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            int[] modelIDs=dao.getOIModelDAO().getAllOIModelIDs();
            connection.commit();
            return modelIDs;
        }
        catch (SQLException e) {
            throw new KAONException("Error obtaining the list of all OI-models");
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Executes an update of the model.
     *
     * @param changeList                        list of changes to apply
     * @return                                  object containing information about the update
     * @throws KAONException                    thrown if there is a problem with fetching or updating information
     */
    protected UpdateResultHolder updateOIModel(List changeList) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getDatabaseConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_daoType);
            ApplyChangeVisitor applyChangeVisitor=new ApplyChangeVisitor(dao);
            applyChangeVisitor.applyChanges(changeList);
            connection.commit();
            UpdateResultHolder result=new UpdateResultHolder();
            result.m_initiatingConnectionIdentifier=m_connectionIdentifier;
            result.m_updatedChangeList=changeList;
            result.m_updatedConceptIDs=applyChangeVisitor.getUpdatedConceptIDs();
            result.m_updatedPropertyIDs=applyChangeVisitor.getUpdatedPropertyIDs();
            result.m_updatedInstanceIDs=applyChangeVisitor.getUpdatedInstanceIDs();
            return result;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseDatabaseConnection(connection);
        }
    }
    /**
     * Returns a connection from the connection pool.
     *
     * @return                                  the connection
     * @throws SQLException                     thrown if there is an error
     */
    public Connection getDatabaseConnection() throws SQLException {
        return m_connectionPool.getConnection();
    }
    /**
     * Releases a database connection.
     *
     * @param connection                        the connection
     */
    public synchronized void releaseDatabaseConnection(Connection connection) {
        try {
            if (!connection.isClosed())
                connection.rollback();
        }
        catch (SQLException ignored) {
        }
        if (isOpen())
            m_connectionPool.releaseConnection(connection);
        else
            try {
                connection.close();
            }
            catch (SQLException ignored) {
            }
    }
    /**
     * Create the OI-model source for this connection.
     *
     * @param serverURI                         the URI of the server
     * @param modelID                           the ID of the model
     * @return                                  the OI-model source for this connection
     * @throws KAONException                    thrown if OI-model source cannot be created
     */
    protected OIModelSource createOIModelSource(String serverURI,int modelID) throws KAONException {
        return new DirectEngineeringSource(serverURI,modelID,this);
    }
    /**
     * Returns the server port.
     *
     * @param serverURI                         the server URI
     * @param defaultPort                       the default port
     * @return                                  the server port
     */
    protected static int getServerPort(URI serverURI,int defaultPort) {
        int port=serverURI.getPort();
        if (port==-1)
            return defaultPort;
        else
            return port;
    }


    /**
     * The resolver for RDF physical URIs.
     */
    protected static class PhysicalURIResolver implements PhysicalURIToDefaultParametersResolver {
        public Map getDefaultParametersForPhysicalURI(String physicalURI,Map contextParameters) {
            try {
                URI uri=new URI(physicalURI);
                if ("direct".equals(uri.getScheme())) {
                    URI serverURI=new URI(uri.getScheme(),uri.getUserInfo(),uri.getHost(),uri.getPort(),uri.getPath(),null,null);
                    Map parameters=new HashMap();
                    parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.engineeringserver.client.DirectKAONConnection");
                    parameters.put(SERVER_URI,serverURI.toString());
                    if (contextParameters!=null && contextParameters.containsKey(SERVER_URI) && contextParameters.containsKey(PASSWORD)) {
                        URI contextServerURI=new URI((String)contextParameters.get(SERVER_URI));
                        if (contextServerURI.equals(serverURI))
                            parameters.put(PASSWORD,contextParameters.get(PASSWORD));
                    }
                    if (parameters.get(PASSWORD)==null)
                        parameters.put(PASSWORD,"???");
                    if (serverURI.getPath().startsWith("/other")) {
                        parameters.put(JDBC_DRIVER,"???");
                        parameters.put(DATABASE_CONNECTION_STRING,"???");
                    }
                    return parameters;
                }
            }
            catch (URISyntaxException e) {
            }
            return null;
        }
    }
}
