package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Collections;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.SQLException;
import java.sql.Connection;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.apiproxy.source.*;
import edu.unika.aifb.kaon.engineeringserver.loader.*;
import edu.unika.aifb.kaon.engineeringserver.dao.*;
import edu.unika.aifb.kaon.engineeringserver.query.*;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.magic.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;
import edu.unika.aifb.kaon.datalog.util.*;

import edu.unika.aifb.kaon.query.ast.*;
import edu.unika.aifb.kaon.query.parser.*;
import edu.unika.aifb.kaon.query.compiler.*;

/**
 * The OI-model source that uses the server directly.
 */
public class DirectEngineeringSource implements OIModelSource {
    /** The key for the engineering server extensional database. */
    protected static final String ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY="ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY";
    /** The key for the in-memory extensional database for temporary predicates. */
    protected static final String TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY="TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY";

    /** The logical URI of this model. */
    protected String m_logicalURI;
    /** The physical URI of this model. */
    protected String m_physicalURI;
    /** The ID of the model. */
    protected int m_modelID;
    /** The KAON connection. */
    protected DirectKAONConnection m_kaonConnection;

    /**
     * Creates an instance of this  class.
     *
     * @param serverPhysicalURI             the physical URI of the server
     * @param modelID                       the ID of the model
     * @param kaonConnection                the KAON connection
     * @throws KAONException                thrown if the model cannot be opened
     */
    public DirectEngineeringSource(String serverPhysicalURI,int modelID,DirectKAONConnection kaonConnection) throws KAONException {
        m_kaonConnection=kaonConnection;
        m_modelID=modelID;
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_kaonConnection.m_daoType);
            m_logicalURI=dao.getOIModelDAO().getOIModelLogicalURI(m_modelID);
            connection.commit();
        }
        catch (SQLException e) {
            throw new KAONException("Cannot find model",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseConnection(connection);
        }
        m_physicalURI=serverPhysicalURI+"?"+m_logicalURI;
    }
    /**
     * Returns the ID of the OI-model.
     *
     * @return                              the ID of the OI-model
     */
    public int getOIModelID() {
        return m_modelID;
    }
    /**
     * Returns the logical URI of this source.
     *
     * @return                              the logical URI of this source
     */
    public String getLogicalURI() {
        return m_logicalURI;
    }
    /**
     * Returns the physical URI of this source.
     *
     * @return                              the physical URI of this source
     */
    public String getPhysicalURI() {
        return m_physicalURI;
    }
    /**
     * Called to load information about specified entities.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @return                              information about loaded entities
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityInfo[] loadEntities(String[] entityURIs,int loadFlag) throws KAONException {
        Connection connection=null;
        try {
            connection=getConnection();
            AbstractObjectLoader loader=AbstractSQLObjectLoader.createSQLObjectLoader(connection,m_modelID,m_kaonConnection.m_objectLoaderType);
            EntityInfo[] entityInfos=loader.loadEntities(entityURIs,loadFlag);
            connection.commit();
            return entityInfos;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Checks whether given instance is connected to specified value through a specified property.
     *
     * @param instanceURI                   the URI of the instance from which the check is made
     * @param propertyURI                   the URI of the property
     * @param targetInstanceURI             the URI of the target instance
     * @param targetValue                   the target value
     * @return                              <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     * @throws KAONException                thrown if the check can't be performed
     */
    public boolean pointsToValue(String instanceURI,String propertyURI,String targetInstanceURI,Object targetValue) throws KAONException {
        DatalogManager datalogManager=getDatalogManager();
        EngineeringServerExtensionalDatabase extensionalDatabase=(EngineeringServerExtensionalDatabase)datalogManager.getExtensionalDatabase(ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY);
        Connection connection=null;
        try {
            connection=getConnection();
            extensionalDatabase.setConnection(connection,m_kaonConnection.m_databaseManager);
            String unadornedQueryPredicateName;
            Constant[] boundValues;
            if (targetInstanceURI!=null) {
                unadornedQueryPredicateName="RelationInstance_a";
                boundValues=new Constant[] { new Constant(new EntityURI(propertyURI)),new Constant(new EntityURI(instanceURI)),new Constant(new EntityURI(targetInstanceURI)) };
            }
            else {
                unadornedQueryPredicateName="AttributeInstance_a";
                boundValues=new Constant[] { new Constant(new EntityURI(propertyURI)),new Constant(new EntityURI(instanceURI)),new Constant(targetValue) };
            }
            QueryOperator operator=getCachedQueryOperator(datalogManager,unadornedQueryPredicateName,boundValues);
            operator.start();
            boolean result=!operator.afterLast();
            operator.stop();
            return result;
        }
        catch (DatalogException error) {
            throw new KAONException("Error processing datalog query",error);
        }
        catch (SQLException error) {
            throw new KAONException("SQL error",error);
        }
        finally {
            MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            temporaryExtensionalDatabase.clear();
            try {
                extensionalDatabase.setConnection(null,null);
            }
            catch (DatalogException ignored) {
            }
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns the information about all property instances connected to given instance, possibly filtered through given property.
     *
     * @param instanceURI                   the URI of the instance for which property instances are loaded
     * @param propertyURI                   the URI of the property for which property instances are loaded (may be <code>null</code>)
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesFrom</code>
     * @throws KAONException                thrown if the property instances can't be determined
     */
    public Object[][] getAllFromPropertyValues(String instanceURI,String propertyURI) throws KAONException {
        return getAllFromToPropertyValues("PropertyInstance_a",new Constant[] { propertyURI==null ? null : new Constant(new EntityURI(propertyURI)),new Constant(new EntityURI(instanceURI)),null,null },true);
    }
    /**
     * Returns the information about all property instances pointing from given instance, possibly filtered through given property.
     *
     * @param instanceURI                   the URI of the instance from which property instances are loaded
     * @param propertyURI                   the URI of the property for which property instances are loaded (may be <code>null</code>)
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesTo</code>
     * @throws KAONException                thrown if the property instances can't be determined
     */
    public Object[][] getAllToPropertyValues(String instanceURI,String propertyURI) throws KAONException {
        return getAllFromToPropertyValues("RelationInstance_a",new Constant[] { propertyURI==null ? null : new Constant(new EntityURI(propertyURI)),null,new Constant(new EntityURI(instanceURI)) },false);
    }
    /**
     * Executes a query for loading instances.
     *
     * @param unadornedQueryPredicateName   the name of the unadorned query predicate
     * @param boundValues                   the array of bound values
     * @param fromValuesRequested           set to <code>true</code> if the from values are requested
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesFrom</code> or <code>InstanceInfo.m_propertyValuesTo</code>
     * @throws KAONException                thrown if the property instances can't be determined
     */
    protected Object[][] getAllFromToPropertyValues(String unadornedQueryPredicateName,Constant[] boundValues,boolean fromValuesRequested) throws KAONException {
        DatalogManager datalogManager=getDatalogManager();
        EngineeringServerExtensionalDatabase extensionalDatabase=(EngineeringServerExtensionalDatabase)datalogManager.getExtensionalDatabase(ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY);
        Connection connection=null;
        try {
            connection=getConnection();
            extensionalDatabase.setConnection(connection,m_kaonConnection.m_databaseManager);
            QueryOperator operator=getCachedQueryOperator(datalogManager,unadornedQueryPredicateName,boundValues);
            Map result=new HashMap();
            operator.start();
            while (!operator.afterLast()) {
                Object[] tuple=operator.tuple();
                EntityURI loadedPropertyURI=(EntityURI)tuple[0];
                Set set=(Set)result.get(loadedPropertyURI);
                if (set==null) {
                    set=new HashSet();
                    result.put(loadedPropertyURI,set);
                }
                if (fromValuesRequested) {
                    if (tuple[2] instanceof EntityURI)
                        set.add(new EntityID(((EntityURI)tuple[2]).getEntityURI(),-2));
                    else if (tuple[3]!=null)
                        set.add(tuple[3]);
                }
                else {
                    if (tuple[1] instanceof EntityURI)
                        set.add(new EntityID(((EntityURI)tuple[1]).getEntityURI(),-2));
                }
                operator.next();
            }
            operator.stop();
            Object[][] resultArray=new Object[result.size()][];
            int rowIndex=0;
            Iterator iterator=result.keySet().iterator();
            while (iterator.hasNext()) {
                EntityURI loadedPropertyURI=(EntityURI)iterator.next();
                Set values=(Set)result.get(loadedPropertyURI);
                Object[] row=new Object[values.size()+1];
                row[0]=new EntityID(loadedPropertyURI.getEntityURI(),-2);
                Iterator valuesIterator=values.iterator();
                int columnIndex=1;
                while (valuesIterator.hasNext())
                    row[columnIndex++]=valuesIterator.next();
                resultArray[rowIndex++]=row;
            }
            return resultArray;
        }
        catch (DatalogException error) {
            throw new KAONException("Error processing datalog query",error);
        }
        catch (SQLException error) {
            throw new KAONException("SQL error",error);
        }
        finally {
            MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            temporaryExtensionalDatabase.clear();
            try {
                extensionalDatabase.setConnection(null,null);
            }
            catch (DatalogException ignored) {
            }
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Obtains the cached query operator.
     *
     * @param datalogManager                the datalog manager
     * @param unadornedQueryPredicateName   the name of the unadorned query predicate
     * @param boundValues                   the array of bound values
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesFrom</code> or <code>InstanceInfo.m_propertyValuesTo</code>
     * @throws DatalogException             thrown if the property instances can't be determined
     */
    protected QueryOperator getCachedQueryOperator(DatalogManager datalogManager,String unadornedQueryPredicateName,Constant[] boundValues) throws DatalogException {
        String queryPredicateName=unadornedQueryPredicateName+"_";
        for (int i=0;i<boundValues.length;i++)
            queryPredicateName+=boundValues[i]==null ? 'f' : 'b';
        Evaluator evaluator=datalogManager.getEvaluator(queryPredicateName);
        if (evaluator==null) {
            Predicate queryPredicate=StandardAxiomatization.PREDICATE_FACTORY.getPredicate(unadornedQueryPredicateName,boundValues.length);
            boolean[] boundVariables=new boolean[boundValues.length];
            for (int i=0;i<boundVariables.length;i++)
                boundVariables[i]=boundValues[i]==null ? false : true;
            MagicSetRewriting rewriting=new MagicSetRewriting(StandardAxiomatization.PROGRAM,queryPredicate,boundVariables,EngineeringServerSIPS.INSTANCE,PredicateBindingPatterns.ALL_BINDING_PATTERNS_ALLOWED);
            datalogManager.createMissingPredicates(rewriting.getMagicProgram(),TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            ProgramCompiler programCompiler=new ProgramCompiler(datalogManager,rewriting.getMagicProgram(),Collections.EMPTY_MAP,queryPredicate);
            evaluator=programCompiler.getEvaluator();
            datalogManager.registerEvaluator(queryPredicateName,evaluator,rewriting);
        }
        MagicSetRewriting rewriting=datalogManager.getMagicSetRewriting(queryPredicateName);
        QueryOperator operator=rewriting.getResultForBoundValues(datalogManager,boundValues);
        evaluator.execute();
        return operator;
    }
    /**
     * Executes given query and returns the results.
     *
     * @param queryString                   the query being executed
     * @return                              the results of the query
     * @throws KAONException                thrown if there is an error executing the query
     */
    public Object[] executeQuery(String queryString) throws KAONException {
        DatalogManager datalogManager=getDatalogManager();
        EngineeringServerExtensionalDatabase extensionalDatabase=(EngineeringServerExtensionalDatabase)datalogManager.getExtensionalDatabase(ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY);
        MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
        Connection connection=null;
        Program program=null;
        Predicate queryPredicate=null;
        try {
            connection=getConnection();
            ExpressionParser parser=new ExpressionParser(new StringReader(queryString));
            parser.setResolutionURI(new URI(getLogicalURI()));
            Expression queryExpression=parser.ExpressionEOF();
            PredicateFactory predicateFactory=new PredicateFactory();
            ExpressionToDatalogCompiler compiler=new ServerExpressionToDatalogCompiler(predicateFactory,m_modelID,connection);
            StandardAxiomatization.getAxiomatization(predicateFactory,compiler.getRules());
            queryPredicate=compiler.compileExpression(queryExpression,null);
            program=compiler.getProgram();
            extensionalDatabase.setConnection(connection,m_kaonConnection.m_databaseManager);
            boolean[] boundVariables=new boolean[queryPredicate.getArity()];
            MagicSetRewriting rewriting=new MagicSetRewriting(program,queryPredicate,boundVariables,EngineeringServerSIPS.INSTANCE,PredicateBindingPatterns.ALL_BINDING_PATTERNS_ALLOWED);
            datalogManager.createMissingPredicates(rewriting.getMagicProgram(),TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            ProgramCompiler programCompiler=new ProgramCompiler(datalogManager,rewriting.getMagicProgram(),Collections.EMPTY_MAP,queryPredicate);
            Evaluator evaluator=programCompiler.getEvaluator();
            Constant[] boundValues=new Constant[queryPredicate.getArity()];
            QueryOperator operator=rewriting.getResultForBoundValues(datalogManager,boundValues);
            evaluator.execute();
            operator.start();
            List result=new LinkedList();
            while (!operator.afterLast()) {
                Object[] tuple=operator.tuple();
                if (tuple.length==1) {
                    Object object=tuple[0];
                    if (object instanceof EntityURI) {
                        EntityURI entityURI=(EntityURI)object;
                        result.add(new EntityID(entityURI.getEntityURI(),-2));
                    }
                    else
                        result.add(object);
                }
                else if (tuple.length==2) {
                    if (tuple[0] instanceof EntityURI) {
                        EntityURI entityURI=(EntityURI)tuple[0];
                        tuple[0]=new EntityID(entityURI.getEntityURI(),-2);
                    }
                    if (tuple[1] instanceof EntityURI) {
                        EntityURI entityURI=(EntityURI)tuple[1];
                        tuple[1]=new EntityID(entityURI.getEntityURI(),-2);
                    }
                    result.add(tuple);
                }
                operator.next();
            }
            Object[] resultArray=new Object[result.size()];
            result.toArray(resultArray);
            return resultArray;
        }
        catch (CompilationException error) {
            throw new KAONException("Error compiling an expression into datalog.",error);
        }
        catch (URISyntaxException error) {
            throw new KAONException("Invalid URI '"+error.getInput()+"'",error);
        }
        catch (ParseException error) {
            throw new KAONException("Error parsing query",error);
        }
        catch (TokenMgrError error) {
            throw new KAONException("Error tokenizing the query",error);
        }
        catch (DatalogException error) {
            throw new KAONException("Error processing datalog query",error);
        }
        catch (SQLException error) {
            throw new KAONException("SQL error",error);
        }
        finally {
            temporaryExtensionalDatabase.clear();
            try {
                extensionalDatabase.setConnection(null,null);
            }
            catch (DatalogException ignored) {
            }
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns the list of IDs of all concepts.
     *
     * @return                              the list of all IDs of concepts
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] getConcepts() throws KAONException {
        Connection connection=null;
        try {
            connection=getConnection();
            AbstractObjectLoader loader=AbstractSQLObjectLoader.createSQLObjectLoader(connection,m_modelID,m_kaonConnection.m_objectLoaderType);
            EntityID[] entityIDs=loader.loadConcepts();
            connection.commit();
            return entityIDs;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns the list of IDs of all properties.
     *
     * @return                              the list of all IDs of properties
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] getProperties() throws KAONException {
        Connection connection=null;
        try {
            connection=getConnection();
            AbstractObjectLoader loader=AbstractSQLObjectLoader.createSQLObjectLoader(connection,m_modelID,m_kaonConnection.m_objectLoaderType);
            EntityID[] entityIDs=loader.loadProperties();
            connection.commit();
            return entityIDs;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns the list of IDs of all instances.
     *
     * @return                              the list of all IDs of instances
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    public EntityID[] getInstances() throws KAONException {
        Connection connection=null;
        try {
            connection=getConnection();
            AbstractObjectLoader loader=AbstractSQLObjectLoader.createSQLObjectLoader(connection,m_modelID,m_kaonConnection.m_objectLoaderType);
            EntityID[] entityIDs=loader.loadInstances();
            connection.commit();
            return entityIDs;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Deletes the model.
     *
     * @throws KAONException                thrown if there is a problem with fetching or updating information
     */
    public void deleteOIModel() throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_kaonConnection.m_daoType);
            dao.getOIModelDAO().deleteOIModel(m_modelID);
            connection.commit();
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns an OI-model for given model ID.
     *
     * @param modelID                       the ID of the OI-model
     * @return                              the OI-model with given ID
     * @throws KAONException                thrown if the ID cannot be translated into an OI-model
     */
    public OIModel getOIModel(int modelID) throws KAONException {
        return m_kaonConnection.getOIModel(modelID);
    }
    /**
     * Sets the attribute of the OI-model.
     *
     * @param key                           the key
     * @param value                         the value (or <code>null</code> if the attribute should be deleted)
     * @throws KAONException                thrown if the attribute cannot be set
     */
    public void setAttribute(String key,String value) throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_kaonConnection.m_daoType);
            dao.getOIModelDAO().setOIModelAttribute(m_modelID,key,value);
            connection.commit();
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns the map of attributes of this OI-model.
     *
     * @return                              the map of attributes of the OI-model
     * @throws KAONException                thrown if the attributes cannot be read
     */
    public Map getAttributes() throws KAONException {
        Connection connection=null;
        EngineeringServerDAO dao=null;
        try {
            connection=getConnection();
            dao=EngineeringServerDAO.createDAO(connection,m_kaonConnection.m_daoType);
            Map result=dao.getOIModelDAO().getOIModelAttributes(m_modelID);
            connection.commit();
            return result;
        }
        catch (SQLException e) {
            throw new KAONException("Error invoking a database call",e);
        }
        finally {
            if (dao!=null)
                dao.close();
            if (connection!=null)
                releaseConnection(connection);
        }
    }
    /**
     * Returns a connection.
     *
     * @return                              the database connection
     * @throws SQLException                 thrown if connection cannot be obtained
     */
    protected Connection getConnection() throws SQLException {
        return m_kaonConnection.getDatabaseConnection();
    }
    /**
     * Releases a connection.
     *
     * @param connection                    connection that was released
     */
    protected void releaseConnection(Connection connection) {
        m_kaonConnection.releaseDatabaseConnection(connection);
    }
    /**
     * Creates a datalog manager.
     *
     * @return                              the datalog manager
     */
    protected DatalogManager getDatalogManager() {
        DatalogManager datalogManager=new DatalogManager();
        datalogManager.registerExtensionalDatabase(ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY,new EngineeringServerExtensionalDatabase(m_modelID));
        datalogManager.registerExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY,new MemoryExtensionalDatabase());
        return datalogManager;
    }
}
