package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.List;
import java.util.ArrayList;
import java.sql.SQLException;
import java.sql.Connection;
import java.sql.DriverManager;

/**
 * Implements a connection pool.
 */
public class ConnectionPool {
    /** The number of users of this pool. */
    protected int m_users;
    /** List of unused connections. */
    protected List m_unusedConnections;
    /** List of used connections. */
    protected List m_usedConnections;
    /** Maximum number of connections. */
    protected int m_maximumConnections;
    /** The connection string. */
    protected String m_connectionString;
    /** The user name. */
    protected String m_userName;
    /** The password. */
    protected String m_password;

    /**
     * Creates and initializes this object.
     *
     * @param connectionString                      the database connection string
     * @param userName                              the name of the user
     * @param password                              the password
     * @param maximumConnections                    the maximum number of connections
     */
    public ConnectionPool(String connectionString,String userName,String password,int maximumConnections) {
        m_connectionString=connectionString;
        m_userName=userName;
        m_password=password;
        m_maximumConnections=maximumConnections;
        m_unusedConnections=new ArrayList();
        m_usedConnections=new ArrayList();
    }
    /**
     * Returns a connection from the pool.
     *
     * @return                                      a connection from the pool
     * @throws SQLException                         thrown if connection cannot be established
     */
    public synchronized Connection getConnection() throws SQLException {
        while (m_unusedConnections.size()==0 && m_usedConnections.size()==m_maximumConnections)
            try {
                wait();
            }
            catch (InterruptedException e) {
                throw new SQLException("Thread interrupted");
            }
        Connection result;
        if (m_unusedConnections.size()!=0) {
            int index=m_unusedConnections.size()-1;
            result=(Connection)m_unusedConnections.get(index);
            m_unusedConnections.remove(index);
        }
        else {
            if (m_userName==null)
                result=DriverManager.getConnection(m_connectionString);
            else
                result=DriverManager.getConnection(m_connectionString,m_userName,m_password);
            result.setAutoCommit(false);
        }
        m_usedConnections.add(result);
        return result;
    }
    /**
     * Returns a connection to the pool.
     *
     * @param connection                            the connection being returned
     */
    public synchronized void releaseConnection(Connection connection) {
        if (!m_usedConnections.remove(connection))
            throw new IllegalArgumentException("Supplied connection is not opened by this connection pool.");
        try {
            if (!connection.isClosed())
                m_unusedConnections.add(connection);
        }
        catch (SQLException ignored) {
            try {
                connection.close();
            }
            catch (SQLException ignoredAgain) {
            }
        }
        notifyAll();
    }
    /**
     * Notifies this object that a user has been attached to this pool.
     */
    public synchronized void addUser() {
        m_users++;
    }
    /**
     * Notifies this object that a user has been detached to this pool. When the number of users drops to 0, the pool is closed.
     */
    public synchronized void removeUser() {
        m_users--;
        if (m_users==0)
            close();
    }
    /**
     * Closes this pool.
     */
    public synchronized void close() {
        for (int i=0;i<m_unusedConnections.size();i++) {
            Connection connection=(Connection)m_unusedConnections.get(i);
            try {
                connection.close();
            }
            catch (SQLException e) {
            }
        }
        m_unusedConnections.clear();
        for (int i=0;i<m_usedConnections.size();i++) {
            Connection connection=(Connection)m_usedConnections.get(i);
            try {
                connection.close();
            }
            catch (SQLException e) {
            }
        }
        m_usedConnections.clear();
    }
}
