package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.Hashtable;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicConnection;
import javax.jms.Topic;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.jms.MessageListener;
import javax.jms.Message;
import javax.jms.ObjectMessage;
import javax.jms.MapMessage;
import javax.jms.JMSException;
import javax.swing.SwingUtilities;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.apiproxy.*;

/**
 * Listens for changes on a JMS topic and updates the OI-model when a change occurs.
 */
public class ChangeListener implements MessageListener {
    /** The KAON connection. */
    protected AbstractKAONConnection m_kaonConnection;
    /** Topic connection of this object. */
    protected TopicConnection m_topicConnection;
    /** Topic of this object. */
    protected Topic m_topic;
    /** Topic session of this object. */
    protected TopicSession m_topicSession;
    /** Topic subscriber of this object. */
    protected TopicSubscriber m_topicSubscriber;
    /** Set to <code>true</code> if message processing should be done on the Swing event thread. */
    protected boolean m_processMessagesOnSwingEventThread;

    /**
     * Creates and initializes this object.
     *
     * @param kaonConnection                        the KAON connection
     * @param processMessagesOnSwingEventThread     set to <code>true</code> if the message should be processed on the Swing event thread
     * @param jndiProperties                        hashtable of parameters or <code>null</code> if default JNDI parameters should be used
     * @throws KAONException                        thrown in case of failure
     */
    public ChangeListener(AbstractKAONConnection kaonConnection,boolean processMessagesOnSwingEventThread,Hashtable jndiProperties) throws KAONException {
        m_kaonConnection=kaonConnection;
        m_processMessagesOnSwingEventThread=processMessagesOnSwingEventThread;
        TopicConnectionFactory topicConnectionFactory;
        try {
            Context context;
            if (jndiProperties==null)
                context=new InitialContext();
            else
                context=new InitialContext(jndiProperties);
            topicConnectionFactory=(TopicConnectionFactory)context.lookup("ConnectionFactory");
            m_topic=(Topic)context.lookup("topic/testTopic");
        }
        catch (NamingException e) {
            throw new KAONException("Cannot lookup topic connection factory from the JNDI.",e);
        }
        try {
            m_topicConnection=topicConnectionFactory.createTopicConnection();
            m_topicSession=m_topicConnection.createTopicSession(false,TopicSession.AUTO_ACKNOWLEDGE);
            m_topicSubscriber=m_topicSession.createSubscriber(m_topic);
            m_topicSubscriber.setMessageListener(this);
            m_topicConnection.start();
        }
        catch (JMSException e) {
            throw new KAONException("Error in opening JMS objects.",e);
        }
    }
    /**
     * Closes this broadcaster.
     *
     * @throws KAONException                        thrown in case of failure
     */
    public synchronized void close() throws KAONException {
        try {
            m_topicConnection.close();
        }
        catch (JMSException e) {
            throw new KAONException("Error in opening JMS connection.",e);
        }
    }
    /**
     * Receives a message and forwards it to the application's main thread.
     *
     * @param message                               the incoming message
     */
    public synchronized void onMessage(final Message message) {
        if (!m_processMessagesOnSwingEventThread)
            processMessage(message);
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    processMessage(message);
                }
            });
    }
    /**
     * Processes a message on the main thread.
     *
     * @param message                               the incoming message
     */
    protected synchronized void processMessage(Message message) {
        try {
            if (message instanceof ObjectMessage)
                processObjectMessage((ObjectMessage)message);
            else if (message instanceof MapMessage)
                processMapMessage((MapMessage)message);
        }
        catch (JMSException ignored) {
        }
    }
    /**
     * Processes the object messages.
     *
     * @param objectMessage                         the incoming object message
     * @throws JMSException                         thrown if there is an error
     */
    protected void processObjectMessage(ObjectMessage objectMessage) throws JMSException {
        try {
            KAONConnectionProxy.UpdateResultHolder updateResultHolder=(KAONConnectionProxy.UpdateResultHolder)objectMessage.getObject();
            m_kaonConnection.updateNotify(updateResultHolder);
        }
        catch (KAONException ignored) {
        }
    }
    /**
     * Processes the map messages.
     *
     * @param mapMessage                            the incoming map message
     * @throws JMSException                         thrown if there is an error
     */
    protected void processMapMessage(MapMessage mapMessage) throws JMSException {
        try {
            String initiatingConnectionIdentifier=mapMessage.getString("initiatingConnectionIdentifier");
            if (initiatingConnectionIdentifier!=null && !m_kaonConnection.getConnectionIdentifier().equals(initiatingConnectionIdentifier)) {
                int modelID=mapMessage.getInt("modelID");
                String text=mapMessage.getString("text");
                OIModelProxy oimodel=m_kaonConnection.getOIModelDontCreate(modelID);
                if (oimodel!=null && "deleted".equals(text))
                    try {
                        oimodel.deleteNotify();
                    }
                    catch (KAONException ignored) {
                    }
            }
        }
        catch (KAONException ignored) {
        }
    }
}
