package edu.unika.aifb.kaon.engineeringserver.client;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.io.UnsupportedEncodingException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.*;
import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * An abstract base class for all KAON connections.
 */
public abstract class AbstractKAONConnection extends KAONConnectionProxy {
    /** The parameters of this connection. */
    protected Map m_parameters;
    /** The map of OI-models indexed by the model ID opened by this connection. */
    protected Map m_oimodels;
    /** The server URI. */
    protected URI m_serverURI;

    /**
     * Creates an instance of this class from a map of parameters.
     */
    public AbstractKAONConnection() {
    }
    /**
     * Initializes this connection.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if connection cannot be created
     */
    protected void initialize(Map parameters) throws KAONException {
        m_parameters=new HashMap(parameters);
        m_oimodels=new HashMap();
    }
    /**
     * Returns the parameters of this connection.
     *
     * @return                                  the parameters of this connection
     */
    public Map getParameters() {
        return m_parameters;
    }
    /**
     * Opens an OI-model with given physical URI.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @return                                  the model
     * @throws KAONException                    thrown if model cannot be opened
     */
    public synchronized OIModel openOIModelPhysical(String physicalURI) throws KAONException {
        try {
            Iterator oimodels=m_oimodels.values().iterator();
            while (oimodels.hasNext()) {
                OIModel oimodel=(OIModel)oimodels.next();
                if (physicalURI.equals(oimodel.getPhysicalURI()))
                    return oimodel;
            }
            URI uri=new URI(physicalURI);
            if (!m_serverURI.getScheme().equals(uri.getScheme()))
                throw new KAONException("Supplied URI has different scheme than the connection's URI.");
            if (!m_serverURI.getHost().equals(uri.getHost()) || uri.getPort()!=m_serverURI.getPort())
                throw new KAONException("Supplied URI references different port that the connection's URI.");
            String logicalURI=URLDecoder.decode(uri.getQuery(),"UTF-8");
            OIModel oimodel=openOIModelLogical(logicalURI);
            if (oimodel==null)
                throw new KAONException("OI-model '"+physicalURI+"' not found.");
            return oimodel;
        }
        catch (UnsupportedEncodingException shouldntHappen) {
            throw new KAONException("Internal error",shouldntHappen);
        }
        catch (URISyntaxException e) {
            throw new KAONException("Physical URI '"+physicalURI+"' cannot be parsed",e);
        }
    }
    /**
     * Opens an OI-model with given logical URI.
     *
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the model
     * @throws KAONException                    thrown if model cannot be opened
     */
    public synchronized OIModel openOIModelLogical(String logicalURI) throws KAONException {
        Iterator oimodels=m_oimodels.values().iterator();
        while (oimodels.hasNext()) {
            OIModel oimodel=(OIModel)oimodels.next();
            if (logicalURI.equals(oimodel.getLogicalURI()))
                return oimodel;
        }
        try {
            int modelID=getModelIDForLogicalURI(logicalURI);
            return getOIModel(modelID);
        }
        catch (KAONException cantOpen) {
            OIModel oimodel=KAONManager.resolveLogicalURI(this,logicalURI);
            if (oimodel==null)
                throw new KAONException("Cannot open OI-model with logical URI '"+logicalURI+"'.");
            return oimodel;
        }
    }
    /**
     * Checks whether the model with supplied URIs can be opened by this connection.
     *
     * @param physicalURI               the physical URI of the new model
     * @param logicalURI                the logical URI of the new model
     * @throws KAONException            thrown if there is an error
     */
    protected void checkCreatingPhysicalURI(String physicalURI,String logicalURI) throws KAONException {
        try {
            Iterator oimodels=m_oimodels.values().iterator();
            while (oimodels.hasNext()) {
                OIModel oimodel=(OIModel)oimodels.next();
                if (physicalURI.equals(oimodel.getPhysicalURI()))
                    throw new KAONException("Model with physical URI '"+physicalURI+"' has already been opened");
                if (logicalURI.equals(oimodel.getLogicalURI()))
                    throw new KAONException("Model with logical URI '"+logicalURI+"' has already been opened");
            }
            URI uri=new URI(physicalURI);
            if (!m_serverURI.getScheme().equals(uri.getScheme()))
                throw new KAONException("Supplied URI has different scheme than the connection's URI.");
            if (!m_serverURI.getHost().equals(uri.getHost()) || uri.getPort()!=m_serverURI.getPort())
                throw new KAONException("Supplied URI references different port that the connection's URI.");
        }
        catch (URISyntaxException e) {
            throw new KAONException("Physical URI '"+physicalURI+"' cannot be parsed",e);
        }
    }
    /**
     * If supplied physical URI is empty, this method transforms it into appropriate one.
     *
     * @param physicalURI                       the physical URI
     * @param logicalURI                        the logical URI
     * @return                                  the tranformed physical URI
     */
    protected String getDefaultPhysicalURI(String physicalURI,String logicalURI) {
        if (physicalURI==null || physicalURI.length()==0)
            return m_serverURI.toString()+"?"+logicalURI;
        else
            return physicalURI;
    }
    /**
     * Notifies the connection that the OI-model has been deleted.
     *
     * @param oimodel                           OI-model that is removed
     */
    public synchronized void notifyOIModelDeleted(OIModel oimodel) {
        Iterator keys=m_oimodels.keySet().iterator();
        while (keys.hasNext()) {
            Integer key=(Integer)keys.next();
            if (m_oimodels.get(key)==oimodel) {
                keys.remove();
                break;
            }
        }
    }
    /**
     * Returns an OI-model for given model ID.
     *
     * @param modelID                           the ID of the model
     * @return                                  OI-model for given parameters
     * @throws KAONException                    thrown in case of error
     */
    public synchronized OIModel getOIModel(int modelID) throws KAONException {
        Integer key=new Integer(modelID);
        OIModel oimodel=(OIModel)m_oimodels.get(key);
        if (oimodel==null) {
            int[] includedModelIDs=getIncludedModelIDs(modelID);
            List models=new ArrayList();
            for (int i=0;i<includedModelIDs.length;i++) {
                OIModel includedModel=getOIModel(includedModelIDs[i]);
                models.add(includedModel);
            }
            OIModel[] includedOIModels=new OIModel[models.size()];
            models.toArray(includedOIModels);
            OIModelSource oimodelSource=createOIModelSource(m_serverURI.toString(),modelID);
            oimodel=new OIModelProxy(this,oimodelSource,includedOIModels);
            m_oimodels.put(key,oimodel);
        }
        return oimodel;
    }
    /**
     * Returns an OI-model for given model ID or <code>null</code> if the OI-model doesn't exist.
     *
     * @param modelID                           the ID of the model
     * @return                                  OI-model (or <code>null</code> if the OI-model doesn't exist)
     * @throws KAONException                    thrown in case of error
     */
    public synchronized OIModelProxy getOIModelDontCreate(int modelID) throws KAONException {
        Integer key=new Integer(modelID);
        return (OIModelProxy)m_oimodels.get(key);
    }
    /**
     * Returns the set of OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized Set getAllOIModels() throws KAONException {
        int[] oimodelIDs=getAllOIModelIDs();
        Set result=new HashSet();
        for (int i=0;i<oimodelIDs.length;i++) {
            OIModel oimodel=getOIModel(oimodelIDs[i]);
            result.add(oimodel);
        }
        return result;
    }
    /**
     * Returns the set of all open OI-model objects.
     *
     * @return                              the set of all open OI-model objects
     */
    public synchronized Set getOpenOIModels() {
        return new HashSet(m_oimodels.values());
    }
    /**
     * Returns the model ID for logical URI.
     *
     * @param logicalURI                        the logical URI
     * @return                                  the model ID of model with given URI
     * @throws KAONException                    thrown if there is an error
     */
    protected abstract int getModelIDForLogicalURI(String logicalURI) throws KAONException;
    /**
     * Create the OI-model source for this connection.
     *
     * @param serverURI                         the URI of the server
     * @param modelID                           the ID of the model
     * @return                                  the OI-model source for this connection
     * @throws KAONException                    thrown if OI-model source cannot be created
     */
    protected abstract OIModelSource createOIModelSource(String serverURI,int modelID) throws KAONException;
    /**
     * Returns the IDs of included models.
     *
     * @param modelID                           the ID of the model
     * @return                                  the array of included models
     * @throws KAONException                    thrown if there is an error
     */
    protected abstract int[] getIncludedModelIDs(int modelID) throws KAONException;
    /**
     * Returns the IDs of all OI-models in the database.
     *
     * @return                                  the array of IDs of all OI-models in the database
     * @throws KAONException                    thrown of there is an error
     */
    protected abstract int[] getAllOIModelIDs() throws KAONException;
}
