package edu.unika.aifb.kaon.engineeringserver.change;

import java.util.List;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.sql.SQLException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.engineeringserver.dao.*;

/**
 * Visits evolution events and creates a set of objects to be loaded.
 */
public class ChangeLoadVisitor implements ChangeVisitor {
    /** Set of entity URIs to load. */
    protected Set m_entityURIsToLoad;

    /**
     * Creates an instance of this class.
     */
    public ChangeLoadVisitor() {
        m_entityURIsToLoad=new HashSet();
    }
    /**
     * Adds the URI of any entity to the set of objects to be loaded.
     *
     * @param entity                    entity to be loaded
     */
    protected void loadEntity(Entity entity) throws KAONException {
        m_entityURIsToLoad.add(entity.getURI());
    }
    /**
     * Adds the URI of any entity to the set of objects to be loaded.
     *
     * @param uri                       URI to be loaded
     */
    public void loadEntity(String uri) {
        m_entityURIsToLoad.add(uri);
    }
    /**
     * Processes given event list.
     *
     * @param changeList                list of change events
     * @throws KAONException            thrown in case of error
     */
    public void processEvents(List changeList) throws KAONException {
        Iterator iterator=changeList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            changeEvent.accept(this);
        }
    }
    /**
     * Returns <code>true</code> if there are objects to be loaded.
     *
     * @return                          <code>true</code> if there are objects to load
     */
    public boolean hasObjectsToLoad() {
        return !m_entityURIsToLoad.isEmpty();
    }
    /**
     * Preloads objects accumulated from events.
     *
     * @param engineeringServerDAO      the engineering server DAO
     * @throws KAONException            thrown in case of error
     */
    public void preloadObjects(EngineeringServerDAO engineeringServerDAO) throws KAONException {
        if (hasObjectsToLoad()) {
            String[] entityURIs=new String[m_entityURIsToLoad.size()];
            m_entityURIsToLoad.toArray(entityURIs);
            try {
                engineeringServerDAO.getObjectManager().loadEntities(entityURIs);
            }
            catch (SQLException e) {
                throw new KAONException("SQL error",e);
            }
        }
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        loadEntity(event.getEntity());
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        loadEntity(event.getEntity());
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        loadEntity(event.getSuperConcept());
        loadEntity(event.getSubConcept());
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        loadEntity(event.getSuperConcept());
        loadEntity(event.getSubConcept());
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException  {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        loadEntity(event.getProperty());
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        loadEntity(event.getSuperProperty());
        loadEntity(event.getSubProperty());
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        loadEntity(event.getSuperProperty());
        loadEntity(event.getSubProperty());
    }
    /**
     * Visits an event for setting the inverse propertyship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        loadEntity(event.getProperty1());
        loadEntity(event.getProperty2());
    }
    /**
     * Visits an event for removing an inverse propertyship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        loadEntity(event.getProperty1());
        loadEntity(event.getProperty2());
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        loadEntity(event.getProperty());
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        loadEntity(event.getProperty());
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        loadEntity(event.getProperty());
        loadEntity(event.getConcept());
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException  {
        loadEntity(event.getConcept());
        loadEntity(event.getInstance());
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        loadEntity(event.getConcept());
        loadEntity(event.getInstance());
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadEntity(propertyInstance.getProperty());
        loadEntity(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadEntity((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadEntity(propertyInstance.getProperty());
        loadEntity(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadEntity((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadEntity(propertyInstance.getProperty());
        loadEntity(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadEntity((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadEntity(propertyInstance.getProperty());
        loadEntity(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadEntity((Instance)propertyInstance.getTargetValue());
        if (event.getNewTargetValue() instanceof Instance)
            loadEntity((Instance)event.getNewTargetValue());
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) {
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) {
    }
}
