package edu.unika.aifb.kaon.engineeringserver.change;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicConnection;
import javax.jms.Topic;
import javax.jms.TopicSession;
import javax.jms.TopicPublisher;
import javax.jms.ObjectMessage;
import javax.jms.MapMessage;
import javax.jms.JMSException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.apiproxy.*;

/**
 * Implements the mechanisms for notification about changes in the system.
 */
public class ChangeBroadcaster {
    /** Topic connection of this object. */
    protected TopicConnection m_topicConnection;
    /** Topic of this object. */
    protected Topic m_topic;
    /** Topic session of this object. */
    protected TopicSession m_topicSession;
    /** Topic publisher of this object. */
    protected TopicPublisher m_topicPublisher;

    /**
     * Creates and initializes this object.
     *
     * @throws KAONException                        thrown in case of failure
     */
    public ChangeBroadcaster() throws KAONException {
        TopicConnectionFactory topicConnectionFactory;
        try {
            Context context=new InitialContext();
            topicConnectionFactory=(TopicConnectionFactory)context.lookup("ConnectionFactory");
            m_topic=(Topic)context.lookup("topic/testTopic");
        }
        catch (NamingException e) {
            throw new KAONException("Cannot lookup topic connection factory from the JNDI.",e);
        }
        try {
            m_topicConnection=topicConnectionFactory.createTopicConnection();
            m_topicSession=m_topicConnection.createTopicSession(false,TopicSession.AUTO_ACKNOWLEDGE);
            m_topicPublisher=m_topicSession.createPublisher(m_topic);
        }
        catch (JMSException e) {
            throw new KAONException("Error in opening JMS objects.",e);
        }
    }
    /**
     * Closes this broadcaster.
     *
     * @throws KAONException                        thrown in case of failure
     */
    public void close() throws KAONException {
        try {
            m_topicConnection.close();
        }
        catch (JMSException e) {
            throw new KAONException("Error in opening JMS connection.",e);
        }
    }
    /**
     * Publishes the notification about an ontology update to others.
     *
     * @param updateResultHolder                    the object containing information about the update
     * @throws KAONException                        thrown in case of failure
     */
    public void updateNotify(KAONConnectionProxy.UpdateResultHolder updateResultHolder) throws KAONException {
        try {
            ObjectMessage message=m_topicSession.createObjectMessage(updateResultHolder);
            m_topicPublisher.publish(message);
        }
        catch (JMSException e) {
            throw new KAONException("Error in publishing JMS message.",e);
        }
    }
    /**
     * Publishes the notification about an OI-model event.
     *
     * @param initiatingConnectionIdentifier        the ID of the initiating connection
     * @param modelID                               the ID of the
     * @param text                                  the message text
     * @throws KAONException                        thrown in case of failure
     */
    public void modelNotify(String initiatingConnectionIdentifier,int modelID,String text) throws KAONException {
        try {
            MapMessage message=m_topicSession.createMapMessage();
            message.setString("initiatingConnectionIdentifier",initiatingConnectionIdentifier);
            message.setInt("modelID",modelID);
            message.setString("text",text);
            m_topicPublisher.publish(message);
        }
        catch (JMSException e) {
            throw new KAONException("Error in publishing JMS message.",e);
        }
    }
}
