package edu.unika.aifb.kaon.engineeringserver.change;

import java.util.List;
import java.util.Iterator;
import java.sql.SQLException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import edu.unika.aifb.kaon.apiproxy.*;
import edu.unika.aifb.kaon.apiproxy.source.*;

import edu.unika.aifb.kaon.engineeringserver.dao.*;

/**
 * Visitor that implements changes for ontology events.
 */
public class ApplyChangeVisitor implements ChangeVisitor {
    /** The ID of the model. */
    protected int m_modelID;
    /** The engieering server DAO. */
    protected EngineeringServerDAO m_engineeringServerDAO;
    /** The object manager. */
    protected ObjectManager m_objectManager;
    /** Visitor for adding. */
    protected EntityVisitor m_addVisitor;
    /** Visitor for removing. */
    protected EntityVisitor m_removeVisitor;
    /** The ID of the root OI-model. */
    protected int m_rootModelID;
    /** The root concept. */
    protected OIModelEntity m_root;

    /**
     * Creates an instance of this class.
     *
     * @param engineeringServerDAO      the engineering server DAO
     */
    public ApplyChangeVisitor(EngineeringServerDAO engineeringServerDAO) throws KAONException {
        m_engineeringServerDAO=engineeringServerDAO;
        m_addVisitor=new AddEntityVisitor();
        m_removeVisitor=new RemoveEntityVisitor();
        try {
            m_objectManager=m_engineeringServerDAO.getObjectManager();
            m_rootModelID=m_objectManager.getOIModelID(KAONConnection.ROOT_OIMODEL_URI);
        }
        catch (SQLException e) {
            throw new KAONException("Cannot create object manager",e);
        }
    }
    /**
     * Processes changes in the change list.
     *
     * @param changeList                list of changes to the model
     */
    public void applyChanges(List changeList) throws KAONException {
        try {
            ChangeLoadVisitor changeLoadVisitor=new ChangeLoadVisitor();
            changeLoadVisitor.processEvents(changeList);
            if (changeLoadVisitor.hasObjectsToLoad()) {
                if (m_root==null)
                    changeLoadVisitor.loadEntity(KAONVocabularyAdaptor.INSTANCE.getRoot());
                changeLoadVisitor.preloadObjects(m_engineeringServerDAO);
                if (m_root==null)
                    m_root=m_engineeringServerDAO.getObjectManager().getOIModelEntity(m_rootModelID,KAONVocabularyAdaptor.INSTANCE.getRoot());
            }
            Iterator iterator=changeList.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(this);
            }
            m_objectManager.adjustVersionsOfAddedOrRemovedEntities();
            m_objectManager.adjustVersionsOfChangedEntities();
            m_engineeringServerDAO.flush();
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Returns the root element.
     *
     * @throws SQLNException        thrown if there is an error
     */
    protected OIModelEntity getRoot() throws SQLException {
        if (m_root==null)
            m_root=m_objectManager.getOIModelEntity(m_rootModelID,KAONVocabularyAdaptor.INSTANCE.getRoot());
        return m_root;
    }
    /**
     * Preprocesses the event.
     *
     * @param event                 the event to be preprocessed
     */
    protected void preprocessEvent(ChangeEvent event) throws KAONException {
        m_modelID=m_objectManager.getOIModelID(event.getOIModel().getLogicalURI());
        if (m_modelID==-1)
            throw new KAONException("OI-model with logical URI '"+event.getOIModel().getLogicalURI()+"' doesn't exist.");
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        preprocessEvent(event);
        event.getEntity().accept(m_addVisitor);
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        preprocessEvent(event);
        event.getEntity().accept(m_removeVisitor);
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity superConcept=getEntity(event.getSuperConcept());
            OIModelEntity subConcept=getEntity(event.getSubConcept());
            m_engineeringServerDAO.getConceptHierarchyDAO().createConceptHierarchy(m_modelID,superConcept,subConcept);
            m_objectManager.conceptUpdated(superConcept);
            m_objectManager.conceptUpdated(subConcept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity superConcept=getEntity(event.getSuperConcept());
            OIModelEntity subConcept=getEntity(event.getSubConcept());
            m_engineeringServerDAO.getConceptHierarchyDAO().deleteConceptHierarchy(m_modelID,superConcept,subConcept);
            m_objectManager.conceptUpdated(superConcept);
            m_objectManager.conceptUpdated(subConcept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            m_engineeringServerDAO.getPropertyDomainDAO().createPropertyDomain(m_modelID,property,concept,0,Integer.MAX_VALUE);
            m_objectManager.propertyUpdated(property);
            m_objectManager.conceptUpdated(concept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            m_engineeringServerDAO.getPropertyDomainDAO().deletePropertyDomain(m_modelID,property,concept);
            m_objectManager.propertyUpdated(property);
            m_objectManager.conceptUpdated(concept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            m_engineeringServerDAO.getPropertyRangeDAO().createPropertyRange(m_modelID,property,concept);
            m_objectManager.propertyUpdated(property);
            m_objectManager.conceptUpdated(concept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            m_engineeringServerDAO.getPropertyRangeDAO().deletePropertyRange(m_modelID,property,concept);
            m_objectManager.propertyUpdated(property);
            m_objectManager.conceptUpdated(concept);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        preprocessEvent(event);
        OIModelEntity property=getEntity(event.getProperty());
        event.setOldIsAttirubte(property.getIsAttribute());
        property.setIsAttribute(event.getIsAttribute());
        m_objectManager.propertyUpdated(property);
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            int[] cardinalities=m_engineeringServerDAO.getPropertyDomainDAO().selectPropertyDomainCardinalities(m_modelID,property,concept);
            event.setOldCardinality(cardinalities[0]);
            m_engineeringServerDAO.getPropertyDomainDAO().updatePropertyDomainCarinalities(m_modelID,property,concept,event.getCardinality(),cardinalities[1]);
            m_objectManager.propertyUpdated(property);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property=getEntity(event.getProperty());
            OIModelEntity concept=getEntity(event.getConcept());
            int[] cardinalities=m_engineeringServerDAO.getPropertyDomainDAO().selectPropertyDomainCardinalities(m_modelID,property,concept);
            event.setOldCardinality(cardinalities[1]);
            m_engineeringServerDAO.getPropertyDomainDAO().updatePropertyDomainCarinalities(m_modelID,property,concept,cardinalities[0],event.getCardinality());
            m_objectManager.propertyUpdated(property);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity superProperty=getEntity(event.getSuperProperty());
            OIModelEntity subProperty=getEntity(event.getSubProperty());
            m_engineeringServerDAO.getPropertyHierarchyDAO().createPropertyHierarchy(m_modelID,superProperty,subProperty);
            m_objectManager.propertyUpdated(superProperty);
            m_objectManager.propertyUpdated(subProperty);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity superProperty=getEntity(event.getSuperProperty());
            OIModelEntity subProperty=getEntity(event.getSubProperty());
            m_engineeringServerDAO.getPropertyHierarchyDAO().deletePropertyHierarchy(m_modelID,superProperty,subProperty);
            m_objectManager.propertyUpdated(superProperty);
            m_objectManager.propertyUpdated(subProperty);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property1=getEntity(event.getProperty1());
            OIModelEntity property2=getEntity(event.getProperty2());
            m_engineeringServerDAO.getInversePropertyDAO().createInverseProperty(m_modelID,property1,property2);
            m_engineeringServerDAO.getInversePropertyDAO().createInverseProperty(m_modelID,property2,property1);
            m_objectManager.propertyUpdated(property1);
            m_objectManager.propertyUpdated(property2);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity property1=getEntity(event.getProperty1());
            OIModelEntity property2=getEntity(event.getProperty2());
            m_engineeringServerDAO.getInversePropertyDAO().deleteInverseProperty(m_modelID,property1,property2);
            m_engineeringServerDAO.getInversePropertyDAO().deleteInverseProperty(m_modelID,property2,property1);
            m_objectManager.propertyUpdated(property1);
            m_objectManager.propertyUpdated(property2);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        preprocessEvent(event);
        OIModelEntity property=getEntity(event.getProperty());
        event.setOldSymmetric(property.getIsSymmetric());
        property.setIsSymmetric(event.getSymmetric());
        m_objectManager.propertyUpdated(property);
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        preprocessEvent(event);
        OIModelEntity property=getEntity(event.getProperty());
        event.setOldTransitive(property.getIsTransitive());
        property.setIsTransitive(event.getTransitive());
        m_objectManager.propertyUpdated(property);
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity concept=getEntity(event.getConcept());
            if (!concept.equals(getRoot())) {
                OIModelEntity instance=getEntity(event.getInstance());
                m_engineeringServerDAO.getConceptInstanceDAO().createConceptInstance(m_modelID,concept,instance);
                m_objectManager.conceptUpdated(concept);
                m_objectManager.instanceUpdated(instance);
            }
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        preprocessEvent(event);
        try {
            OIModelEntity concept=getEntity(event.getConcept());
            if (!concept.equals(getRoot())) {
                OIModelEntity instance=getEntity(event.getInstance());
                m_engineeringServerDAO.getConceptInstanceDAO().deleteConceptInstance(m_modelID,concept,instance);
                m_objectManager.conceptUpdated(concept);
                m_objectManager.instanceUpdated(instance);
            }
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        preprocessEvent(event);
        try {
            PropertyInstance eventPropertyInstance=event.getPropertyInstance();
            OIModelEntity property=getEntity(eventPropertyInstance.getProperty());
            OIModelEntity sourceInstance=getEntity(eventPropertyInstance.getSourceInstance());
            OIModelEntity targetInstance=null;
            if (eventPropertyInstance.getTargetValue() instanceof Instance)
                targetInstance=getEntity((Instance)eventPropertyInstance.getTargetValue());
            String textValue=null;
            if (eventPropertyInstance.getTargetValue() instanceof String)
                textValue=(String)eventPropertyInstance.getTargetValue();
            m_engineeringServerDAO.getPropertyInstanceDAO().createPropertyInstance(m_modelID,property,sourceInstance,targetInstance,textValue);
            m_objectManager.propertyUpdated(property);
            m_objectManager.instanceUpdated(sourceInstance);
            if (targetInstance!=null)
                m_objectManager.instanceUpdated(targetInstance);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        preprocessEvent(event);
        try {
            PropertyInstance eventPropertyInstance=event.getPropertyInstance();
            OIModelEntity property=getEntity(eventPropertyInstance.getProperty());
            OIModelEntity sourceInstance=getEntity(eventPropertyInstance.getSourceInstance());
            OIModelEntity targetInstance=null;
            if (eventPropertyInstance.getTargetValue() instanceof Instance)
                targetInstance=getEntity((Instance)eventPropertyInstance.getTargetValue());
            String textValue=null;
            if (eventPropertyInstance.getTargetValue() instanceof String)
                textValue=(String)eventPropertyInstance.getTargetValue();
            m_engineeringServerDAO.getPropertyInstanceDAO().deletePropertyInstance(m_modelID,property,sourceInstance,targetInstance,textValue);
            m_objectManager.propertyUpdated(property);
            m_objectManager.instanceUpdated(sourceInstance);
            if (targetInstance!=null)
                m_objectManager.instanceUpdated(targetInstance);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for settomg the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        preprocessEvent(event);
        try {
            PropertyInstance eventPropertyInstance=event.getPropertyInstance();
            OIModelEntity property=getEntity(eventPropertyInstance.getProperty());
            OIModelEntity sourceInstance=getEntity(eventPropertyInstance.getSourceInstance());
            Object[] propertyValues=m_engineeringServerDAO.getPropertyInstanceDAO().getPropertyInstances(m_modelID,property,sourceInstance);
            PropertyInstance[] removedInstances=new PropertyInstance[propertyValues.length];
            for (int index=0;index<propertyValues.length;index++) {
                Object targetProxy;
                if (propertyValues[index] instanceof String) {
                    String textValue=(String)propertyValues[index];
                    targetProxy=textValue;
                    m_engineeringServerDAO.getPropertyInstanceDAO().deleteAttributeInstance(m_modelID,property,sourceInstance,textValue);
                }
                else {
                    EntityID entityID=(EntityID)propertyValues[index];
                    targetProxy=new LexicalEntryProxy(entityID.m_uri,entityID.m_version);
                    OIModelEntity targetInstance=m_objectManager.getOIModelEntity(m_modelID,entityID.m_uri);
                    m_objectManager.instanceUpdated(targetInstance);
                    m_engineeringServerDAO.getPropertyInstanceDAO().deleteRelationInstance(m_modelID,property,sourceInstance,targetInstance);
                }
                removedInstances[index]=new PropertyInstanceProxy((PropertyProxy)eventPropertyInstance.getProperty(),(InstanceProxy)eventPropertyInstance.getSourceInstance(),targetProxy);
            }
            event.setRemovedInstances(removedInstances);
            OIModelEntity targetInstance=null;
            if (eventPropertyInstance.getTargetValue() instanceof Instance)
                targetInstance=getEntity((Instance)eventPropertyInstance.getTargetValue());
            String textValue=null;
            if (eventPropertyInstance.getTargetValue() instanceof String)
                textValue=(String)eventPropertyInstance.getTargetValue();
            m_engineeringServerDAO.getPropertyInstanceDAO().createPropertyInstance(m_modelID,property,sourceInstance,targetInstance,textValue);
            m_objectManager.propertyUpdated(property);
            m_objectManager.instanceUpdated(sourceInstance);
            if (targetInstance!=null)
                m_objectManager.instanceUpdated(targetInstance);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        preprocessEvent(event);
        try {
            PropertyInstance eventPropertyInstance=event.getPropertyInstance();
            OIModelEntity property=getEntity(eventPropertyInstance.getProperty());
            OIModelEntity sourceInstance=getEntity(eventPropertyInstance.getSourceInstance());
            OIModelEntity targetInstance=null;
            if (eventPropertyInstance.getTargetValue() instanceof Instance)
                targetInstance=getEntity((Instance)eventPropertyInstance.getTargetValue());
            String textValue=null;
            if (eventPropertyInstance.getTargetValue() instanceof String)
                textValue=(String)eventPropertyInstance.getTargetValue();
            m_engineeringServerDAO.getPropertyInstanceDAO().deletePropertyInstance(m_modelID,property,sourceInstance,targetInstance,textValue);
            OIModelEntity newTargetInstance=null;
            if (event.getNewTargetValue() instanceof Instance)
                newTargetInstance=getEntity((Instance)event.getNewTargetValue());
            String newTextValue=null;
            if (event.getNewTargetValue() instanceof String)
                newTextValue=(String)event.getNewTargetValue();
            m_engineeringServerDAO.getPropertyInstanceDAO().createPropertyInstance(m_modelID,property,sourceInstance,newTargetInstance,newTextValue);
            m_objectManager.propertyUpdated(property);
            m_objectManager.instanceUpdated(sourceInstance);
            if (targetInstance!=null)
                m_objectManager.instanceUpdated(targetInstance);
            if (newTargetInstance!=null)
                m_objectManager.instanceUpdated(newTargetInstance);
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        preprocessEvent(event);
        try {
            String logicalURI=event.getIncludedOIModel().getLogicalURI();
            int includedModelID=m_engineeringServerDAO.getOIModelDAO().getOIModelID(logicalURI);
            m_engineeringServerDAO.getOIModelDAO().addIncludedOIModel(m_modelID,includedModelID);
            m_objectManager.updateIncludedOIModels();
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        preprocessEvent(event);
        try {
            String logicalURI=event.getIncludedOIModel().getLogicalURI();
            int includedModelID=m_engineeringServerDAO.getOIModelDAO().getOIModelID(logicalURI);
            m_engineeringServerDAO.getOIModelDAO().removeIncludedOIModel(m_modelID,includedModelID);
            m_objectManager.updateIncludedOIModels();
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Checks whether the version of the entity matches that of the OI-model entity bean.
     *
     * @param entity                        OI-model entity
     * @param oimodelEntity                 OI-model entity bean
     * @throws KAONException                thrown if versions don't match
     */
    protected void checkVersion(Entity entity,OIModelEntity oimodelEntity) throws KAONException {
        int entityVersion=((AbstractEntityProxy)entity).getVersion();
        int oimodelEntityVersion;
        if (entity instanceof Concept)
            oimodelEntityVersion=oimodelEntity.getConceptVersion();
        else if (entity instanceof Property)
            oimodelEntityVersion=oimodelEntity.getPropertyVersion();
        else if (entity instanceof Instance)
            oimodelEntityVersion=oimodelEntity.getInstanceVersion();
        else
            throw new KAONException("Internal error - invalid entity type.");
        if (entityVersion!=0 && entityVersion!=oimodelEntityVersion)
            throw new StaleDataException("Entity URI '"+entity.getURI()+"': sent version is "+entityVersion+", current version is "+oimodelEntityVersion);
    }
    /**
     * Returns an entity bean for given OI-model entity. In case the entity is not found,
     * exception is thrown. If an entity is found, its version is compared to the OI-model entity's version.
     * If version differ, an exception is thrown.
     *
     * @param entity                        OI-model entity
     * @return                              entity bean for given entity
     * @throws KAONException                thrown if entity is not found or if versions don't match
     */
    protected OIModelEntity getEntity(Entity entity) throws KAONException {
        try {
            String entityURI=entity.getURI();
            OIModelEntity result=m_objectManager.getOIModelEntity(m_modelID,entityURI);
            checkVersion(entity,result);
            return result;
        }
        catch (SQLException e) {
            throw new KAONException("SQL error",e);
        }
    }
    /**
     * Returns the IDs of updated concepts.
     *
     * @return                              IDs of updated concepts
     */
    public EntityID[] getUpdatedConceptIDs() {
        return m_objectManager.getUpdatedConceptIDs();
    }
    /**
     * Returns the IDs of updated properties.
     *
     * @return                              IDs of updated properties
     */
    public EntityID[] getUpdatedPropertyIDs() {
        return m_objectManager.getUpdatedPropertyIDs();
    }
    /**
     * Returns the IDs of updated instances.
     *
     * @return                              IDs of updated instances
     */
    public EntityID[] getUpdatedInstanceIDs() {
        return m_objectManager.getUpdatedInstanceIDs();
    }

    /**
     * Visitor that adds elements to the ontology.
     */
    protected class AddEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            OIModelEntity oimodelEntity=getOrCreateOIModelEntity(entity);
            ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
            if (info.m_isConcept)
                throw new KAONException("Concept already exists.");
            info.m_isConcept=true;
            m_objectManager.conceptUpdated(oimodelEntity);
            if (!info.m_isInstance)
                addInstanceForEntity(oimodelEntity);
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            OIModelEntity oimodelEntity=getOrCreateOIModelEntity(entity);
            ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
            if (info.m_isProperty)
                throw new KAONException("Property already exists.");
            info.m_isProperty=true;
            m_objectManager.propertyUpdated(oimodelEntity);
            if (!info.m_isInstance)
                addInstanceForEntity(oimodelEntity);
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            OIModelEntity oimodelEntity=getOrCreateOIModelEntity(entity);
            ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
            if (!info.m_isInstance)
                addInstanceForEntity(oimodelEntity);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
        /**
         * Adds an instance for the specified entity.
         */
        protected void addInstanceForEntity(OIModelEntity oimodelEntity) throws KAONException {
            try {
                ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
                if (info.m_isInstance)
                    throw new KAONException("Instance already exists.");
                info.m_isInstance=true;
                m_objectManager.instanceUpdated(oimodelEntity);
                m_objectManager.conceptUpdated(getRoot());
                m_engineeringServerDAO.getConceptInstanceDAO().createConceptInstance(m_modelID,getRoot(),oimodelEntity);
            }
            catch (SQLException e) {
                throw new KAONException("SQL error",e);
            }
        }
        /**
         * Loads or creates given entity.
         *
         * @param entity                        the entity
         * @return                              given entity
         */
        protected OIModelEntity getOrCreateOIModelEntity(Entity entity) throws KAONException {
            OIModelEntity oimodelEntity=null;
            try {
                oimodelEntity=getEntity(entity);
            }
            catch (KAONException e) {
            }
            if (oimodelEntity==null)
                try {
                    oimodelEntity=m_objectManager.createOIModelEntity(m_modelID,entity.getURI());
                }
                catch (SQLException e) {
                    throw new KAONException("Cannot create entity",e);
                }
            return oimodelEntity;
        }
    }

    /**
     * Visitor that removes elements to the ontology.
     */
    protected class RemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            OIModelEntity oimodelEntity=getEntity(entity);
            ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
            if (!info.m_isConcept)
                throw new KAONException("Concept is already removed.");
            info.m_isConcept=false;
            m_objectManager.conceptUpdated(oimodelEntity);
            removeInstanceForEntity(entity);
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            OIModelEntity oimodelEntity=getEntity(entity);
            ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
            if (!info.m_isProperty)
                throw new KAONException("Property is already removed.");
            info.m_isProperty=false;
            m_objectManager.propertyUpdated(oimodelEntity);
            removeInstanceForEntity(entity);
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            removeInstanceForEntity(entity);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
        /**
         * Removes an instance for specified entity.
         */
        protected void removeInstanceForEntity(Entity entity) throws KAONException {
            try {
                OIModelEntity oimodelEntity=getEntity(entity);
                ObjectManager.AdditionalEntityInfo info=m_objectManager.getAdditionalEntityInfo(oimodelEntity);
                if (!info.m_isInstance)
                    throw new KAONException("Instance is already removed.");
                info.m_isInstance=false;
                m_objectManager.instanceUpdated(oimodelEntity);
                m_objectManager.conceptUpdated(getRoot());
                m_engineeringServerDAO.getConceptInstanceDAO().deleteConceptInstance(m_modelID,getRoot(),oimodelEntity);
            }
            catch (SQLException e) {
                throw new KAONException("SQL error",e);
            }
        }
    }
}
