package edu.unika.aifb.kaon.datalog.program;

import java.util.Set;
import java.util.HashSet;

/**
 * Represents a rule in the program. A rule consists of a positive head literal and of a conjunction of
 * several body literals. A rule must be range-restricted (also called safe or allowed), which means that
 * each variable from the head must occur in some positive literal in the body.
 */
public class Rule {
    /** The label of the rule. */
    protected String m_label;
    /** The head literal of the rules. (Must be positive.) */
    protected Literal[] m_head;
    /** The array of body literals. No ordering is assumed. */
    protected Literal[] m_body;

    /**
     * Creates an instance of this class.
     *
     * @param label                         the label of the rule
     * @param head                          the head of the rule (must be a positive literal)
     * @param body                          the body of the rule
     */
    public Rule(String label,Literal head,Literal[] body) {
        this(label,new Literal[] { head },body);
    }
    /**
     * Creates an instance of this class.
     *
     * @param label                         the label of the rule
     * @param head                          the head of the rule (must be positive literals)
     * @param body                          the body of the rule
     */
    public Rule(String label,Literal[] head,Literal[] body) {
        m_label=label;
        m_head=head;
        m_body=body;
        for (int i=0;i<m_head.length;i++)
            if (!m_head[i].isPositive())
                throw new IllegalArgumentException("Head of the rule must be a positive literal.");
    }
    /**
     * Returns the label of the rule.
     *
     * @return                              the label of the rule
     */
    public String getLabel() {
        return m_label;
    }
    /**
     * Returns the head literal. This method can be called only for Horn rules, otherwise it throws an exception.
     *
     * @return                              the head literal
     */
    public Literal getHeadLiteral() {
        if (m_head.length==1)
            return m_head[0];
        else
            throw new IllegalStateException("This rule doesn't have exactly one head literal.");
    }
    /**
     * Returns the number of head literals.
     *
     * @return                              the number of head literals
     */
    public int getHeadLength() {
        return m_head.length;
    }
    /**
     * Returns the head literal with given index.
     *
     * @param headLiteralIndex              the index of the head literal
     * @return                              the head literal with given index
     */
    public Literal getHeadLiteral(int headLiteralIndex) {
        return m_head[headLiteralIndex];
    }
    /**
     * Returns the head literals of the rule.
     *
     * @return                              the head literals of the rule
     */
    public Literal[] getHeadLiterals() {
        return m_head;
    }
    /**
     * Returns the number of body literals.
     *
     * @return                              the number of body literals
     */
    public int getBodyLength() {
        return m_body.length;
    }
    /**
     * Returns the body literal with given index.
     *
     * @param bodyLiteralIndex              the index of the body literal
     * @return                              the literal with given index
     */
    public Literal getBodyLiteral(int bodyLiteralIndex) {
        return m_body[bodyLiteralIndex];
    }
    /**
     * Returns the array of literals of the rule.
     *
     * @return                              the array of literals of the rule
     */
    public Literal[] getBodyLiterals() {
        return m_body;
    }
    /**
     * Returns <code>true</code> if this is a Horn rule with one head literal.
     *
     * @return                              <code>true</code> if this is a Horn rule with one head literal
     */
    public boolean isHorn() {
        return m_head.length==1;
    }
    /**
     * Checks whether the rule is range-restricted.
     *
     * @return                              <code>true</code> if the rule is range-restricted
     */
    public boolean isRangeRestricted() {
        Set allVariables=new HashSet();
        Set allPositiveBodyLiteralVariables=new HashSet();
        for (int headLiteralIndex=0;headLiteralIndex<m_head.length;headLiteralIndex++) {
            Literal headLiteral=m_head[headLiteralIndex];
            for (int headVariableIndex=0;headVariableIndex<headLiteral.getArity();headVariableIndex++) {
                Variable variable=headLiteral.getArgumentVariable(headVariableIndex);
                if (variable!=null)
                    allVariables.add(variable);
                UnaryFunctionCall unaryFunctionCall=headLiteral.getArgumentUnaryFunctionCall(headVariableIndex);
                if (unaryFunctionCall!=null && unaryFunctionCall.getCallTerm() instanceof Variable)
                    allVariables.add(unaryFunctionCall.getCallTerm());
            }
        }
        for (int bodyLiteralIndex=0;bodyLiteralIndex<m_body.length;bodyLiteralIndex++) {
            Literal bodyLiteral=m_body[bodyLiteralIndex];
            for (int bodyLiteralVariableIndex=0;bodyLiteralVariableIndex<bodyLiteral.getArity();bodyLiteralVariableIndex++) {
                Variable bodyVariable=bodyLiteral.getArgumentVariable(bodyLiteralVariableIndex);
                if (bodyVariable!=null) {
                    allVariables.add(bodyVariable);
                    if (bodyLiteral.isPositive())
                        allPositiveBodyLiteralVariables.add(bodyVariable);
                }
                UnaryFunctionCall unaryFunctionCall=bodyLiteral.getArgumentUnaryFunctionCall(bodyLiteralVariableIndex);
                if (unaryFunctionCall!=null && unaryFunctionCall.getCallTerm() instanceof Variable) {
                    allVariables.add(unaryFunctionCall.getCallTerm());
                    if (bodyLiteral.isPositive())
                        allPositiveBodyLiteralVariables.add(unaryFunctionCall.getCallTerm());
                }
            }
        }
        return allPositiveBodyLiteralVariables.containsAll(allVariables);
    }
    /**
     * Returns the string representation of this rule.
     *
     * @return                              the string representation of this rule
     */
    public String toString() {
        StringBuffer buffer=new StringBuffer();
        if (m_label!=null) {
            buffer.append(m_label);
            buffer.append(": ");
        }
        for (int i=0;i<m_head.length;i++) {
            if (i>0)
                buffer.append(" v ");
            buffer.append(m_head[i].toString());
        }
        if (m_body.length!=0) {
            buffer.append(" :- ");
            for (int i=0;i<m_body.length;i++) {
                if (i>0)
                    buffer.append(',');
                buffer.append(m_body[i].toString());
            }
        }
        buffer.append('.');
        return buffer.toString();
    }
}
