package edu.unika.aifb.kaon.datalog.program;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

/**
 * A factory class for predicates. It makes sure that there is exactly one instance of a predicate with given name and arity.
 */
public class PredicateFactory {
    /** The map of predicates indexed by their full names. */
    protected Map m_predicatesByFullNames;

    /**
     * Creates an instance of this class.
     */
    public PredicateFactory() {
        m_predicatesByFullNames=new HashMap();
    }
    /**
     * Creates an instance of this class.
     *
     * @param predicateFactory                  the factory whose predicates are copied to the new factory
     */
    public PredicateFactory(PredicateFactory predicateFactory) {
        m_predicatesByFullNames=new HashMap(predicateFactory.m_predicatesByFullNames);
    }
    /**
     * Returns a positive predicate with given name and arity. If no such predicate exists, a new one is created.
     *
     * @param name                              the name of the predicate (excluding positive/negative status and arity)
     * @param arity                             the arity of the predicate
     * @return                                  the positive predicate with given name and arity
     */
    public Predicate getPredicate(String name,int arity) {
        return getPredicate(true,name,arity);
    }
    /**
     * Returns a predicate with given name, positive/negative status and arity. If no such predicate exists, a new one is created.
     *
     * @param isPositive                        <code>true</code> if the predicate should be classically positive
     * @param name                              the name of the predicate (excluding positive/negative status and arity)
     * @param arity                             the arity of the predicate
     * @return                                  the predicate with given name, positive/negative status and arity
     */
    public Predicate getPredicate(boolean isPositive,String name,int arity) {
        String fullName=(isPositive ? "" : "-")+name+"/"+arity;
        Predicate predicate=(Predicate)m_predicatesByFullNames.get(fullName);
        if (predicate==null) {
            predicate=new Predicate(isPositive,name,arity);
            m_predicatesByFullNames.put(predicate.getFullName(),predicate);
        }
        return predicate;
    }
    /**
     * Returns a predicate which is classically complementary to the given one. If no such predicate exists, a new one is created.
     *
     * @param predicate                         the predicate whose complement is required
     * @return                                  the predicate classically complementary to the given one
     */
    public Predicate getComplementPredicate(Predicate predicate) {
        return getPredicate(!predicate.isPositive(),predicate.getName(),predicate.getArity());
    }
    /**
     * Returns the set of all predicates in the factory.
     *
     * @return                                  the set of predicates in the factory/
     */
    public Set getAllPredicates() {
        return new HashSet(m_predicatesByFullNames.values());
    }
}
